package com.bxm.newidea.component.uuid.snowflake;

import com.bxm.newidea.component.strategy.ReturnedStrategyExecutor;
import com.bxm.newidea.component.tools.NetworkInterfaceManager;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.uuid.config.ComponentUUIDConfigurationProperties;
import com.bxm.newidea.component.uuid.config.SequenceConfigHolder;
import com.bxm.newidea.component.uuid.constant.SequenceConstant;
import com.bxm.newidea.component.uuid.strategy.RandomAcquireConfigStrategy;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.util.NumberUtils;

import static com.bxm.newidea.component.uuid.constant.SequenceConstant.PROP_DATA_CENTER_ID;
import static com.bxm.newidea.component.uuid.constant.SequenceConstant.PROP_WORKER_ID;

/**
 * 雪花算法实现的序列创建类
 *
 * @author liujia
 * @date 2020/04/23 09:40
 */
@Slf4j
public class SnowflakeSequenceCrater implements SequenceCreater, ApplicationRunner {

    private Sequence sequence;

    private ComponentUUIDConfigurationProperties properties;

    private ReturnedStrategyExecutor returnedStrategyExecutor;

    public SnowflakeSequenceCrater(ComponentUUIDConfigurationProperties properties, ReturnedStrategyExecutor returnedStrategyExecutor) {
        this.properties = properties;
        this.returnedStrategyExecutor = returnedStrategyExecutor;
    }

    @Override
    public Long nextLongId() {
        return this.sequence.nextId();
    }

    @Override
    public String nextStringId() {
        return this.sequence.nextId().toString();
    }

    @Override
    public void run(ApplicationArguments args) {

        SequenceConfigHolder acquire = null;

        if (properties.isFastMode()) {
            acquire = new RandomAcquireConfigStrategy().execute(null);
        } else if (null != returnedStrategyExecutor) {
            acquire = returnedStrategyExecutor.execute(
                    SequenceConstant.ACQUIRE_CONFIG_GROUP,
                    properties.getAcquireConfigStrategy(),
                    null);
        }

        if (null == acquire) {
            acquire = new RandomAcquireConfigStrategy().execute(null);
        }

        Long workerId = acquire.getWorkerId();
        Long dataCenterId = acquire.getDataCenterId();

        if (null != properties.getDataCenterId()) {
            dataCenterId = properties.getDataCenterId();
        }

        //优先使用环境配置中的值，具有最高优先级
        String configWorkerId = System.getProperty(PROP_WORKER_ID);
        if (null != configWorkerId) {
            log.debug("使用系统配置的[{}]:[{}]", PROP_WORKER_ID, configWorkerId);
            workerId = NumberUtils.parseNumber(configWorkerId, Long.class);
        }

        String configDataCenterId = System.getProperty(PROP_DATA_CENTER_ID);
        if (null != configDataCenterId) {
            log.debug("使用系统配置的[{}]:[{}]", PROP_DATA_CENTER_ID, configDataCenterId);
            dataCenterId = NumberUtils.parseNumber(configDataCenterId, Long.class);
        }

        log.info("ID生成配置，serverName:[{}],ip:[{}],port:[{}],dataCenterId:[{}],workerId:[{}]",
                properties.getLeafName(),
                NetworkInterfaceManager.INSTANCE.getLocalHostAddress(),
                properties.getLeafPort(),
                dataCenterId,
                workerId);

        this.sequence = new Sequence(workerId, dataCenterId);
        this.sequence.setClock(true);
    }
}
