package com.bxm.newidea.component.controller;

import com.bxm.newidea.component.binder.ByteEditor;
import com.bxm.newidea.component.binder.DateEditor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.time.DateFormatUtils;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.TimeZone;

/**
 * controller 基类
 * 预留，后续用于处理公共内容
 *
 * @author liujia 2018/3/30 17:20
 */
@Slf4j
public class BaseController {

    private static String encodeURIComponent(String value) throws UnsupportedEncodingException {
        return URLEncoder.encode(value, "UTF-8").replaceAll("\\+", "%20");
    }

    /**
     * 属性值绑定转换编辑器
     *
     * @param binder 绑定集合
     */
    @InitBinder
    public void initBinding(WebDataBinder binder) {
        //byte转换
        binder.registerCustomEditor(Byte.class, new ByteEditor());
        //string类型转换为空时，转换为null而不是空字符串
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(true));
        //使用自定义的日期转换
        binder.registerCustomEditor(Date.class, new DateEditor());
    }

    protected void write(HttpServletResponse response, String content) {
        try {
            response.getWriter().write(content);
            response.getWriter().flush();
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        }
    }

    /**
     * 对下载文件的文件名进行处理，设置contentType
     *
     * @param fileName 显示的下载文件名
     * @param response 响应
     */
    protected void processDownload(String fileName, HttpServletResponse response) {
        try {
            String dateString = DateFormatUtils.format(new Date(), "EEE, dd MMM yyyy HH:mm:ss z", TimeZone.getTimeZone("GMT"));
            String contentType = "application/x-download";

            response.setStatus(HttpServletResponse.SC_OK);
            response.setHeader("Last-modified", dateString);
            response.setHeader("Connection", "keep-alive");
            response.setHeader("Date", dateString);
            response.setHeader("Cache-Control", "no-cache");
            response.setContentType(contentType);

            String headerValue = "attachment;";
            headerValue += " filename=\"" + encodeURIComponent(fileName) + "\";";
            headerValue += " filename*=utf-8''" + encodeURIComponent(fileName);
            response.setHeader("Content-Disposition", headerValue);
        } catch (UnsupportedEncodingException e) {
            log.error("文件名转换发生异常,文件名为[{}]", fileName);
        }
    }
}
