package com.bxm.newidea.component.config;

import com.bxm.newidea.component.log.LogContextFilter;
import com.bxm.newidea.component.mvc.ApiVersionRequestMapping;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import org.hibernate.validator.HibernateValidator;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.boot.autoconfigure.web.servlet.WebMvcAutoConfiguration;
import org.springframework.boot.autoconfigure.web.servlet.WebMvcRegistrations;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import javax.annotation.Resource;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

/**
 * 公共的web配置
 *
 * @author liujia
 */
@Configuration(proxyBeanMethods = false)
@AutoConfigureAfter(WebMvcAutoConfiguration.class)
@EnableConfigurationProperties(ComponentWebConfigurationProperties.class)
public class ComponentWebAutoConfiguration implements WebMvcRegistrations, WebMvcConfigurer {

    private ComponentWebConfigurationProperties properties;

    @Resource
    private HttpMessageConverters httpMessageConverters;

    public ComponentWebAutoConfiguration(ComponentWebConfigurationProperties properties) {
        this.properties = properties;
    }

    @Override
    public RequestMappingHandlerMapping getRequestMappingHandlerMapping() {
        return new ApiVersionRequestMapping();
    }

    @Override
    public void extendMessageConverters(List<HttpMessageConverter<?>> converters) {
        converters.addAll(httpMessageConverters.getConverters());
    }

    @Bean
    public MappingJackson2HttpMessageConverter jsonConverter() {
        MappingJackson2HttpMessageConverter jsonConverter = new MappingJackson2HttpMessageConverter();

        List<MediaType> supportedMediaTypes = new ArrayList<>();
        supportedMediaTypes.add(new MediaType(MediaType.TEXT_PLAIN, Charset.forName("UTF-8")));
        supportedMediaTypes.add(new MediaType(MediaType.APPLICATION_JSON, Charset.forName("UTF-8")));

        jsonConverter.setSupportedMediaTypes(supportedMediaTypes);

        ObjectMapper objectMapper = jsonConverter.getObjectMapper();

        // 将长整型转化为字符串，解决前端长整型精度丢失的问题
        SimpleModule simpleModule = new SimpleModule();
        simpleModule.addSerializer(Long.class, ToStringSerializer.instance);
        simpleModule.addSerializer(long.class, ToStringSerializer.instance);
        simpleModule.addSerializer(Long.TYPE, ToStringSerializer.instance);
        objectMapper.registerModule(simpleModule);

        return jsonConverter;
    }

    @Bean
    @ConditionalOnProperty(prefix = "component.web", value = "enableLogFilter", matchIfMissing = true)
    public LogContextFilter logContextFilter() {
        return new LogContextFilter(properties);
    }

    /**
     * 自定义验证工具，覆盖默认的验证工具
     *
     * @return 自定义验证器
     */
    @Bean
    @Primary
    public Validator validator() {
        ValidatorFactory validatorFactory = Validation.byProvider(HibernateValidator.class)
                                                      .configure()
                                                      //启用快速失败，当验证条件有一个不符合时即返回错误
                                                      .failFast(properties.isFastValid())
                                                      .buildValidatorFactory();
        return validatorFactory.getValidator();
    }
}
