package com.bxm.newidea.component.wechat.config;

import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.wechat.extend.CustomApacheHttpClientBuilder;
import com.bxm.newidea.component.wechat.extend.WxMpInRedisConfigStorage;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.mp.api.WxMpConfigStorage;
import me.chanjar.weixin.mp.api.WxMpInMemoryConfigStorage;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.api.impl.WxMpServiceImpl;

import java.util.Map;

/**
 * 微信公众号上下文信息
 * 负责获取场景相匹配的公众号信息
 *
 * @author liujia
 * @date 2020/5/29 14:00
 **/
@Slf4j
public class WechatMpContext {

    private final WechatMpProperties wechatMPProperties;

    /**
     * 默认的微信公众号
     */
    private WxMpService defaultWxMpService;

    /**
     * 场景值与微信服务接口的映射关系
     */
    private Map<String, WxMpService> wxMpServiceMap;

    /**
     * 公众号ID与账号信息的映射关系，用于快速定位
     */
    private Map<String, WechatMpAccount> accountMap;

    private final RedisStringAdapter redisStringAdapter;

    private final CustomApacheHttpClientBuilder customApacheHttpClientBuilder;

    public WechatMpContext(WechatMpProperties wechatMPProperties,
                           RedisStringAdapter redisStringAdapter,
                           CustomApacheHttpClientBuilder customApacheHttpClientBuilder) {
        this.wechatMPProperties = wechatMPProperties;
        this.redisStringAdapter = redisStringAdapter;
        this.customApacheHttpClientBuilder = customApacheHttpClientBuilder;
        refresh();
    }

    /**
     * 通过场景获取对应的微信公众号服务类
     * 如果场景值为空，则尝试获取默认的公众号 - 增加一定的容错性
     * 场景值对应的公众号可用，则直接返回
     * 场景值对应的公众号未配置，则使用默认公众号，并提示调用者
     *
     * @param scene 业务场景值
     * @return 微信公众号服务接口，第三方库已封装了公众号调用的实现细节
     */
    public WxMpService obtain(String scene) {
        log.debug("获取微信公众号服务：{}", scene);

        WxMpService matchWxMapService = null;

        if (StringUtils.isBlank(scene)) {
            if (null != defaultWxMpService) {
                log.debug("场景值为空，使用默认的微信公众号：{}", defaultWxMpService.getWxMpConfigStorage().getAppId());
                matchWxMapService = defaultWxMpService;
            }
        } else {
            matchWxMapService = wxMpServiceMap.get(scene);

            if (null == matchWxMapService) {
                log.warn("场景[{}]未找到对应的公众号，请确定场景对应的公众号是处于启用状态，是否给场景指定了公众号");
                matchWxMapService = defaultWxMpService;
            }
        }

        if (matchWxMapService == null) {
            log.error("场景[{}]未找到对应的公众号，请检查是否配置了场景和微信的关联关系、是否配置了别名对应的公众号，是否配置了默认的公众号",
                    scene);
        } else {
            log.debug("根据[{}]获取的微信公众号实例：{}", scene, matchWxMapService.getWxMpConfigStorage().getAppId());
        }

        return matchWxMapService;
    }

    /**
     * 根据微信公众号服务信息获取原始配置
     *
     * @param service 公众号服务
     * @return 原始配置账号信息
     */
    public WechatMpAccount obtainAccount(WxMpService service) {
        WechatMpAccount wechatMpAccount = accountMap.get(service.getWxMpConfigStorage().getAppId());

        //减少后面判断空的处理
        if (wechatMpAccount == null) {
            wechatMpAccount = new WechatMpAccount();
            wechatMpAccount.setAuthDomainUrls(Lists.newArrayList());
            wechatMpAccount.setJsSignatureDomainUrls(Lists.newArrayList());
        }

        return wechatMpAccount;
    }

    /**
     * 配置信息变更时刷新配置信息
     */
    public void refresh() {
        Map<String, WxMpService> swapMap = Maps.newHashMap();
        Map<String, WechatMpAccount> swapAccountMap = Maps.newHashMap();

        wechatMPProperties.getRelation().forEach((scnen, alias) -> {
            boolean hit = false;
            // 解析配置，构建公众号服务接口
            for (WechatMpAccount account : wechatMPProperties.getAccounts()) {
                if (!checkAccount(account)) {
                    return;
                }

                WxMpService wxMpService = build(account);
                if (StringUtils.equals(alias, account.getAlias())) {
                    swapMap.put(scnen, wxMpService);
                    hit = true;
                }

                if (StringUtils.equals(wechatMPProperties.getMasterMpAlias(), account.getAlias())) {
                    defaultWxMpService = wxMpService;
                }

                swapAccountMap.put(account.getAppId(), account);
            }

            if (!hit) {
                log.error("请检查配置信息，场景[{}]对应的公众号[{}]不存在", scnen, alias);
            }
        });

        accountMap = swapAccountMap;
        wxMpServiceMap = swapMap;
    }

    /**
     * 检查账号信息是否配置正确
     *
     * @param account 配置的账号信息
     */
    private boolean checkAccount(WechatMpAccount account) {
        boolean pass = true;
        if (StringUtils.isBlank(account.getAppId())) {
            log.error("公众号APPID必须配置,配置信息：{}", account);
            pass = false;
        }

        if (StringUtils.isBlank(account.getAppSecret())) {
            log.error("公众号密钥必须配置,配置信息：{}", account);
            pass = false;
        }

        if (StringUtils.isBlank(account.getAlias())) {
            log.error("公众号别名必须配置,配置信息：{}", account);
            pass = false;
        }

        return pass;
    }

    private WxMpService build(WechatMpAccount account) {
        WxMpService service = new WxMpServiceImpl();
        service.setWxMpConfigStorage(wxConfigStorage(account));
        return service;
    }

    /**
     * 配置存储策略
     *
     * @param account 配置文件中的账号信息
     * @return 配置存储策略
     */
    private WxMpConfigStorage wxConfigStorage(WechatMpAccount account) {
        WxMpInMemoryConfigStorage wxConfigProvider = new WxMpInRedisConfigStorage(redisStringAdapter, customApacheHttpClientBuilder);
        wxConfigProvider.setAppId(account.getAppId());
        wxConfigProvider.setSecret(account.getAppSecret());
        wxConfigProvider.setAesKey(account.getAesKey());
        return wxConfigProvider;
    }
}
