package com.bxm.newidea.component.schedule.rpc;

import com.alibaba.fastjson.JSON;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.config.ExecutorParam;
import com.bxm.newidea.component.schedule.config.XxlJobConfigurationProperties;
import com.bxm.newidea.component.schedule.constant.CommonConstant;
import com.bxm.newidea.component.schedule.task.ScheduleTask;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Value;

/**
 * 使用配置中的远程调度服务地址，通过http请求创建、更新、删除定时任务
 * @author liujia
 */
@Slf4j
public class ScheduleServiceImpl implements ScheduleService {

    private ScheduleFeignService scheduleFeignService;

    private XxlJobConfigurationProperties properties;

    @Value("${eureka.instance.metadataMap.version:}")
    private String serviceVersion;

    public ScheduleServiceImpl(XxlJobConfigurationProperties properties,
                               ScheduleFeignService scheduleFeignService) {
        this.properties = properties;
        this.scheduleFeignService = scheduleFeignService;
    }

    @Override
    public boolean push(ScheduleTask task) {
        if (validate(task)) {
            //临时性任务要做更新
            if (task.onceTask()) {
                scheduleFeignService.update(task.taskName(),
                        task.cron(),
                        task.description(),
                        getExecutorVersion(),
                        CommonConstant.HANDLER_NAME,
                        buildParam(task),
                        task.routeStrategy().name(),
                        task.blockStrategy().name(),
                        task.author());
            } else {
                //永久性任务只做新增
                scheduleFeignService.add(task.taskName(),
                        task.cron(),
                        task.description(),
                        getExecutorVersion(),
                        CommonConstant.HANDLER_NAME,
                        buildParam(task),
                        task.routeStrategy().name(),
                        task.blockStrategy().name(),
                        task.author());
            }
            return true;
        }
        return false;
    }

    @Override
    public boolean remove(String taskName) {
        if (StringUtils.isNotBlank(taskName)) {
            scheduleFeignService.remove(taskName);
            return true;
        }
        return false;
    }

    @Override
    public String getExecutorVersion() {
        return properties.getAppName() + "_v" + properties.getVersion();
    }

    private String buildParam(ScheduleTask task) {
        return JSON.toJSONString(new ExecutorParam(task.callback().beanName(), task.callbackParam()));
    }

    private boolean validate(ScheduleTask task) {
        if (StringUtils.isBlank(task.taskName())) {
            log.error("任务名未配置");
            return false;
        }
        if (null == task.cron()) {
            log.error("任务未设置执行时间");
            return false;
        }
        if (null == task.callback()) {
            log.error("任务回调服务类未设置");
            return false;
        }

        return true;
    }
}

