package com.bxm.localnews.news.convert;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.integration.VirtualUserIntegrationService;
import com.bxm.localnews.news.config.UserProperties;
import com.bxm.localnews.news.domain.NewsReplyMapper;
import com.bxm.localnews.news.enums.Converter;
import com.bxm.localnews.news.enums.NewsConstant;
import com.bxm.localnews.news.enums.ReplyTypeEnum;
import com.bxm.localnews.news.facade.service.ForumPostInnerService;
import com.bxm.localnews.news.model.param.NewsReplyAddParam;
import com.bxm.localnews.news.model.vo.NewsReply;
import com.bxm.localnews.news.util.HtmlContentUtil;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.newidea.component.filter.InputFilter;
import com.bxm.newidea.component.service.BaseService;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.news.enums.ReplyStatusEnum.IS_SHOW;
import static com.bxm.localnews.news.enums.ReplyStatusEnum.WAIT_REVIEW;
import static java.lang.Boolean.TRUE;

@AllArgsConstructor
@Component
public class NewsReplyConver extends BaseService implements Converter<NewsReplyAddParam, NewsReply> {

    private final UserIntegrationService userIntegrationService;

    private final UserProperties userProperties;

    private final NewsReplyMapper newsReplyMapper;

    private final VirtualUserIntegrationService virtualUserIntegrationService;

    private final ForumPostInnerService forumPostInnerService;

    @Override
    public NewsReply convert(NewsReplyAddParam param) {
        NewsReply newsReply = new NewsReply();
        BeanUtils.copyProperties(param, newsReply);
        // 3.12.0 判断是否有图片评论
        if (StringUtils.isNotBlank(param.getReplyImg())) {
            // 如果有则需要审核
            newsReply.setStatus(WAIT_REVIEW.getCode());
        } else {
            // 否则默认直接通过
            newsReply.setStatus(IS_SHOW.getCode());
        }

        newsReply.setLikeCount(0);
        newsReply.setRootId(0L);
        newsReply.setLevel((byte) 0);
        newsReply.setAddTime(new Date());
        newsReply.setInteractiveCount(0);
        newsReply.setId(nextSequence());
        newsReply.setAreaCode(param.getAreaCode());
        //得到用户的头像昵称
        if (null != param.getUserId()) {
            UserBean userBean = userIntegrationService.selectUserFromCache(param.getUserId());
            if (StringUtils.isBlank(userBean.getHeadImg()) || StringUtils.isBlank(userBean.getNickname())) {
                logger.info("评论用户头像或昵称为空,userInfo={}", JSON.toJSONString(userBean));
            }
            newsReply.setHeadImg(StringUtils.isBlank(userBean.getHeadImg()) ? userProperties.getDefaultHeadImgUrl() : userBean.getHeadImg());
            newsReply.setUserNickname(userBean.getNickname());
        }

        newsReply.setType(ReplyTypeEnum.POST_REPLY.getCode());

        //马甲号发帖标识
        boolean vestPost = false;

        //如果是对新闻的直接评论则设置父评论信息为null
        if (newsReply.getParentId() == null || newsReply.getParentId() == 0) {
            newsReply.setRootId(0L);
            newsReply.setParentId(0L);
            newsReply.setLevel(NewsConstant.REPLY_LEVEL_0);
            //根评论且发帖用户是马甲号时,马甲号发帖时 马甲号发帖标识 设置为true;
            if (param.getType() == ReplyTypeEnum.POST_REPLY.getCode()) {
                UserBean postVoUser = userIntegrationService.selectUserFromCache(forumPostInnerService.getPostUserId(param.getNewsId()));

                vestPost = Objects.nonNull(postVoUser) && Objects.nonNull(postVoUser.getState()) && postVoUser.getState() == 3;
            }
        } else {
            NewsReply newsReplyDb = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReply.getParentId(), param.getNewsId());
            if (newsReplyDb != null) {
                newsReply.setRootId(newsReplyDb.getRootId());
                Byte level = newsReplyDb.getLevel();
                if (level == 0) {
                    newsReply.setLevel(NewsConstant.REPLY_LEVEL_1);
                    newsReply.setRootId(newsReplyDb.getId());
                } else {
                    newsReply.setLevel(NewsConstant.REPLY_LEVEL_2);
                    newsReply.setRootId(newsReplyDb.getRootId());
                }
            }
        }
        if (Objects.nonNull(param.getParentUserId())) {
            UserBean parentUser = userIntegrationService.selectUserFromCache(param.getParentUserId());
            vestPost = vestPost || (Objects.nonNull(parentUser) && Objects.nonNull(parentUser.getState()) && parentUser.getState() == 3);

        }
        //正式用户对马甲号用户回复时,或对马甲号发帖评论时添加未互动标记
        if (vestPost) {
            newsReply.setVestInteract((byte) 0);
        }

        // 判断是否是马甲号
        if (Objects.equals(virtualUserIntegrationService.exists(param.getUserId()), TRUE)) {
            // 标识为当前评论为马甲号评论
            newsReply.setReplyUserType(1);
        } else {
            // 真实用户评论 判断被评论人是否是马甲号 被评论人可能是评论的作者，也可能是帖子的作者
            Long parentUserId;
            if (Objects.isNull(param.getParentUserId())) {
                parentUserId = forumPostInnerService.getPostUserId(param.getNewsId());
            } else {
                parentUserId = param.getParentUserId();
            }

            // 如果真实用户评论的作者是马甲号，则需要单独标识为真实用户评论马甲号
            newsReply.setReplyUserType(virtualUserIntegrationService.exists(parentUserId) ? 2 : 0);
        }

        // 对回复内容进行过滤
        newsReply.setReplyContent(InputFilter.builder(newsReply.getReplyContent()).filte());

        // 对回复内容的@数据进行转义
        String contentWithHtml = replaceAtElement(newsReply.getReplyContent());
        // 避免使用转移的注入 这里将转移之后的数据分两个字段存储
        // 改字段存储的数据为携带了html标签的内容 返回给app端
        newsReply.setReplyContentHtml(contentWithHtml);
        // 该字段不包含html内容，用来返回给运营后台
        newsReply.setReplyContent(HtmlContentUtil.getForumPostTextFiled("<p>" + contentWithHtml + "</p>"));

        return newsReply;
    }

    private String replaceAtElement(String content) {
        content = content.replace("\\u003c", "<");
        content = content.replace("\\u003e", ">");
        return content;
    }

}
