package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.integration.VirtualUserIntegrationService;
import com.bxm.localnews.news.comment.service.CommentFacadeService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.config.ReplyConfigProperties;
import com.bxm.localnews.news.domain.AllReplyMapper;
import com.bxm.localnews.news.domain.NewsReplyMapper;
import com.bxm.localnews.news.domain.UserReplyMapper;
import com.bxm.localnews.news.enums.NewsConstant;
import com.bxm.localnews.news.facade.service.ForumPostInnerService;
import com.bxm.localnews.news.model.dto.NewsReplyDTO;
import com.bxm.localnews.news.model.dto.NewsReplyMirrorDTO;
import com.bxm.localnews.news.model.entity.NewsReplyAllCountInfo;
import com.bxm.localnews.news.model.param.NewsReplyParam;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.model.vo.NewsReply;
import com.bxm.localnews.news.model.vo.UserReply;
import com.bxm.localnews.news.model.vo.VirtualUser;
import com.bxm.localnews.news.util.JudgeUtil;
import com.bxm.newidea.component.sync.core.CacheHolder;
import com.bxm.newidea.component.sync.core.SyncCacheHolderFactory;
import com.bxm.newidea.component.tools.DateBeautifyUtils;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.bxm.localnews.news.constant.MemoryCacheKey.HOT_REPLY_CACHE;
import static com.bxm.localnews.news.constant.MemoryCacheKey.REPLY_COUNT_CACHE;
import static com.bxm.localnews.news.utils.CommentUtil.filter;
import static java.util.Comparator.comparing;

/**
 * @author liujia
 * @date 1/15/21 10:43 AM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class CommentFacadeServiceImpl implements CommentFacadeService {

    private NewsReplyMapper newsReplyMapper;

    private ForumProperties forumProperties;

    private ForumPostInnerService forumPostInnerService;

    private CacheHolder cacheHolder;

    private AllReplyMapper allReplyMapper;

    private UserReplyMapper userReplyMapper;

    private ReplyConfigProperties replyConfigProperties;

    private VirtualUserIntegrationService virtualUserIntegrationService;

    @Override
    public Integer replyNum(Long newsId, Long userId, String areaCode) {
        List<NewsReplyDTO> allReplayList = cacheHolder.get(REPLY_COUNT_CACHE, newsId);

        if (CollectionUtils.isEmpty(allReplayList)) {
            return 0;
        }

        allReplayList = filter(allReplayList, userId, areaCode);

        return allReplayList.size();
    }

    @PostConstruct
    private void initCache() {
        SyncCacheHolderFactory.builder()
                .keyGenerator(HOT_REPLY_CACHE)
                .duration(5)
                .timeUnit(TimeUnit.MINUTES)
                .cacheLoader(this::queryHotReplyWithPostId)
                .build();
        SyncCacheHolderFactory.builder()
                .keyGenerator(REPLY_COUNT_CACHE)
                .duration(60)
                .timeUnit(TimeUnit.SECONDS)
                .cacheLoader(newsReplyMapper::queryTotalReplay)
                .build();
    }

    private List<NewsReplyDTO> queryHotReplyWithPostId(Long postId) {
        NewsReplyParam param = new NewsReplyParam();
        param.setNewsId(Long.valueOf(postId));

        return getHotReplayWithLimit(param, 10);
    }

    @Override
    public List<NewsReplyDTO> getCacheHotReply(Long postId, Long userId, String areaCode) {
        List<NewsReplyDTO> result = cacheHolder.get(HOT_REPLY_CACHE, postId);

        if (!CollectionUtils.isEmpty(result)) {
            // copy一份出去
            result = result.stream().map(reply -> {
                NewsReplyDTO res = new NewsReplyDTO();
                BeanUtils.copyProperties(reply, res);
                return res;
            }).collect(Collectors.toList());
        }

        result = filter(result, userId, areaCode);
        if (result.size() > 0) {
            NewsReplyDTO hotReply = result.get(0);
            hotReply.setList(null);
            return Collections.singletonList(hotReply);
        }

        return Lists.newArrayList();
    }

    private List<NewsReplyDTO> getHotReplayWithLimit(NewsReplyParam newsReplyParam, int limit) {
        List<NewsReplyDTO> allReplyList = newsReplyMapper.queryAllReplyWithContent(newsReplyParam.getNewsId(), newsReplyParam.getUserId());

        return execReplyLogic(allReplyList, newsReplyParam, false).stream()
                .filter(reply -> Objects.equals(reply.getHotReply(), 1))
                .limit(limit)
                .collect(Collectors.toList());
    }

    @Override
    public List<NewsReplyDTO> getHotReplays(NewsReplyParam newsReplyParam) {
        List<NewsReplyDTO> result = cacheHolder.get(HOT_REPLY_CACHE, newsReplyParam.getNewsId());
        return result.stream()
                .limit(forumProperties.getHotReplyViewNum())
                .map(reply -> {
                    NewsReplyDTO res = new NewsReplyDTO();
                    BeanUtils.copyProperties(reply, res);
                    return res;
                })
                .collect(Collectors.toList());
    }

    @Override
    public List<NewsReplyAllCountInfo> getAuditReplyCount(Date startTime, Date endTime) {
        return allReplyMapper.countAuditReply(startTime, endTime);
    }

    @Override
    public List<NewsReplyDTO> execReplyLogic(List<NewsReplyDTO> allReplyList, NewsReplyParam newsReplyParam, boolean isHotReplies) {
        // 处理不显示的数据
        allReplyList = filter(allReplyList, newsReplyParam.getUserId(), newsReplyParam.getAreaCode());
        if (allReplyList.size() == 0) {
            return allReplyList;
        }

        List<Long> replyLikesTemp = Collections.emptyList();
        Long authorId = 0L;

        if (Objects.nonNull(newsReplyParam.getUserId())) {
            if (JudgeUtil.isPost(newsReplyParam.getNewsId())) {
                authorId = forumPostInnerService.getPostUserId(newsReplyParam.getNewsId());
            }

            //查询该用户在该条新闻下的所有评论的额点赞记录
            replyLikesTemp = newsReplyMapper.selectReplyLike(newsReplyParam.getNewsId(), newsReplyParam.getUserId(), null);
        }

        List<Long> replyLikes = replyLikesTemp;
        Long finalAuthorId = authorId;

        // 填充数据
        allReplyList.forEach(reply -> {
            //若是帖子，则需要加上评论人是否楼主的判断
            reply.setUserIsAuthor(Objects.equals(reply.getUserId(), finalAuthorId));
            reply.setParentUserIsAuthor(Objects.equals(reply.getParentUserId(), finalAuthorId));

            //兼容老版本客户端
            if (Objects.equals(reply.getDeleteFlag(), 2)) {
                reply.setDeleteFlag((byte) 1);
                reply.setReplyContent("该评论已删除!");
            }

            reply.setReplyTime(DateBeautifyUtils.timeConvertString(reply.getAddTime()));
            //用户是否点赞
            reply.setIsLike(checkIsLike(replyLikes, reply.getId()));
        });

        // 是否是查询热评列表（非缓存查询）
        if (isHotReplies) {
            // 是 只需要处理子类点赞即可 因为这个数据在缓存的时候是无法设置的
            allReplyList.forEach(reply -> {
                if (!CollectionUtils.isEmpty(reply.getList())) {
                    reply.getList().forEach(sub -> {
                        dealSubMirrorDTO(sub, newsReplyParam.getUserId(), replyLikes);
                    });
                }
            });
            return allReplyList;
        } else {
            // 否则的话就是普通列表或者是热评的缓存列表查询 需要设置上下级关系
            // 获取顶级评论
            List<NewsReplyDTO> parentReplyList = allReplyList.stream()
                    .filter(reply -> Objects.equals(reply.getParentId(), 0L))
                    .collect(Collectors.toList());

            // 给顶级评论设置二级评论
            for (NewsReplyDTO parentReply : parentReplyList) {
                List<NewsReplyMirrorDTO> subReplyList = allReplyList.stream()
                        .filter(reply -> Objects.equals(reply.getRootId(), parentReply.getId()))
                        .sorted(comparing(NewsReplyDTO::getAddTime))
                        .map(reply -> {
                            NewsReplyMirrorDTO mirrorDTO = new NewsReplyMirrorDTO();
                            BeanUtils.copyProperties(reply, mirrorDTO);
                            return mirrorDTO;
                        })
                        .collect((Collectors.toList()));

                subReplyList.forEach(subReply -> {
                    dealSubMirrorDTO(subReply, newsReplyParam.getUserId(), replyLikes);
                });

                parentReply.setList(subReplyList);
            }

            return parentReplyList;
        }
    }

    private int checkIsLike(List<Long> replyLikes, Long replyId) {
        return replyLikes.contains(replyId) ? 1 : 0;
    }

    @Override
    public void dealSubMirrorDTO(NewsReplyMirrorDTO replay, Long userId, List<Long> replyLikes) {
        replay.setIsLike(checkIsLike(replyLikes, replay.getId()));
        //隐藏父级回复，详细需求问强哥(鬼知道这个是什么逻辑呀)
        if (replay.getLevel().equals(NewsConstant.REPLY_LEVEL_1)) {
            replay.setParentHeadImg(null);
            replay.setParentUserId(null);
            replay.setParentUserNickname(null);
        }
        replay.setReplyTime(DateBeautifyUtils.timeConvertString(replay.getAddTime()));
    }

    @Async
    @Override
    public void addPostReply(ForumPostVo forumPostVo) {
        try {
            //需求：评论显示时间：用户新人贴发布当天，马上跟上3-5个评论，然后在第二天跟上3-5个评论，然后第三天回复1-2个评论
            List<String> list = newsReplyMapper.selectRandomReplyLibrary(replyConfigProperties.getNewReportId(), 24);
            // 需要添加评论的数量 多给一点
            List<VirtualUser> userList = virtualUserIntegrationService.getRandom(24);
//                    virtualUserIntegrationService.getVirtualUserListNew(24,
//                    Lists.newArrayList(1, 2, 3),
//                    forumPostVo.getAreaCode());

            //马上跟上3-5个评论
            int choiceNum = ThreadLocalRandom.current().nextInt(3, 6);
            addNewsReplyDo(forumPostVo,
                    System.currentTimeMillis(),
                    subList(list, 0, choiceNum),
                    subList(userList, 0, choiceNum));

            //然后在第二天跟上3-5个评论
            choiceNum = ThreadLocalRandom.current().nextInt(3, 6);
            addNewsReplyDo(forumPostVo, (System.currentTimeMillis() + 3600 * 1000 * 24),
                    subList(list, 5, choiceNum),
                    subList(userList, 5, choiceNum));

            //然后第三天回复1-2个评论
            choiceNum = ThreadLocalRandom.current().nextInt(1, 2);
            addNewsReplyDo(forumPostVo, (System.currentTimeMillis() + 3600 * 1000 * 48),
                    subList(list, 10, choiceNum),
                    subList(userList, 10, choiceNum));

        } catch (Exception e) {
            log.error("增加评论失败, forumPostVo: {}", JSON.toJSONString(forumPostVo), e);
        }
    }


    private <T> List<T> subList(List<T> list, int start, int limit) {
        return list.stream().skip(start).limit(limit).collect(Collectors.toList());
    }

    private void addNewsReplyDo(ForumPostVo forumPostVo, long currentTime, List<String> list, List<VirtualUser> userList) {
        if (org.apache.commons.collections.CollectionUtils.isEmpty(list) || org.apache.commons.collections.CollectionUtils.isEmpty(userList)) {
            return;
        }

        int i = 0;
        for (String replyContent : list) {
            int nextTime = ThreadLocalRandom.current().nextInt(120000);
            VirtualUser virtualUser = userList.get(i);
            NewsReply newsReply = new NewsReply();
            newsReply.setAddTime(new Date(currentTime + nextTime));
            newsReply.setDeleteFlag((byte) 0);
            newsReply.setHeadImg(virtualUser.getHeadImg());
            newsReply.setId(SequenceHolder.nextLongId());
            newsReply.setLevel((byte) 0);
            newsReply.setLikeCount(0);
            newsReply.setNewsId(forumPostVo.getId());
            newsReply.setReplyContent(replyContent);
            //如果是帖子
            newsReply.setType((byte) 3);
            newsReply.setParentId(0L);
            //都设置为待展示，等待扫描
            newsReply.setStatus((byte) 0);
            newsReply.setUserId(virtualUser.getId());
            newsReply.setUserNickname(virtualUser.getNickname());
            newsReply.setInteractiveCount(0);
            newsReply.setRootId(0L);
            // 马甲号评论
            newsReply.setReplyUserType(1);
            newsReplyMapper.insertSelective(newsReply);
            UserReply userReply = new UserReply();
            BeanUtils.copyProperties(newsReply, userReply);
            userReplyMapper.insertSelective(userReply);
            i++;
        }
    }
}
