package com.bxm.localnews.news.service.impl;

import com.bxm.egg.user.enums.WarmRuleEnum;
import com.bxm.egg.user.param.UserWarmActionParam;
import com.bxm.localnews.integration.MessageService;
import com.bxm.localnews.integration.UserWarmIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.MemoryCacheKey;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.NewsReplyMapper;
import com.bxm.localnews.news.enums.ReplyStatusEnum;
import com.bxm.localnews.news.model.dto.NewsReplyDTO;
import com.bxm.localnews.news.model.param.ReplyOperateParam;
import com.bxm.localnews.news.model.vo.NewsReply;
import com.bxm.localnews.news.model.vo.ReplyOperate;
import com.bxm.localnews.news.service.HotReplayService;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.sync.core.CacheHolder;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @author liujia
 * @date 1/22/21 4:01 PM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class HotReplayServiceImpl implements HotReplayService {

    private NewsReplyMapper newsReplyMapper;

    private ForumProperties forumProperties;

    private MessageService messageService;

    private DistributedLock distributedLock;

    private CacheHolder cacheHolder;

    private UserWarmIntegrationService userWarmIntegrationService;

    private static final int PAGE_NUM = 50;

    @Override
    public void calHotReplay() {
        for (long index = 0; index < 10; index++) {
            queryHotReplayIterator(index, 0);
        }
    }

    /**
     * 递归查询不同分表的评论，检查获取热门评论
     *
     * @param index 分表索引
     * @param page  当前页数
     */
    private void queryHotReplayIterator(Long index, int page) {
        List<NewsReplyDTO> hotReplayList = newsReplyMapper.getNoHotReplayByLikeCount(index,
                forumProperties.getHotReplyForLikeCountNum(),
                page * PAGE_NUM, PAGE_NUM);

        if (hotReplayList.size() > 0) {
            setHotReplay(hotReplayList);

            if (hotReplayList.size() == PAGE_NUM) {
                queryHotReplayIterator(index, page + 1);
            }
        }
    }

    /**
     * 将符合条件的回复设置为热评
     *
     * @param hotReplayList 回复列表
     */
    private void setHotReplay(List<NewsReplyDTO> hotReplayList) {
        for (NewsReplyDTO newsReplyDTO : hotReplayList) {
            operateReply(ReplyOperateParam.builder()
                    .replyId(newsReplyDTO.getId())
                    .newsId(newsReplyDTO.getNewsId())
                    .operate(2)
                    .build());
        }
    }

    @Override
    public Boolean operateReply(ReplyOperateParam param) {
        NewsReply newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(param.getReplyId(), param.getNewsId());
        if (Objects.nonNull(newsReply)
                && !Objects.equals(newsReply.getStatus(), ReplyStatusEnum.IS_SHOW.getCode())) {
            log.warn("帖子: {} 评论: {}的状态不在展示状态 不能做处理", param.getNewsId(), param.getReplyId());
            return false;
        }

        ReplyOperate replyOperate = new ReplyOperate();
        replyOperate.setReplyId(param.getReplyId());
        replyOperate.setNewsId(param.getNewsId());
        switch (param.getOperate()) {
            case 1:
                replyOperate.setDisplayOwner(1);
                // 仅自己可见状态
                replyOperate.setStatus(ReplyStatusEnum.SELF.getCode());
                //发送消息召回
                messageService.sendMessageFilterForReply(param.getUserId(), param.getReplyId());
                break;
            case 2:
                replyOperate.setHotReply(1);
                replyOperate.setHotReplyTime(new Date());
                break;
            case 3:
                replyOperate.setHotReply(0);
                replyOperate.setHotReplyTime(new Date());
                break;
            default:
        }

        //锁key
        KeyGenerator lockKey = RedisConfig.REPLY_UPDATE_OWNER_HOT.copy()
                .appendKey(param.getUserId())
                .appendKey(param.getReplyId());
        if (distributedLock.lock(lockKey.gen())) {
            try {
                newsReplyMapper.updateDisplayOwnerAndHotReply(replyOperate);
                handOutWarm(replyOperate.getHotReply(), param.getReplyId());
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            } finally {
                distributedLock.unlock(lockKey.gen());
            }
        }

        cacheHolder.sendEvictCmd(MemoryCacheKey.HOT_REPLY_CACHE, param.getNewsId());
        cacheHolder.sendEvictCmd(MemoryCacheKey.REPLY_COUNT_CACHE, param.getNewsId());

        return Boolean.TRUE;
    }

    private void handOutWarm(Integer hotReply, Long replyId) {
        if (Objects.nonNull(hotReply)) {
            NewsReply newsReply = newsReplyMapper.selectByPrimaryKey(replyId);
            if (Objects.nonNull(newsReply) && Objects.nonNull(newsReply.getUserId())) {
                userWarmIntegrationService.updateWarm(UserWarmActionParam.builder()
                        .userId(newsReply.getUserId())
                        .warmRuleEnum(Objects.equals(hotReply, 1) ? WarmRuleEnum.COMMENT_UP_HOT : WarmRuleEnum.CANCEL_COMMENT_HOT)
                        .build());
            }
        }
    }

}
