package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.VirtualUserIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.GlobalConstant;
import com.bxm.localnews.news.domain.AdminAllReplyMapper;
import com.bxm.localnews.news.facade.service.ForumPostInnerService;
import com.bxm.localnews.news.model.dto.VirtualReplyDTO;
import com.bxm.localnews.news.model.dto.VirtualUserOverviewDTO;
import com.bxm.localnews.news.model.entity.VirtualReply;
import com.bxm.localnews.news.model.param.VirtualReplyListParam;
import com.bxm.localnews.news.service.VirtualReplyService;
import com.bxm.localnews.news.vo.PageWarper;
import com.bxm.newidea.component.tools.DateUtils;
import com.github.pagehelper.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.localnews.news.constant.GlobalConstant.GLOBAL_NAME;
import static com.github.pagehelper.page.PageMethod.startPage;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * 虚拟用户评论service
 *
 * @author gonzo
 * @date 2020-10-28 11:23
 **/
@Slf4j
@Service
@AllArgsConstructor
public class VirtualReplyServiceImpl implements VirtualReplyService {

    private final AdminAllReplyMapper adminAllReplyMapper;

    private final ForumProperties forumProperties;

    private final VirtualUserIntegrationService virtualUserIntegrationService;

    private final LocationIntegrationService locationIntegrationService;

    private final ForumPostInnerService forumPostInnerService;

    @Override
    public PageWarper<VirtualReplyDTO> listByParam(VirtualReplyListParam param) {
        Date now = new Date();
        Date start = DateUtils.getClearDate(DateUtils.addField(now, Calendar.DAY_OF_MONTH, forumProperties.getMaxReplyQueryDay() * -1));
        if (Objects.isNull(param.getStartTime())) {
            param.setStartTime(start);
        } else {
            // 如果开始时间超过了最长可查询的时间，则设置为最长查询的时间
            if (param.getStartTime().before(start)) {
                param.setStartTime(start);
            }
        }

        if (Objects.isNull(param.getEndTime())) {
            param.setEndTime(now);
        }

        Page<VirtualReply> page = startPage(param).doSelectPage(() -> adminAllReplyMapper.listVirtualReply(param));

        List<Long> postIdList = page.getResult().stream().map(VirtualReply::getNewsId).collect(Collectors.toList());
        Map<Long, List<String>> postRelationMap = forumPostInnerService.queryPostRelation(postIdList);

        PageWarper result = new PageWarper(page);
        result.setList(page.getResult().stream().map(replay -> {
            return convert(replay, postRelationMap);
        }).collect(Collectors.toList()));
        return result;
    }

    private VirtualReplyDTO convert(VirtualReply reply, Map<Long, List<String>> postRelationMap) {
        VirtualReplyDTO replyDTO = new VirtualReplyDTO();
        replyDTO.setId(reply.getId());
        replyDTO.setNewsId(reply.getNewsId());

        if (isBlank(reply.getTitle())) {
            replyDTO.setNewsTitle(reply.getTextField());
        } else {
            replyDTO.setNewsTitle(reply.getTitle());
        }

        if (Objects.nonNull(reply.getLevel()) && reply.getLevel().intValue() == 0) {
            replyDTO.setVirtualNickName(reply.getPostUserNickName());
            replyDTO.setVirtualUserId(reply.getPostUserId());
        } else {
            replyDTO.setVirtualNickName(reply.getParentUserNickname());
            replyDTO.setVirtualUserId(reply.getParentUserId());
        }
        replyDTO.setUserNickName(reply.getUserNickname());
        replyDTO.setUserId(reply.getUserId());
        replyDTO.setReplyStatus(Objects.isNull(reply.getVestInteract()) ? 0 : reply.getVestInteract().intValue());
        replyDTO.setReplyContent(reply.getReplyContent());
        replyDTO.setReplyTime(DateUtils.formatDateTime(reply.getAddTime()));

        VirtualUserOverviewDTO virtualUserOverviewDTO = virtualUserIntegrationService.getByIdCacheFirst(replyDTO.getVirtualUserId());

        if (Objects.nonNull(virtualUserOverviewDTO)
                && isNotBlank(virtualUserOverviewDTO.getAreaCode())
                && !Objects.equals(virtualUserOverviewDTO.getAreaCode(), GlobalConstant.GLOBAL_AREA)) {
            replyDTO.setVirtualAreaNames(locationIntegrationService.batchGetDetailJson(postRelationMap.get(reply.getNewsId())));
        } else {
            replyDTO.setVirtualAreaNames(GLOBAL_NAME);
        }

        return replyDTO;
    }
}
