package com.bxm.egg.user.integration;

import com.bxm.egg.user.constant.DomainScene;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.foundation.base.facade.enums.DomainViewScene;
import com.bxm.foundation.base.facade.param.DomainFetchParam;
import com.bxm.foundation.base.facade.service.DomainFacadeService;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboReference;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/9/7 8:44 下午
 */
@Service
public class DomainIntegrationService {
    @DubboReference(protocol = "dubbo",
            mock = "com.bxm.egg.user.integration.fallback.DomainFacadeServiceMock",
            cluster = "failover",
            version = "1.0.0",
            cache = "lru",
            check = false,
            lazy = true
    )
    private DomainFacadeService domainFacadeService;

    @Resource
    private UserProperties userProperties;

    /**
     * 根据场景获取域名
     *
     * @param scene     场景值
     * @param appId     appId 某些场景如微信环境的授权，是需要appId和域名相互绑定的 所以需要传入app id来获取对应的域名
     * @param viewScene 落地页场景值 如果选择的是站外分享，则需要指定具体的落地页类型 {@link DomainScene.DomainViewScene}
     * @return 包含了域名、baseUrl（如果有需要则会拼接）的信息
     */
    public String getBaseUrlInfoByScene(DomainScene scene, DomainScene.DomainViewScene viewScene,
                                        String appId) {
        DomainFetchParam param = new DomainFetchParam();

        param.setAppId(appId);
        param.setScene(com.bxm.foundation.base.facade.enums.DomainScene.forScene(scene.getScene()));
        param.setViewScene(DomainViewScene.forViewScene(viewScene.getScene()));
        return domainFacadeService.getBaseUrlInfo(param);
    }

    /**
     * 获取内部H5 base url
     *
     * @return 内部H5 base url
     */
    public String getInnerH5BaseUrl() {
        return domainFacadeService.getInnerH5BaseUrl(userProperties.getSrcApp());
    }

    /**
     * 获取分享的base url
     *
     * @param viewScene 分享落地页的类型
     * @return 分享的base url
     */
    public String getOutSideShareBaseUrl(DomainScene.DomainViewScene viewScene) {
        String baseUrlInfoByScene = getBaseUrlInfoByScene(DomainScene.OUTSIDE_SHARE, viewScene, null);
        return StringUtils.isNotBlank(baseUrlInfoByScene) ? baseUrlInfoByScene : "";
    }

}