package com.bxm.egg.user.integration;

import com.bxm.egg.common.url.ProtocolFactory;
import com.bxm.egg.facade.dto.ClientConfigParam;
import com.bxm.egg.facade.service.ConfigClientFacadeService;
import com.bxm.egg.message.facade.service.MessageFacadeService;
import com.bxm.egg.mq.common.constant.MessageTypeEnum;
import com.bxm.egg.mq.common.constant.PushMessageEnum;
import com.bxm.egg.mq.common.message.AddFriendMessage;
import com.bxm.egg.mq.common.model.dto.PushMessage;
import com.bxm.egg.mq.common.param.DingtalkMessage;
import com.bxm.egg.user.model.param.invite.ApplyFriendParam;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.DubboReference;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/9/7 8:05 下午
 */
@Slf4j
@Service
@Getter
public class MessageFacadeIntegrationService {

    @DubboReference(protocol = "dubbo",
            mock = "com.bxm.egg.user.integration.fallback.MessageIntegrationServiceMock",
            cluster = "failover",
            version = "1.0.0",
            check = false,
            lazy = true
    )
    private MessageFacadeService messageFacadeService;

    @DubboReference(protocol = "dubbo",
            mock = "com.bxm.egg.user.integration.fallback.ConfigClientFacadeServiceMock",
            cluster = "failover",
            version = "1.0.0",
            check = false,
            lazy = true
    )
    private ConfigClientFacadeService configClientFacadeService;


    public void sendPushMessage(PushMessage message) {
        messageFacadeService.sendPushMessage(message);
    }

    public Integer getUnReadMessageNum(Long userId, MessageTypeEnum messageType) {

        return messageFacadeService.getUnreadMsg(userId, messageType);
    }

    /**
     * 新增粉丝时，同步到消息服务，给用户个人消息中显示
     *
     * @param userId 对应的用户ID
     */
    public void addFollowMessage(Long userId) {
        PushMessage pushMessage = PushMessage.build();
        pushMessage.setPersistence(false);
        pushMessage.setTitle("新增粉丝");
        pushMessage.setContent("粉丝+1");
        pushMessage.getPayloadInfo().setType(PushMessageEnum.ADD_FUNS.getType());

        addMessage(pushMessage, userId);
    }

    /**
     * 发送钉钉消息
     *
     * @param message 钉钉消息内容
     */
    public void sendDingtalk(DingtalkMessage message) {
        messageFacadeService.sendDingtalk(message);
    }

    private void addMessage(PushMessage pushMessage, Long userId) {
        pushMessage.assign(userId);

        messageFacadeService.sendUserMessage(pushMessage);
    }

    /**
     * 发送好友申请推送
     *
     * @param senderId 发送用户id
     * @param nickName 发送用户昵称
     * @param targetId 目标用户id
     */
    @Async
    public void sendAddFriendMessage(Long senderId, String nickName, Long targetId) {

        PushMessage message = PushMessage.build("请求添加好友",
                nickName + "请求添加您为蛋蛋佳好友");
        message.assign(targetId);
        message.getPayloadInfo().setProtocol(ProtocolFactory.tabbarIndex().create().addAndRoid("message").addIos("TTMessageVC").build());
        messageFacadeService.sendPushMessage(message);

        AddFriendMessage addFriendMessage = AddFriendMessage.builder()
                .applyUserId(senderId)
                .content("朋友，添加我为你的好友吧~")
                .build();
        messageFacadeService.sendImMessage(addFriendMessage, senderId, new Long[]{targetId});
    }

    /**
     * 通过好友申请消息推送
     *
     * @param applyFriendParam 入参
     * @param nickName         发送用户昵称
     */
    @Async
    public void adoptFriendMessage(ApplyFriendParam applyFriendParam, String nickName) {
        PushMessage message = PushMessage.build("添加好友成功",
                "您已成功添加" + nickName + "为您的好友，现在可以去互动了~");
        message.assign(applyFriendParam.getApplyUserId());
        message.getPayloadInfo().setProtocol(ProtocolFactory.appH5().url(getFriendsFramePage(applyFriendParam)).build());
        messageFacadeService.sendPushMessage(message);
    }

    private String getFriendsFramePage(ApplyFriendParam friendParam) {
        return configClientFacadeService.getSpecificConfig(
                ClientConfigParam.builder()
                        .key("toFriendsFarmUrl")
                        .chnl(friendParam.getChnl())
                        .platform(friendParam.getPlatform())
                        .curVer(friendParam.getCurVer())
                        .build());
    }

}