package com.bxm.egg.user.attribute.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.bxm.egg.common.constant.InfoCombineStateEnum;
import com.bxm.egg.common.enums.UserStatusEnum;
import com.bxm.egg.user.attribute.UserPersonalInfoService;
import com.bxm.egg.user.attribute.UserTagService;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserInformationService;
import com.bxm.egg.user.info.msg.UserInfoChangeSender;
import com.bxm.egg.user.integration.SensitiveIntegrationService;
import com.bxm.egg.user.location.UserLocationService;
import com.bxm.egg.user.mapper.UserInformationMapper;
import com.bxm.egg.user.mapper.UserTagMapper;
import com.bxm.egg.user.model.PopCache;
import com.bxm.egg.user.model.dto.UserAccountMaterialGuideDTO;
import com.bxm.egg.user.model.dto.UserPersonalInfoDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.entity.UserInformationEntity;
import com.bxm.egg.user.model.entity.UserLocationEntity;
import com.bxm.egg.user.model.param.*;
import com.bxm.egg.user.model.vo.UserInfoComplete;
import com.bxm.egg.user.model.vo.UserTag;
import com.bxm.egg.user.model.vo.UserTagRedis;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.emoji.EmojiCodeParser;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import com.bxm.newidea.component.tools.CharUtil;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.*;

@Slf4j
@Service
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class UserPersonalInfoServiceImpl implements UserPersonalInfoService, ApplicationContextAware {

    /**
     * 引导资料完善弹窗key
     */
    private static final String KEY = "PERFECT_INFORMATION_POP";

    /**
     * 记录引导资料完善弹窗的时间
     */
    private static final String LAST_MODIFY_TIME = "lastModifyTime";

    /**
     * 正则表达式预编译[cq]
     */
    private static final Pattern WECHAT_NAME_PATTERN = Pattern.compile("^[a-zA-Z0-9_-]{6,20}$");

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserTagService userTagService;

    private final UserTagMapper userTagMapper;

    private final UserProperties userProperties;

    private final UserInformationMapper userInformationMapper;

    private final RedisSetAdapter redisSetAdapter;

    private ApplicationContext applicationContext;

    private final SensitiveIntegrationService sensitiveIntegrationService;

    private final UserInformationService userInformationService;

    private final UserLocationService userLocationService;

    private final UserInfoService userInfoService;

    private final UserInfoChangeSender userInfoChangeSender;

    @Override
    public Message saveCustomTag(CustomTagParam customTagParam) {

        //标签长度检测
        if (CharUtil.getWordCount(customTagParam.getLabel()) > 10) {
            return Message.build(false, "标签超过5个长度");
        }

        //判断标签是否包含敏感词汇
        if (sensitiveIntegrationService.verifySensitiveWord(customTagParam.getLabel())) {
            return Message.build(false, "标签含有敏感词词，请重新输入");
        }

        Message m = preExecute(customTagParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        UserInfoEntity userEntity = m.getParam("user");

        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(customTagParam.getUserId());

        //因为新创建的标签需要加在标签列表的最前面，所以需要获取标签的最小的顺序号
        List<UserTag> userTagList = userTagService.getUserTagFromRedisDb(customTagParam.getUserId());

        //判断已选标签是否小于15，否则抛出异常“标签不能超过15个哦”
        long count = userTagList.stream().filter(e -> 0 == e.getDeleteFlag()).count();
        if (count >= 15) {
            return Message.build(false, "标签不能超过15个哦");
        }

        int minOder = 0;

        //获得最小的顺序号
        if (!CollectionUtils.isEmpty(userTagList)) {
            minOder = userTagList.get(0).getOrder();
        } else {
            userTagList = new ArrayList<>();
        }

        //构建标签实体并入库
        UserTag userTag = UserTag.buildUserTag(customTagParam.getUserId(), minOder - 1, customTagParam.getLabel(), SequenceHolder.nextLongId());
        this.userTagMapper.insert(userTag);

        userTagList.add(0, userTag);

        //补全用户完善资料程度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();
        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(customTagParam.getUserId(), infoCompleteStatus));
        }

        afterExecute(userEntity, userInformationEntity, false, true, false, userTagList);

        //返回新增标签的实体，方便客户端回显
        return Message.build(true).addParam("tag", userTagService.convertUserTag2UserTagDto(userTag));
    }


    @Override
    public Message doUpdateUserTag(UserTagParam user) {
        log.debug("更新用户标签信息：[{}]", JSONObject.toJSON(user));
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        long count = user.getSpecificUserTagParamList().parallelStream().filter(e -> e.getChooseFlag() != 0).count();
        if (count > 15) {
            return Message.build(false, "标签选择个数限定15个");
        }

        UserInfoEntity userEntity = m.getParam("user");

        //用户资料信息
        UserInformationEntity userInformation = userInformationService.getUserInformationByUserId(user.getUserId());

        //更新标签
        List<UserTag> userTagList = new ArrayList<>();
        Date date = new Date();
        int index = 0;
        for (UserTagParam.SpecificUserTagParam specificUserTagParam : user.getSpecificUserTagParamList()) {
            //如果未选中并且是默认的，则不保存
            if (0 == specificUserTagParam.getChooseFlag() && 1 == specificUserTagParam.getDefaultFlag()) {
                continue;
            }
            UserTag userTag = UserTag.builder()
                    .createTime(date)
                    .id(SequenceHolder.nextLongId())
                    .label(specificUserTagParam.getLabel())
                    .order(index)
                    .defaultFlag(specificUserTagParam.getDefaultFlag())
                    .userId(user.getUserId())
                    .deleteFlag((byte) 0).build();
            //如果未选中并且是自定义的，则按<删除状态>保存
            if (0 == specificUserTagParam.getChooseFlag() && 0 == specificUserTagParam.getDefaultFlag()) {
                userTag.setDeleteFlag((byte) 1);
                userTag.setDeleteTime(date);
            }
            userTagList.add(userTag);
            index++;
        }

        this.userTagMapper.deleteByUserId(user.getUserId());

        //批量操作
        for (UserTag userTag : userTagList) {
            userTagMapper.insert(userTag);
        }

        //补全用户完善资料程度
        long infoCompleteStatus = userInformation.getInfoCompleteState();
        if (!CollectionUtils.isEmpty(user.getSpecificUserTagParamList())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
        } else {
            infoCompleteStatus = BitOperatorUtil.clearBit(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
        }
        userInformation.setInfoCompleteState(infoCompleteStatus);
        userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(user.getUserId(), infoCompleteStatus));

        afterExecute(userEntity, userInformation, false, true, false, userTagList);

        return Message.build(true);
    }

    @Override
    public Message doUpdateHeadImg(UserHeadImgParam userHeadImgParam) {
        Message m = preExecute(userHeadImgParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        UserInfoEntity userEntity = m.getParam("user");
        userEntity.setHeadImg(userHeadImgParam.getHeadImg());


        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userHeadImgParam.getUserId());

        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();

        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.HEAD_IMG_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HEAD_IMG_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userHeadImgParam.getUserId(), infoCompleteStatus));
        }

        UserInfoEntity updateUserHeadImgEntity = new UserInfoEntity();

        updateUserHeadImgEntity.setId(userHeadImgParam.getUserId());
        updateUserHeadImgEntity.setHeadImg(userHeadImgParam.getHeadImg());

        int count = userInfoService.updateUserInfo(updateUserHeadImgEntity);

        if (count <= 0) {
            return Message.build(false);
        }

        afterExecute(userEntity, userInformationEntity, true, false, false, null);

        return Message.build(true);
    }

    @Override
    public Message doUpdateNickname(UserNicknameParam userNicknameParam) {
        Message m = preExecute(userNicknameParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        if (sensitiveIntegrationService.verifySensitiveWord(userNicknameParam.getNickname())) {
            return Message.build(false, "昵称包含敏感词汇");
        }

        UserInfoEntity userEntity = m.getParam("user");

        userEntity.setNickname(EmojiCodeParser.replaceSoftbankEmoji(userNicknameParam.getNickname()));

        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userNicknameParam.getUserId());
        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();

        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.NICKNAME_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.NICKNAME_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userNicknameParam.getUserId(), infoCompleteStatus));
        }

        UserInfoEntity updateUserHeadImgEntity = new UserInfoEntity();

        updateUserHeadImgEntity.setId(userNicknameParam.getUserId());
        updateUserHeadImgEntity.setNickname(userNicknameParam.getNickname());

        int count = userInfoService.updateUserInfo(updateUserHeadImgEntity);
        if (count > 0) {
            afterExecute(userEntity, userInformationEntity, true, false, false, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateBirth(UserBirthParam userBirthParam) {
        Message m = preExecute(userBirthParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        UserInfoEntity userEntity = m.getParam("user");

        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userBirthParam.getUserId());

        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();

        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.BIRTHDAY_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.BIRTHDAY_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userBirthParam.getUserId(), infoCompleteStatus));
        }

        int count = userInformationService.updateUserBirthday(userBirthParam.getUserId(), userBirthParam.getBirthday());

        if (count > 0) {
            afterExecute(userEntity, userInformationEntity, false, false, true, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateHometown(UserHomeTownParam userHomeTownParam) {
        Message m = preExecute(userHomeTownParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        UserInfoEntity userEntity = m.getParam("user");

        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userHomeTownParam.getUserId());

        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();


        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.HOMETOWN_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HOMETOWN_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userHomeTownParam.getUserId(), infoCompleteStatus));
        }

        int count = userLocationService.updateUserHometownInfo(userHomeTownParam);

        if (count > 0) {
            afterExecute(userEntity, userInformationEntity, false, false, true, null);
        }

        return Message.build(true);
    }


    /**
     * 前置操作：判断用户存在与否 && 用户是否处于正常状态
     *
     * @param userId 用户id
     */
    private Message preExecute(Long userId) {
        UserInfoEntity user = userInfoService.selectUserById(userId);

        //判断用户是否存在
        if (null == user) {
            return Message.build(false, "用户不存在");
        }

        //判断用户登录状态
        if (UserStatusEnum.ENABLE.getCode() != user.getState()) {
            // 如果状态不正常 修改为正常状态
            log.error("出现暂未定位原因的异常数据，将其更新为正常状态，用户ID：{}", userId);

            return Message.build(false);
        }

        return Message.build(true).addParam("user", user);
    }

    /**
     * 后置操作：更新用户冗余信息 & 判断是否完善个人资料 & 保存用户信息至redis中
     *
     * @param user               用户实体
     * @param isRedundancyChange 是否需要更新用户冗余信息
     * @param isTagChange        是否更改了用户的兴趣标签
     * @param userTagList        用户的兴趣标签，如果没有则没有值
     */
    private void afterExecute(
            UserInfoEntity user,
            UserInformationEntity userInformationEntity,
            boolean isRedundancyChange,
            boolean isTagChange,
            boolean isPush,
            List<UserTag> userTagList) {

        //如果是改了兴趣的话，得到修改的兴趣标签并保存
        if (isTagChange) {
            UserTagRedis userTagRedis = UserTagRedis.buildUserTagRedis(user.getId(), userTagList);
            this.redisHashMapAdapter.put(RedisConfig.USER_TAG, String.valueOf(user.getId()), JSON.toJSONString(userTagRedis));
        }
        //更新用户资料清空缓存
        userInfoChangeSender.sendUserChangeMsg(user.getId());
    }


    @Override
    public Message doUpdateEmotion(UserEmotionParam userEmotionParam) {
        Message m = preExecute(userEmotionParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        UserInfoEntity userEntity = m.getParam("user");
        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userEmotionParam.getUserId());

        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();

        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.RELATION_SHIP_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.RELATION_SHIP_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userEmotionParam.getUserId(), infoCompleteStatus));
        }

        int count = updateUserEmotion(userEmotionParam);

        if (count > 0) {
            afterExecute(userEntity, userInformationEntity, false, false, false, null);
        }

        return Message.build(true);
    }

    /**
     * 更新用户情感状态
     *
     * @param userEmotionParam 入参
     * @return 影响行数
     */
    private int updateUserEmotion(UserEmotionParam userEmotionParam) {
        UserInformationEntity updateUserInformationEntity = new UserInformationEntity();

        updateUserInformationEntity.setRelationshipStatus(userEmotionParam.getRelationshipStatus());
        LambdaQueryWrapper<UserInformationEntity> updateQueryWrapper = new LambdaQueryWrapper<>();

        updateQueryWrapper.eq(UserInformationEntity::getUserId, userEmotionParam.getUserId());

        return userInformationMapper.update(updateUserInformationEntity, updateQueryWrapper);
    }

    @Override
    public Message doUpdateJob(UserJobParam userJobParam) {
        log.debug("更新职业,参数为:[{}]", JSONObject.toJSON(userJobParam));
        Message m = preExecute(userJobParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        UserInfoEntity userEntity = m.getParam("user");

        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userJobParam.getUserId());
        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();

        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.JOB_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.JOB_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);

            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userJobParam.getUserId(), infoCompleteStatus));
        }

        int count = updateJobInfo(userJobParam);

        if (count > 0) {
            afterExecute(userEntity, userInformationEntity, false, false, true, null);
        }

        return Message.build(true);
    }

    /**
     * 更新工作信息
     *
     * @param userJobParam 工作信息
     */
    private int updateJobInfo(UserJobParam userJobParam) {
        UserInformationEntity updateUserInformation = new UserInformationEntity();

        updateUserInformation.setCompany(userJobParam.getCompany());
        updateUserInformation.setJobTitle(userJobParam.getJobTitle());
        updateUserInformation.setJobCategory(userJobParam.getJobCategory());
        updateUserInformation.setJobCategoryName(userJobParam.getJobCategoryName());
        updateUserInformation.setJobType(userJobParam.getJobType());
        updateUserInformation.setJobTypeName(userJobParam.getJobTypeName());

        LambdaQueryWrapper<UserInformationEntity> updateQueryWrapper = new LambdaQueryWrapper<>();

        updateQueryWrapper.eq(UserInformationEntity::getUserId, userJobParam.getUserId());

        return userInformationMapper.update(updateUserInformation, updateQueryWrapper);
    }

    @Override
    public Message doUpdatePersonalProfile(UserIntroductionParam userIntroductionParam) {
        //判断标签是否包含敏感词汇
        if (sensitiveIntegrationService.verifySensitiveWord(userIntroductionParam.getPersonalProfile())) {
            return Message.build(false, "个人简介含有敏感词，请重新输入");
        }

        Message m = preExecute(userIntroductionParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        UserInfoEntity userEntity = m.getParam("user");

        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userIntroductionParam.getUserId());
        userInformationEntity.setPersonalProfile(userIntroductionParam.getPersonalProfile());

        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();

        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.PERSONAL_PROFILE_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil
                    .setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PERSONAL_PROFILE_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userIntroductionParam.getUserId(), infoCompleteStatus));

        }

        int count = updatePersonalProfileInfo(userIntroductionParam);

        if (count > 0) {
            afterExecute(userEntity, userInformationEntity, false, false, false, null);
        }

        return Message.build(true);
    }

    /**
     * 更新个人简介信息
     *
     * @param userIntroductionParam 个人简介信息
     */
    private int updatePersonalProfileInfo(UserIntroductionParam userIntroductionParam) {
        UserInformationEntity updateUserInformation = new UserInformationEntity();

        updateUserInformation.setPersonalProfile(userIntroductionParam.getPersonalProfile());

        LambdaQueryWrapper<UserInformationEntity> updateQueryWrapper = new LambdaQueryWrapper<>();
        updateQueryWrapper.eq(UserInformationEntity::getUserId, userIntroductionParam.getUserId());

        return userInformationMapper.update(updateUserInformation, updateQueryWrapper);
    }

    @Override
    public Message doUpdateUserSex(UserSexParam userSexParam) {
        Message m = preExecute(userSexParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        //如果不传性别则不做改动
        if (null == userSexParam.getSex() || 0 == userSexParam.getSex()) {
            return Message.build(true);
        }

        UserInfoEntity userEntity = m.getParam("user");
        userEntity.setSex((int) userSexParam.getSex());

        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userSexParam.getUserId());

        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();

        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.SEX_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.SEX_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userSexParam.getUserId(), infoCompleteStatus));

        }

        UserInfoEntity updateUserInfoEntity = new UserInfoEntity();

        updateUserInfoEntity.setId(userSexParam.getUserId());
        updateUserInfoEntity.setSex((int) userSexParam.getSex());

        int count = userInfoService.updateUserInfo(updateUserInfoEntity);

        if (count > 0) {
            afterExecute(userEntity, userInformationEntity, true, false, true, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateUserWechat(UserWechatParam userWechatParam) {
        Message m = preExecute(userWechatParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        Message isValid = validWechat(userWechatParam.getWechatId());
        if (!isValid.isSuccess()) {
            return isValid;
        }

        UserInfoEntity userEntity = m.getParam("user");

        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userWechatParam.getUserId());

        userEntity.setWeixin(userWechatParam.getWechatId());

        //更新完善资料度
        long infoCompleteStatus = userInformationEntity.getInfoCompleteState();


        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.WECHAT_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.WECHAT_COMPLETE.getIndex());
            userInformationEntity.setInfoCompleteState(infoCompleteStatus);
            userInformationMapper.updateUserInfoCompleteStatus(new UserInfoComplete(userWechatParam.getUserId(), infoCompleteStatus));

        }
        UserInfoEntity updateUserInfoEntity = new UserInfoEntity();

        updateUserInfoEntity.setId(userWechatParam.getUserId());
        updateUserInfoEntity.setWeixin(userWechatParam.getWechatId());

        int count = userInfoService.updateUserInfo(updateUserInfoEntity);

        if (count > 0) {
            afterExecute(userEntity, userInformationEntity, true, false, true, null);
        }

        return Message.build(true);
    }

    /**
     * 判断微信号是否合法
     */
    private Message validWechat(String qqorwx) {
        if (StringUtils.isEmpty(qqorwx) || qqorwx.length() < 6 || qqorwx.length() > 20) {
            return Message.build(false, "请输入6-20个字符，支持字母，数字，下划线和减号");
        }
        //QQ号最短5位，微信号最短是6位最长20位
        Matcher m = WECHAT_NAME_PATTERN.matcher(qqorwx);
        boolean isValid = m.matches();
        if (!isValid) {
            return Message.build(false, "请不要输入中文和其他特殊符号");
        }
        return Message.build(true);
    }

    @Override
    public UserPersonalInfoDTO getUserPersonalInfo(Long userId) {

        //获取用户
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(userId);
        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userId);
        UserLocationEntity userLocationEntity = userLocationService.getUserLocationByUserId(userId);

        if (null == userInfoEntity) {
            return null;
        }

        //得到个人资料完善度百分比
        int infoCompletePercent = InfoCombineStateEnum.getInfoCompletePercent(
                userInformationEntity.getInfoCompleteState() == null ? 0 : userInformationEntity.getInfoCompleteState()
        );

        String infoCompleteTaskText = userProperties.getInfoCompleteTaskText();

        //获取用户标签列表
        List<UserTag> userTagList = userTagService.getUserTagFromRedisDb(userId);

        //包装用户实体
        UserPersonalInfoDTO userPersonalInfoDTO = convertUserInfo2UserPersonalInfo(userInfoEntity, userInformationEntity
                , userLocationEntity);
        userPersonalInfoDTO.setInfoCompletePercent(infoCompletePercent);
        userPersonalInfoDTO.setInfoCompleteTaskText(infoCompleteTaskText);

        //包装用户标签
        userPersonalInfoDTO.setUserTagList(userTagList.stream()
                .filter(e -> e.getDeleteFlag() == 0)
                .map(UserTag::getLabel).collect(Collectors.toList()));

        return userPersonalInfoDTO;

    }

    /**
     * 用途如其名
     */
    private UserPersonalInfoDTO convertUserInfo2UserPersonalInfo(UserInfoEntity userInfoEntity,
                                                                 UserInformationEntity userInformationEntity,
                                                                 UserLocationEntity userLocationEntity) {
        //包装用户实体
        UserPersonalInfoDTO.IndustryInfo industryInfo = null;
        if (null != userInformationEntity) {
            industryInfo = UserPersonalInfoDTO.IndustryInfo.builder()
                    .company(userInformationEntity.getCompany())
                    .jobCategory(userInformationEntity.getJobCategory())
                    .jobCategoryName(userInformationEntity.getJobCategoryName())
                    .jobTitle(userInformationEntity.getJobTitle())
                    .jobType(userInformationEntity.getJobType())
                    .jobTypeName(userInformationEntity.getJobTypeName()).build();
        }

        return UserPersonalInfoDTO.builder()
                .personalProfile(userInformationEntity.getPersonalProfile())
                .isDefaultPersonalProfile(isEmpty(userInformationEntity.getPersonalProfile()))
                .birthday(userInformationEntity.getBirthday())
                .headImg(userInfoEntity.getHeadImg())
                .industryInfo(industryInfo)
                .industry(getIndustryInfo(userInformationEntity))
                // 3.4.0版本不需要用户引导 志化说返回1就可以去掉引导
                .newbieGuideFlag((byte) 1)
                //3.0.0 取消临时昵称，允许修改
                .isTempNickName((byte) 1)
                .generation(getGeneration(userInformationEntity.getBirthday()))
                .nickname(userInfoEntity.getNickname())
                .phone(userInfoEntity.getPhone())
                .id(userInfoEntity.getId())
                .relationshipStatus(userInformationEntity.getRelationshipStatus())
                .sex(userInfoEntity.getSex())
                .wechatId(userInfoEntity.getWeixin())
                .hometownCode(userLocationEntity.getHometownCode())
                .hometownName(userLocationEntity.getHometownName())
                .build();
    }

    private String getIndustryInfo(UserInformationEntity userInformationEntity) {
        //处理职业信息的展示
        StringBuilder industryBuilder = new StringBuilder();
        //公司和职位不为空（这几个判断有点蠢，待优化）
        if (StringUtils.isNotBlank(userInformationEntity.getJobTitle()) && StringUtils.isNotBlank(userInformationEntity.getCompany())) {
            industryBuilder.append(userInformationEntity.getCompany()).append(userInformationEntity.getJobTitle());
        }
        //公司和职位为空
        if (isBlank(userInformationEntity.getJobTitle()) && isBlank(userInformationEntity.getCompany()) && StringUtils
                .isNotBlank(userInformationEntity.getJobCategoryName())) {
            industryBuilder.append(userInformationEntity.getJobCategoryName());
        }
        //职业不为空，公司为空
        if (StringUtils.isNotBlank(userInformationEntity.getJobTitle()) && isBlank(userInformationEntity.getCompany()) && StringUtils
                .isNotBlank(userInformationEntity.getJobCategoryName())) {
            industryBuilder.append(userInformationEntity.getJobTitle());
        }
        //职业为空，公司不为空
        if (isBlank(userInformationEntity.getJobTitle()) && StringUtils.isNotBlank(userInformationEntity.getCompany())) {
            industryBuilder.append(userInformationEntity.getCompany());
        }
        if (industryBuilder.length() > 0) {
            return industryBuilder.toString();
        }
        return null;
    }

    @Override
    public Long initUserInfoCompleteStatus(UserInfoEntity userInfoEntity,
                                           UserInformationEntity userInformationEntity,
                                           UserLocationEntity userLocationEntity) {
        long infoCompleteStatus = 0L;
        //用户在登录时可能携带 性别、昵称、头像等信息，需要将这些作为完善资料度的参考
        if (null != userInfoEntity.getSex() && 0 != userInfoEntity.getSex()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.SEX_COMPLETE.getIndex());
        }
        if (isNotBlank(userInfoEntity.getNickname())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.NICKNAME_COMPLETE.getIndex());
        }
        if (isNotBlank(userInfoEntity.getHeadImg())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HEAD_IMG_COMPLETE.getIndex());
        }
        if (null != userInformationEntity.getBirthday()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.BIRTHDAY_COMPLETE.getIndex());
        }
        if (null != userInformationEntity.getRelationshipStatus()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.RELATION_SHIP_COMPLETE.getIndex());
        }
        if (isNotBlank(userLocationEntity.getHometownCode())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HOMETOWN_COMPLETE.getIndex());
        }
        if (null != userInformationEntity.getJobCategory()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.JOB_COMPLETE.getIndex());
        }
        if (isNotBlank(userInformationEntity.getPersonalProfile())) {
            infoCompleteStatus = BitOperatorUtil
                    .setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PERSONAL_PROFILE_COMPLETE.getIndex());
        }
        if (isNotBlank(userInfoEntity.getPhone())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PHONE_COMPLETE.getIndex());
        }
        if (isNotBlank(userInfoEntity.getWeixin())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.WECHAT_COMPLETE.getIndex());
        }
        List<UserTag> userTagList = userTagService.getUserTagFromRedisDb(userInfoEntity.getId());
        if (!CollectionUtils.isEmpty(userTagList)) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
        }
        return infoCompleteStatus;
    }

    @Override
    public Boolean getCompleteInformationWindow(Long userId) {
        log.debug("获取用户完善个人资料弹窗，userId:[{}]", userId);
        //用户资料完成度
        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userId);

        int status = InfoCombineStateEnum.getInfoUnCompleteStatus(userInformationEntity.getInfoCompleteState());
        //当用户资料未完成项>=2时 ,需要弹出该弹窗
        if (status >= 2) {
            KeyGenerator cacheKey = RedisConfig.USER_CACHE_POPUP.copy().appendKey(userId);
            //初始化缓存map，比对长度后增量
            Map<String, PopCache> cacheMap = redisHashMapAdapter.entries(cacheKey, PopCache.class);
            PopCache popCache = cacheMap.get(KEY);
            if (Objects.isNull(popCache)) {
                popCache = new PopCache();
            }
            //上次弹窗的时间
            Long popTime = popCache.getParam(LAST_MODIFY_TIME);
            long timeMillis = System.currentTimeMillis();

            //缓存时间为空或者当前时间与上次弹窗时间相隔7天,需弹窗
            if (popTime == null || (timeMillis - popTime) > DateUtils.WEEK_MILLISECOND) {
                popCache.setPopUp(Boolean.TRUE);
                popCache.addParam(LAST_MODIFY_TIME, System.currentTimeMillis());
                //兼容首页次数弹窗
                Integer num = popCache.getParam("num");
                if (Objects.isNull(num)) {
                    num = 1;
                } else {
                    num += 1;
                }
                popCache.addParam("num", num);
                redisHashMapAdapter.put(cacheKey, KEY, popCache);
                log.debug("弹出个人资料引导弹窗:[{}]", JSONObject.toJSONString(popCache));
                return Boolean.TRUE;
            } else {
                return Boolean.FALSE;
            }
        }
        return Boolean.FALSE;
    }

    /**
     * 年龄段组装
     */
    private String getGeneration(Date brithday) {
        String generationStr = "";
        if (null != brithday) {
            LocalDate birthday = brithday.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            String gen = getGeneration(birthday);
            if (!org.springframework.util.StringUtils.isEmpty(gen)) {
                generationStr = gen;
            }
        }
        return generationStr;
    }

    private String getGeneration(LocalDate birthday) {
        String generationStr = null;
        if (null != birthday) {
            int year = birthday.getYear();
            int decade = year / 10;
            String s = String.valueOf(decade);
            generationStr = s.substring(s.length() - 1) + "0后";
        }
        return generationStr;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public UserAccountMaterialGuideDTO getAccountMaterialGuide(Long userId) {
        //新版本第一次调用接口redis都是null进入初始值赋值
        Boolean tips = redisHashMapAdapter.get(RedisConfig.ACCOUNT_MATERIAL_TIPS.copy(), userId.toString(), Boolean.class);
        if (tips == null) {
            //只展示一次tips
            redisHashMapAdapter.put(RedisConfig.ACCOUNT_MATERIAL_TIPS.copy(), userId.toString(), Boolean.FALSE);
        }
        Boolean perfect = redisHashMapAdapter.get(RedisConfig.ACCOUNT_MATERIAL_PERFECT.copy(), userId.toString(), Boolean.class);
        Set<Integer> allMembers = redisSetAdapter.getAllMembers(RedisConfig.USER_LATELY_VISIT_NUM.copy().appendKey(userId.toString()), Integer.class);
        int infoCompletePercent = 0;
        int perfectPercent = 100;
        if (perfect == null || !perfect) {
            infoCompletePercent = this.getInfoCompletePercent(userId);
            if (infoCompletePercent == perfectPercent) {
                perfect = Boolean.TRUE;
                //tips的redis为false,不展示tips
                tips = Boolean.FALSE;
            } else {
                perfect = Boolean.FALSE;
            }
            redisHashMapAdapter.put(RedisConfig.ACCOUNT_MATERIAL_PERFECT.copy(), userId.toString(), perfect);
        }
        UserAccountMaterialGuideDTO userAccountMaterialGuideDTO = UserAccountMaterialGuideDTO.builder().build();
        userAccountMaterialGuideDTO.setInfoCompletePercent(infoCompletePercent);
        //tips只展示一次，所以tips==null时且infoCompletePercent<100则展示，否则都不展示
        userAccountMaterialGuideDTO.setTips(tips == null ? Boolean.TRUE : tips);
        userAccountMaterialGuideDTO.setCurrInterviewNum(allMembers == null ? 0 : allMembers.size());
        userAccountMaterialGuideDTO.setPerfect(perfect);
        return userAccountMaterialGuideDTO;
    }

    private int getInfoCompletePercent(Long userId) {
        //获取用户
        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userId);
        if (null == userInformationEntity) {
            return 0;
        }
        //得到个人资料完善度百分比
        return InfoCombineStateEnum.getInfoCompletePercent(userInformationEntity.getInfoCompleteState() == null ? 0 : userInformationEntity.getInfoCompleteState());
    }
}
