package com.bxm.egg.user.facade;

import com.alibaba.fastjson.JSON;
import com.bxm.egg.common.enums.UserStatusEnum;
import com.bxm.egg.user.facade.bo.UserInfoBO;
import com.bxm.egg.user.facade.bo.UserStatisticsBO;
import com.bxm.egg.user.facade.service.UserInfoFacadeService;
import com.bxm.egg.user.info.UserInfoCacheService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserInformationService;
import com.bxm.egg.user.info.UserStatisticsService;
import com.bxm.egg.user.model.bo.UserCacheInfoBO;
import com.bxm.egg.user.model.bo.UserLocationBO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.entity.UserInformationEntity;
import com.bxm.egg.user.model.entity.UserStatisticsEntity;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.beans.BeanUtils;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/9/27 8:11 下午
 */
@DubboService(protocol = {"dubbo"}, version = "1.0.0")
@Slf4j
public class UserInfoFacadeServiceImpl implements UserInfoFacadeService {

    @Resource
    private UserInfoCacheService userInfoCacheService;

    @Resource
    private UserStatisticsService userStatisticsService;

    @Resource
    private UserInformationService userInformationService;

    @Resource
    private UserInfoService userInfoService;

    @Override
    public UserInfoBO selectUserFromCache(Long userId) {

        if (Objects.isNull(userId)) {
            log.error("用户传递的用户id为null,返回默认用户信息");
            return buildInvalidUserCacheInfo(userId);
        }

        UserCacheInfoBO userCacheInfoBO = userInfoCacheService.load(userId);

        UserInfoBO userInfoBO = new UserInfoBO();

        BeanUtils.copyProperties(userCacheInfoBO, userInfoBO);

        UserLocationBO userLocationBO = userCacheInfoBO.getUserLocationBO();
        if (userLocationBO != null) {
            userInfoBO.setLocationCode(userLocationBO.getLocationCode());
            userInfoBO.setLocationName(userLocationBO.getLocationName());
        }

        if (log.isDebugEnabled()) {
            log.debug("内部接口查询用户缓存，用户id:{}, 用户信息：{}", userId, JSON.toJSONString(userCacheInfoBO));
        }

        return userInfoBO;
    }

    @Override
    public List<UserInfoBO> selectUserListFromCache(List<Long> userIdList) {

        if (CollectionUtils.isEmpty(userIdList)) {
            return Collections.EMPTY_LIST;
        }
        return userIdList.stream().map(this::selectUserFromCache).collect(Collectors.toList());
    }

    @Override
    public Map<Long, UserInfoBO> selectUserMapFromCache(List<Long> userIdList) {
        return selectUserListFromCache(userIdList).stream()
                .collect(
                        Collectors.groupingBy(UserInfoBO::getUserId,
                                Collectors.collectingAndThen(Collectors.toList(),
                                        value -> value.get(0))));
    }

    @Override
    public UserStatisticsBO getUserStatisticsInfo(Long userId) {

        UserStatisticsEntity userStatisticsEntity = userStatisticsService.selectUserStatisticsByUserId(userId);

        UserStatisticsBO userStatisticsBO = new UserStatisticsBO();
        if (userStatisticsEntity != null) {
            BeanUtils.copyProperties(userStatisticsEntity, userStatisticsBO);
        }

        return userStatisticsBO;
    }

    @Override
    public Long getInfoCompleteState(Long userId) {
        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userId);

        if (userInformationEntity != null) {
            return userInformationEntity.getInfoCompleteState();
        }
        return 0L;
    }

    @Override
    public Boolean isNewPlatformUser(Long userId) {
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(userId);

        if (userInfoEntity == null) {
            return false;
        }

        Integer eggUser = userInfoEntity.getEggUser();

        return eggUser == 0;
    }

    private UserInfoBO buildInvalidUserCacheInfo(Long userId) {
        UserInfoBO userCacheInfoBO = new UserInfoBO();

        userCacheInfoBO.setHeadImg("http://bxm-news.oss-cn-hangzhou.aliyuncs.com/img/defaultHeadImg.png");
        userCacheInfoBO.setUserId(userId);
        userCacheInfoBO.setNickname("未知用户");
        userCacheInfoBO.setState(UserStatusEnum.DISABLE.getCode());

        return userCacheInfoBO;
    }

}