package com.bxm.egg.user.info.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.bxm.egg.user.constant.BaseDistributeKeyConstant;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserWechatGrantService;
import com.bxm.egg.user.info.msg.UserInfoChangeSender;
import com.bxm.egg.user.mapper.UserGrantWechatAppMapper;
import com.bxm.egg.user.model.bo.UserWechatBindBO;
import com.bxm.egg.user.model.entity.UserGrantWechatAppEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.UserBindWxParam;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/10/8 5:50 下午
 */
@Slf4j
@Service
public class UserWechatGrantServiceImpl implements UserWechatGrantService {

    @Resource
    private UserGrantWechatAppMapper userGrantWechatAppMapper;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private UserInfoService userInfoService;

    @Resource
    private UserInfoChangeSender userInfoChangeSender;

    @Resource
    private UserExtendService userExtendService;

    @Override
    public Optional<UserGrantWechatAppEntity> userInfoUnionIdExist(String unionId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUnionId, unionId);
        queryWrapper.eq(UserGrantWechatAppEntity::getDeleted, 0);

        UserGrantWechatAppEntity userGrantWechatAppEntity = userGrantWechatAppMapper.selectOne(queryWrapper);

        if (Objects.nonNull(userGrantWechatAppEntity)) {
            return Optional.of(userGrantWechatAppEntity);
        }

        return Optional.empty();
    }

    @Override
    public int updateWechatGrantBindUserId(Long userId, String unionId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUnionId, unionId);

        UserGrantWechatAppEntity updateEntity = new UserGrantWechatAppEntity();

        updateEntity.setUserId(userId);

        return userGrantWechatAppMapper.update(updateEntity, queryWrapper);

    }

    @Override
    public Message userBindWx(UserBindWxParam userBindWxParam) {

        String resource = buildBindWxUserLockKey(userBindWxParam.getUserId());

        if (!distributedLock.lock(resource)) {
            log.info("出现重复点击，请求参数：{}", JSON.toJSONString(userBindWxParam));
            return null;
        }

        //判断用户是否已经绑定微信
        if (userHasBindWx(userBindWxParam.getUserId())) {
            distributedLock.unlock(resource);
            return Message.build(false, "该用户已经绑定过微信，请勿重复绑定");
        }

        UserWechatBindBO userWechatBindBO = new UserWechatBindBO();
        BeanUtils.copyProperties(userBindWxParam, userWechatBindBO);

        //添加微信授权记录
        grantBindWxRecord(userWechatBindBO);

        //微信授权后更新用户头像等信息
        updateUserWxBindRelationInfo(userBindWxParam);

        //清空用户信息缓存
        userInfoChangeSender.sendUserChangeMsg(userBindWxParam.getUserId());

        //初始化资料完成度
        userExtendService.initComplete(userBindWxParam.getUserId());

        distributedLock.unlock(resource);
        return Message.build(true);
    }

    /**
     * 微信授权后更新用户头像等信息
     *
     * @param userBindWxParam 微信绑定入参
     */
    private void updateUserWxBindRelationInfo(UserBindWxParam userBindWxParam) {
        UserInfoEntity updateUserInfo = new UserInfoEntity();

        String newHeadImg = userExtendService.uploadWechatImage(userBindWxParam.getHeadImageUrl(),
                userBindWxParam,
                userBindWxParam.getUserId());

        updateUserInfo.setId(userBindWxParam.getUserId());
        updateUserInfo.setHeadImg(newHeadImg);
        updateUserInfo.setNickname(userBindWxParam.getNickName());
        updateUserInfo.setSex(userBindWxParam.getSex());

        userInfoService.updateUserInfo(updateUserInfo);
    }

    /**
     * 判断用户是否已经绑定微信
     *
     * @param userId 用户id
     * @return 是否绑定
     */
    @Override
    public Boolean userHasBindWx(Long userId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUserId, userId);
        queryWrapper.eq(UserGrantWechatAppEntity::getDeleted, 0);

        List<UserGrantWechatAppEntity> grantWechatAppEntityList = userGrantWechatAppMapper.selectList(queryWrapper);

        return !CollectionUtils.isEmpty(grantWechatAppEntityList);
    }

    @Override
    public Boolean grantBindWxRecord(UserWechatBindBO userWechatBindBO) {
        // 保留微信授权信息
        UserGrantWechatAppEntity entity = new UserGrantWechatAppEntity();
        entity.setId(SequenceHolder.nextLongId());

        entity.setCountry(userWechatBindBO.getCountry());
        entity.setProvince(userWechatBindBO.getProvince());
        entity.setCity(userWechatBindBO.getCity());
        entity.setNickName(userWechatBindBO.getNickName());
        entity.setHeadImg(userWechatBindBO.getHeadImageUrl());
        entity.setOpenId(userWechatBindBO.getOpenId());
        entity.setUnionId(userWechatBindBO.getUnionId());
        entity.setSex(userWechatBindBO.getSex());

        entity.setUserId(userWechatBindBO.getUserId());
        entity.setCreateTime(new Date());

        return userGrantWechatAppMapper.insert(entity) > 0;
    }

    /**
     * 构建用户绑定微信分布式锁，保证不重复请求
     *
     * @param userId 用户id
     * @return 分布式锁key
     */
    private String buildBindWxUserLockKey(Long userId) {
        return BaseDistributeKeyConstant.USER_BIND_WX_LOCK_KEY.copy().appendKey(userId).gen();
    }
}