package com.bxm.egg.user.invite.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.enums.AwardTypeEnum;
import com.bxm.egg.user.enums.InviteRecordStateEnum;
import com.bxm.egg.user.enums.InviteRecordStatusEnum;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.invite.InviteRecordService;
import com.bxm.egg.user.mapper.InviteRecordMapper;
import com.bxm.egg.user.model.dto.BountyDTO;
import com.bxm.egg.user.model.dto.InviteRecordDTO;
import com.bxm.egg.user.model.dto.UserPayPromoteDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.entity.UserRegisterLogEntity;
import com.bxm.egg.user.model.param.InviteExpireTaskParam;
import com.bxm.egg.user.model.param.InviteRecordParam;
import com.bxm.egg.user.model.vo.InviteRecord;
import com.bxm.egg.user.register.UserRegisterService;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author zhaoyadong 2019/3/6 9:25
 * @desc
 */
@Slf4j
@Service
public class InviteRecordServiceImpl implements InviteRecordService {

    @Resource
    private InviteRecordMapper inviteRecordMapper;

    @Resource
    private UserRegisterService userRegisterService;

    @Resource
    private UserInfoService userInfoService;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Override
    public InviteRecord getInviteRecordByInvitedUser(Long inviteUserId, Long invitedUserId) {
        if (inviteUserId == null || invitedUserId == null) {
            return null;
        }

        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setUserId(inviteUserId);
        inviteRecordParam.setInvitedUserId(invitedUserId);
        inviteRecordParam.setStatus(InviteRecordStatusEnum.VALID.getStatus());

        return inviteRecordMapper.getInviteRecordByUser(inviteRecordParam);
    }

    @Override
    public InviteRecord getInviteRecordById(Long inviteUserId, Long id) {
        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setId(id);
        inviteRecordParam.setUserId(inviteUserId);
        return inviteRecordMapper.getInviteRecordByUser(inviteRecordParam);
    }

    @Override
    public Boolean updateInviteRecord(InviteRecord inviteRecord) {
        log.info("更新邀请关系[{}]", JSON.toJSONString(inviteRecord));
        this.inviteRecordMapper.updateInviteRecord(inviteRecord);
        return Boolean.TRUE;
    }

    @Override
    public BountyDTO getUserBountyInfo(Long userId) {
        List<InviteRecord> inviteRecords = getUserAllInviteRecordByCash(userId);

        return generateBounty(inviteRecords);
    }

    @Override
    public UserPayPromoteDTO getUserPayPromote(Long userId) {
        List<InviteRecord> inviteRecords = getUserAllInviteRecordByCash(userId);
        return generatePayPromote(inviteRecords, userId);
    }

    @Override
    public Long addInviteRecord(Long inviteUserId, Long invitedUserId, String type) {
        if (null == inviteUserId || null == invitedUserId) {
            return 0L;
        }
        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setUserId(inviteUserId);
        inviteRecordParam.setInvitedUserId(invitedUserId);
        inviteRecordParam.setStatus(InviteRecordStatusEnum.VALID.getStatus());
        List<InviteRecord> inviteRecordList = inviteRecordMapper.queryUserInviteRecord(inviteRecordParam);

        if (CollectionUtils.isEmpty(inviteRecordList)) {
            Long id = SequenceHolder.nextLongId();
            Date now = new Date();
            Date effectiveTime = DateUtils.addDays(now, 1);
            UserInfoEntity userInfoEntity = userInfoService.selectUserById(invitedUserId);
            //分布式锁保证不会重复插入
            if (distributedLock.lock(invitedUserId.toString(), 1, TimeUnit.SECONDS)) {
                InviteRecord inviteRecord = new InviteRecord();
                inviteRecord.setId(id);
                inviteRecord.setUserId(inviteUserId);
                inviteRecord.setInvitedUserId(invitedUserId);
                inviteRecord.setType(type);
                inviteRecord.setInviteState(InviteRecordStateEnum.ACCEPT_INVITE.name());
                inviteRecord.setEffectTime(effectiveTime);
                inviteRecord.setInvitedUserName(userInfoEntity.getNickname());
                inviteRecord.setInvitedUserImg(userInfoEntity.getHeadImg());
                inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
                inviteRecord.setCreateTime(now);
                inviteRecord.setModifyTime(now);

                inviteRecordMapper.addInviteRecord(inviteRecord);

                InviteExpireTaskParam param = new InviteExpireTaskParam();
                param.setInviteUserId(inviteRecord.getUserId());
                param.setInvitedUserId(inviteRecord.getInvitedUserId());

                return id;
            }
        }

        return 0L;
    }

    @Override
    public Long updateInviteCash(Long inviteUserId, Long invitedUserId, BigDecimal cash) {
        if (null == inviteUserId || null == invitedUserId) {
            return 0L;
        }

        InviteRecordParam param = new InviteRecordParam();
        param.setUserId(inviteUserId);
        param.setInvitedUserId(invitedUserId);
        param.setStatus(InviteRecordStatusEnum.VALID.getStatus());
        param.setAwardType(AwardTypeEnum.CASH.name());
        List<InviteRecord> records = inviteRecordMapper.queryUserInviteRecord(param);

        if (!CollectionUtils.isEmpty(records)) {
            inviteRecordMapper.updateInviteCash(inviteUserId, invitedUserId, cash);
            return records.get(0).getId();
        }

        return 0L;
    }

    @Override
    public Boolean checkIsAssociation(Long userId) {
        return inviteRecordMapper.checkIsAssociation(userId) > 0;
    }

    @Override
    public Boolean insertNormal(Long inviteUserId, Long invitedUserId) {
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(invitedUserId);
        UserRegisterLogEntity userRegisterLogEntity = userRegisterService.selectByUserId(invitedUserId);

        InviteRecord inviteRecord = new InviteRecord();
        inviteRecord.setId(SequenceHolder.nextLongId());
        inviteRecord.setUserId(inviteUserId);
        inviteRecord.setInvitedUserId(invitedUserId);
        inviteRecord.setType(Objects.nonNull(userRegisterLogEntity) ? userRegisterLogEntity.getChannel() : null);
        inviteRecord.setInviteState(InviteRecordStateEnum.ACCEPT_INVITE.name());
        inviteRecord.setEffectTime(DateUtils.addDays(new Date(), 1));
        inviteRecord.setInvitedUserName(userInfoEntity.getNickname());
        inviteRecord.setInvitedUserImg(userInfoEntity.getHeadImg());
        inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
        inviteRecord.setStatus(InviteRecordStatusEnum.VALID.getStatus());
        inviteRecord.setCreateTime(new Date());
        return inviteRecordMapper.addInviteRecord(inviteRecord) > 0;

    }

    /**
     * 获取用户所有现金奖励类型的数据
     */
    private List<InviteRecord> getUserAllInviteRecordByCash(Long userId) {
        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setUserId(userId);
        inviteRecordParam.setAwardType(AwardTypeEnum.CASH.toString());
        inviteRecordParam.setAward(BigDecimal.ZERO);
        return inviteRecordMapper.queryUserInviteRecord(inviteRecordParam);
    }

    /**
     * 生成赏金信息
     */
    private BountyDTO generateBounty(List<InviteRecord> inviteRecords) {
        BountyDTO bountyDTO = new BountyDTO();
        //筛选有效和无效的赏金信息
        if (CollectionUtils.isEmpty(inviteRecords)) {
            return bountyDTO;
        }

        //有效赏金
        List<InviteRecordDTO> validList = filterValid(inviteRecords);
        if (!CollectionUtils.isEmpty(validList)) {
            List<InviteRecordDTO> availableCashList = bountyDTO.getAvailableCashList();
            BigDecimal availableCash = BigDecimal.ZERO;
            for (InviteRecordDTO inviteRecord : validList) {
                if (InviteRecordStateEnum.ACCEPT_INVITE.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.ACCEPT_INVITE.getDesc());
                    availableCashList.add(inviteRecord);
                } else if (InviteRecordStateEnum.LOGIN_APP.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(getInviteStateDesc(inviteRecord.getInviteState(), inviteRecord.getInvitedUserId()));
                    inviteRecord.setEffectTime(null);
                    availableCashList.add(inviteRecord);
                }
                availableCash = availableCash.add(inviteRecord.getAward());
            }
            bountyDTO.setAvailableCash(availableCash);
            bountyDTO.setAvailableCashList(availableCashList);
        }

        //失效赏金
        List<InviteRecordDTO> invalidList = filterInValid(inviteRecords);
        if (!CollectionUtils.isEmpty(invalidList)) {
            List<InviteRecordDTO> discardCashList = bountyDTO.getDiscardCashList();
            BigDecimal discardCash = BigDecimal.ZERO;
            for (InviteRecordDTO inviteRecord : invalidList) {
                if (InviteRecordStateEnum.OVERDUE_INVALID.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.OVERDUE_INVALID.getDesc());
                } else if (InviteRecordStateEnum.NOT_NEW_USER.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.NOT_NEW_USER.getDesc());
                }
                inviteRecord.setEffectTime(null);
                discardCashList.add(inviteRecord);
                discardCash = discardCash.add(inviteRecord.getAward());
            }
            bountyDTO.setDiscardCash(discardCash);
            bountyDTO.setDiscardCashList(discardCashList);
        }

        return bountyDTO;
    }

    /**
     * 筛选在路上的赏金
     */
    private List<InviteRecordDTO> filterValid(List<InviteRecord> inviteRecords) {
        return inviteRecords.stream()
                .map(this::convertInviteRecord)
                .filter(inviteRecord ->
                        InviteRecordStatusEnum.VALID.getStatus().equals(inviteRecord.getStatus()) &&
                                !InviteRecordStateEnum.SUCCESS_WITHDRAW.getName().equals(inviteRecord.getInviteState()))
                .collect(Collectors.toList());
    }

    /**
     * 筛选已废弃的赏金
     */
    private List<InviteRecordDTO> filterInValid(List<InviteRecord> inviteRecords) {
        return inviteRecords.stream()
                .map(this::convertInviteRecord)
                .filter(inviteRecord ->
                        InviteRecordStatusEnum.INVALID.getStatus().equals(inviteRecord.getStatus()))
                .collect(Collectors.toList());
    }

    /**
     * 筛选已获得的赏金
     */
    private List<InviteRecordDTO> filterObtain(List<InviteRecord> inviteRecords) {
        return inviteRecords.stream()
                .map(this::convertInviteRecord)
                .filter(inviteRecord ->
                        InviteRecordStatusEnum.VALID.getStatus().equals(inviteRecord.getStatus()) &&
                                InviteRecordStateEnum.SUCCESS_WITHDRAW.getName().equals(inviteRecord.getInviteState()))
                .collect(Collectors.toList());
    }

    /**
     * 转换对象
     */
    private InviteRecordDTO convertInviteRecord(InviteRecord inviteRecord) {
        InviteRecordDTO inviteRecordDTO = new InviteRecordDTO();
        BeanUtils.copyProperties(inviteRecord, inviteRecordDTO);
        return inviteRecordDTO;
    }

    /**
     * 生成推广赚钱信息
     */
    private UserPayPromoteDTO generatePayPromote(List<InviteRecord> inviteRecords, Long userId) {
        BountyDTO bountyDTO = generateBounty(inviteRecords);
        UserPayPromoteDTO userPayPromoteDTO = new UserPayPromoteDTO();
        userPayPromoteDTO.setAvailableCash(bountyDTO.getAvailableCash());
        userPayPromoteDTO.setAvailableCashList(bountyDTO.getAvailableCashList());
        userPayPromoteDTO.setDiscardCash(bountyDTO.getDiscardCash());
        userPayPromoteDTO.setDiscardCashList(bountyDTO.getDiscardCashList());

        //获取已获得赏金
        List<InviteRecordDTO> obtainList = filterObtain(inviteRecords);
        if (!CollectionUtils.isEmpty(obtainList)) {
            List<InviteRecordDTO> havingCashList = userPayPromoteDTO.getHavingCashList();
            BigDecimal havingCash = BigDecimal.ZERO;
            for (InviteRecordDTO inviteRecord : obtainList) {
                inviteRecord.setInviteState(getInviteStateDesc(inviteRecord.getInviteState(), userId));
                havingCashList.add(inviteRecord);
                havingCash = havingCash.add(inviteRecord.getAward());
            }

            userPayPromoteDTO.setHavingCash(havingCash);
            userPayPromoteDTO.setHavingCashList(havingCashList);
        }

        BigDecimal havingCash = userPayPromoteDTO.getHavingCash();

        //累计赚钱
        userPayPromoteDTO.setTotalGainCash(userPayPromoteDTO.getAvailableCash().add(userPayPromoteDTO.getHavingCash()));
        return userPayPromoteDTO;
    }

    /**
     * 获取邀请状态描述
     */
    private String getInviteStateDesc(String name, Long userId) {
        if (InviteRecordStateEnum.LOGIN_APP.getName().equals(name)) {
            Integer inviteState = redisHashMapAdapter.get(RedisConfig.USER_WITHDRAW_LOGIN_TWICE.copy(), userId.toString(), Integer.class);
            if (null != inviteState && 1 == inviteState) {
                return "需第二天登录";
            } else {
                return InviteRecordStateEnum.LOGIN_APP.getDesc();
            }
        } else {
            return InviteRecordStateEnum.valueOf(name).getDesc();
        }
    }

}
