package com.bxm.egg.user.token.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.jwt.bo.JwtTokenBO;
import com.bxm.component.jwt.builder.TokenBuildParam;
import com.bxm.component.jwt.util.JwtUtil;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.facade.bo.UserTokenBO;
import com.bxm.egg.user.model.dto.token.TokenInfoDTO;
import com.bxm.egg.user.model.param.token.RenewTokenParam;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.egg.user.token.TokenService;
import com.bxm.newidea.component.param.BasicParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.utils.SystemClock;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;

import static com.bxm.egg.user.constant.UserConstant.USER_ID_KEY;

/**
 * TOKEN刷新实现类
 *
 * @author wzy
 * @date 2021年09月14日15:38:24
 **/
@Service
@Slf4j
@AllArgsConstructor
public class TokenServiceImpl implements TokenService {

    private RedisStringAdapter redisStringAdapter;

    private UserProperties userProperties;

    @Override
    public TokenInfoDTO renew(RenewTokenParam param) {
        Preconditions.checkNotNull(param.getUserId());
        Preconditions.checkNotNull(param.getRefreshToken());
        String requestRefreshToken = param.getRefreshToken();

        // 获取用户对应的refreshToken
        String cacheRefreshToken = getCacheRefreshToken(param.getUserId());

        if (log.isDebugEnabled()) {
            log.debug("重新请求accessToken，参数：{}", JSON.toJSONString(param));
        }
        if (null == cacheRefreshToken) {
            log.info("[{}] 缓存中不存在对应的RefreshToken", param.getUserId());
            return null;
        }

        if (!requestRefreshToken.equals(cacheRefreshToken)) {
            log.warn("请求提供的RefreshToken与缓存不一致，uid：{},except：{},actual：{}",
                    param.getUserId(),
                    cacheRefreshToken,
                    param.getRefreshToken());
            return null;
        }

        // 判断refreshToken，创建一个新的RefreshToken
        String secretKey = userProperties.getAppTokenSecret();
        JwtTokenBO jwtTokenBO = JwtUtil.parseToken(requestRefreshToken, secretKey);

        if (jwtTokenBO.getExpirationDate().getTime() < SystemClock.now()) {
            log.warn("[{}]refreshToken已过期", param.getUserId());
            return null;
        }

        return create(param.getUserId(), param);
    }

    private String getCacheRefreshToken(Long userId) {
        KeyGenerator key = RedisConfig.USER_REFRESH_TOKEN_CACHE_KEY.copy().appendKey(userId);
        return redisStringAdapter.get(key, String.class);
    }

    @Override
    public TokenInfoDTO create(Long userId, BasicParam param) {
        String secretKey = userProperties.getAppTokenSecret();

        TokenInfoDTO tokenInfoDTO = TokenInfoDTO.builder()
                .refreshToken(createRefreshToken(userId, secretKey))
                .accessToken(doCreateAccessToken(userId, secretKey))
                .build();

        if (log.isDebugEnabled()) {
            log.debug("创建用户Token信息：{}", tokenInfoDTO);
        }

        return tokenInfoDTO;
    }

    /**
     * 创建访问的token，默认30分钟过期
     *
     * @param userId 用户ID
     * @param key    系统对应的加密密钥
     * @return 访问token，短期的凭证
     */
    private String doCreateAccessToken(Long userId, String key) {
        Date accessTokenExpiredDate = DateUtils.addField(new Date(), Calendar.SECOND,
                userProperties.getAccessTokenExpiredSeconds());

        return JwtUtil.generateToken(TokenBuildParam.create()
                .putClaim(USER_ID_KEY, userId.toString())
                .setIssueDate(new Date())
                .setExpirationTime(accessTokenExpiredDate)
                .setSignatureKey(key));
    }

    @Override
    public UserTokenBO renewToken(Long userId) {
        RenewTokenParam param = new RenewTokenParam();

        String cacheRefreshToken = getCacheRefreshToken(userId);
        param.setUserId(userId);
        param.setRefreshToken(cacheRefreshToken);

        TokenInfoDTO tokenInfoDTO = renew(param);

        if (tokenInfoDTO == null) {
            return null;
        }

        UserTokenBO userTokenBO = new UserTokenBO();

        userTokenBO.setReNewAccessToken(tokenInfoDTO.getAccessToken());
        userTokenBO.setReNewRefreshToken(tokenInfoDTO.getRefreshToken());

        return userTokenBO;
    }

    /**
     * 创建用于刷新accessToken的RefreshToken，默认提供一个较长的过期时间
     *
     * @param userId 用户ID
     * @param key    用户对应的应用的加密密钥
     * @return 刷新token
     */
    private String createRefreshToken(Long userId, String key) {
        Date refreshTokenExpiredDate = DateUtils.addField(new Date(), Calendar.SECOND,
                userProperties.getRefreshTokenExpiredSeconds());

        String refreshToken = JwtUtil.generateToken(TokenBuildParam.create()
                .putClaim(USER_ID_KEY, userId.toString())
                .setIssueDate(new Date())
                .setExpirationTime(refreshTokenExpiredDate)
                .setSignatureKey(key));

        // RefreshToken需要保存到缓存，后续用于校验
        KeyGenerator cacheKey = RedisConfig.USER_REFRESH_TOKEN_CACHE_KEY.copy().appendKey(userId);
        redisStringAdapter.set(cacheKey, refreshToken, userProperties.getRefreshTokenExpiredSeconds());

        return refreshToken;
    }
}
