package com.bxm.egg.user.warmlevel.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.dto.UserWarmValueUpDTO;
import com.bxm.egg.user.enums.WarmRuleEnum;
import com.bxm.egg.user.enums.WarmValueEquityEnum;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserStatisticsService;
import com.bxm.egg.user.info.msg.UserInfoChangeSender;
import com.bxm.egg.user.integration.DomainIntegrationService;
import com.bxm.egg.user.integration.MessageFacadeIntegrationService;
import com.bxm.egg.user.mapper.warmlevel.*;
import com.bxm.egg.user.model.UserEquityDTO;
import com.bxm.egg.user.model.dto.warmlevel.*;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.entity.UserStatisticsEntity;
import com.bxm.egg.user.model.param.warmlevel.UserWarmValueDetailParam;
import com.bxm.egg.user.model.vo.warmlevel.*;
import com.bxm.egg.user.param.UserWarmActionParam;
import com.bxm.egg.user.properties.H5JumpAddressProperties;
import com.bxm.egg.user.properties.UserWarmLevelProperties;
import com.bxm.egg.user.warmlevel.UserWarmLevelService;
import com.bxm.egg.user.warmlevel.impl.context.WarmActionContext;
import com.bxm.newidea.component.redis.*;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.egg.user.constant.RedisConfig.*;

/**
 * @author lowi
 * @date 2021/3/2 10:16
 */
@Service
@AllArgsConstructor
@Slf4j
public class UserWarmLevelServiceImpl implements UserWarmLevelService {

    private final UserWarmLevelProperties userWarmLevelProperties;

    private final H5JumpAddressProperties h5JumpAddressProperties;

    private final WarmLevelMapper warmLevelMapper;

    private final WarmEquityDetailMapper warmEquityDetailMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final RedisSetAdapter redisSetAdapter;

    private final RedisListAdapter redisListAdapter;

    private final WarmHandlerProxy warmHandlerProxy;

    private final WarmRuleDetailMapper warmRuleDetailMapper;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserLevelLogMapper userLevelLogMapper;

    private final SequenceCreater sequenceCreate;

    private final UserWarmValueFlowMapper userWarmValueFlowMapper;

    private final MessageFacadeIntegrationService messageSender;

    private final DomainIntegrationService domainIntegrationService;

    private final UserInfoService userInfoService;

    private final UserStatisticsService userStatisticsService;

    private final UserInfoChangeSender userInfoChangeSender;

    @Override
    public UserEquityDTO getLevelEquityInfo(Integer warmValue) {
        WarmLevel warmLevel = warmLevelMapper.getWarmLevelByWarmValue(warmValue);
        if (Objects.isNull(warmLevel)) {
            return new UserEquityDTO();
        }
        List<WarmEquityDetail> warmEquityDetails = warmEquityDetailMapper.listWarmEquityByLevel(warmLevel.getLevel());
        UserEquityDTO userEquityDTO = new UserEquityDTO();
        userEquityDTO.setShowLevel(Boolean.TRUE);
        userEquityDTO.setLevel(warmLevel.getLevel());
        userEquityDTO.setLevelUrl(warmLevel.getLevelUrl());
        if (warmEquityDetails.isEmpty()) {
            userEquityDTO.setEquityType(new String[]{});
            userEquityDTO.setDazzling(Boolean.FALSE);
        } else {
            String[] equityInfo = new String[warmEquityDetails.size()];
            List<String> collect = warmEquityDetails.stream().map(WarmEquityDetail::getType).collect(Collectors.toList());
            equityInfo = collect.toArray(equityInfo);
            userEquityDTO.setEquityType(equityInfo);
            userEquityDTO.setDazzling(collect.contains(WarmValueEquityEnum.DAZZLING_ID.name()));
        }
        return userEquityDTO;
    }

    @Override
    public List<UserWarmValueRuleDetailDTO> getWarmValueAddInfo() {
        String addRuleDetail = userWarmLevelProperties.getAddRuleDetail();
        if (StringUtils.isNotBlank(addRuleDetail)) {
            try {
                return JSONObject.parseArray(addRuleDetail, UserWarmValueRuleDetailDTO.class);
            } catch (Exception e) {
                log.error("apollo配置获取温暖值增加规则转换错误 addRuleDetail：{}", addRuleDetail, e);
                return Lists.newArrayList();
            }
        }
        return Lists.newArrayList();
    }

    @Override
    public List<UserWarmValueRuleDetailDTO> getWarmValueDeductionInfo() {
        String decRuleDetail = userWarmLevelProperties.getDecRuleDetail();
        if (StringUtils.isNotBlank(decRuleDetail)) {
            try {
                return JSONObject.parseArray(decRuleDetail, UserWarmValueRuleDetailDTO.class);
            } catch (Exception e) {
                log.error("apollo配置获取温暖值扣除规则转换错误 addRuleDetail：{}", decRuleDetail, e);
                return Lists.newArrayList();
            }
        }
        return Lists.newArrayList();
    }

    @Override
    public UserWarmLevelInfoDTO getUserWarmValueInfo(Long userId) {
        Boolean exists = redisSetAdapter.exists(USER_WARM_FIRST_INTO, userId);
        UserWarmLevelInfoDTO warmLevelInfoDTO = getUserWarmInfoCache(userId);
        if (exists) {
            //已经展示过了,不再展示tips
            warmLevelInfoDTO.setShowTips(Boolean.FALSE);
        } else {
            redisSetAdapter.add(USER_WARM_FIRST_INTO, userId);
            warmLevelInfoDTO.setShowTips(Boolean.TRUE);
        }
        return warmLevelInfoDTO;
    }

    @Override
    public UserWarmLevelInfoDTO getUserWarmInfoCache(Long userId) {
        UserWarmLevelInfoDTO warmLevelInfoDTO = this.redisStringAdapter.get(buildUserCacheKey(userId), UserWarmLevelInfoDTO.class);
        if (Objects.isNull(warmLevelInfoDTO)) {
            warmLevelInfoDTO = loadUserWarmInfoToRedis(userId);
        }
        return warmLevelInfoDTO;
    }

    /**
     * 把用户温暖值信息放入redis
     *
     * @param userId 用户id
     * @return 用户温暖值信息
     */
    private UserWarmLevelInfoDTO loadUserWarmInfoToRedis(Long userId) {
        UserInfoEntity user = userInfoService.selectAllUserById(userId);
        UserStatisticsEntity userStatisticsEntity = userStatisticsService.selectUserStatisticsByUserId(userId);

        if (userStatisticsEntity == null) {
            return buildDefaultInvalidWarmLevelInfo(userId, user);
        }
        UserEquityDTO levelEquityInfo = this.getLevelEquityInfo(userStatisticsEntity.getWarmValue());
        UserWarmLevelInfoDTO userWarmLevelInfoDTO = new UserWarmLevelInfoDTO();
        userWarmLevelInfoDTO.setHeadImg(user.getHeadImg());
        userWarmLevelInfoDTO.setLevel(levelEquityInfo.getLevel());
        userWarmLevelInfoDTO.setLevelUrl(levelEquityInfo.getLevelUrl());
        userWarmLevelInfoDTO.setNickname(user.getNickname());
        userWarmLevelInfoDTO.setUserId(userId);
        userWarmLevelInfoDTO.setWarmValue(userStatisticsEntity.getWarmValue());
        userWarmLevelInfoDTO.setEquityDTO(levelEquityInfo);
        this.redisStringAdapter.set(buildUserCacheKey(userId), userWarmLevelInfoDTO, 3600);
        return userWarmLevelInfoDTO;
    }

    private UserWarmLevelInfoDTO buildDefaultInvalidWarmLevelInfo(Long userId, UserInfoEntity user) {
        UserWarmLevelInfoDTO invalidUserWarmLevelInfoDTO = new UserWarmLevelInfoDTO();
        invalidUserWarmLevelInfoDTO.setNickname(user.getNickname());
        invalidUserWarmLevelInfoDTO.setUserId(userId);
        invalidUserWarmLevelInfoDTO.setWarmValue(0);
        invalidUserWarmLevelInfoDTO.setHeadImg(user.getHeadImg());

        UserEquityDTO levelEquityInfo = getLevelEquityInfo(0);
        invalidUserWarmLevelInfoDTO.setLevel(levelEquityInfo.getLevel());

        invalidUserWarmLevelInfoDTO.setLevelUrl(levelEquityInfo.getLevelUrl());
        invalidUserWarmLevelInfoDTO.setShowTips(false);

        invalidUserWarmLevelInfoDTO.setEquityDTO(getLevelEquityInfo(0));

        return invalidUserWarmLevelInfoDTO;

    }

    private KeyGenerator buildUserCacheKey(Long userId) {
        return USER_WARM_INFO.copy().appendKey(userId);
    }

    @Override
    public List<UserLevelListDTO> getUserLevelList(Long userId) {
        //获取用户温暖值等级信息
        UserWarmLevelInfoDTO userWarmValueInfo = this.getUserWarmValueInfo(userId);
        //获取等级列表信息
        List<WarmLevelDTO> warmLevels = this.getWarmLevelList();

        List<UserLevelLog> userLevelList = userLevelLogMapper.getUserLevelList(userId);
        Map<Integer, UserLevelLog> levelLogMap = new HashMap<>();
        userLevelList.forEach(userLevelLog -> levelLogMap.put(userLevelLog.getLevel(), userLevelLog));
        //获取权益信息
        List<LevelEquityDTO> levelEquityList = this.getLevelEquityList();
        List<UserLevelListDTO> userLevelListDTOS = warmLevels.stream().map(warmLevelDTO -> this.covertUserLevelInfo(userWarmValueInfo.getWarmValue(),
                warmLevelDTO, levelEquityList, levelLogMap)).collect(Collectors.toList());
        return userLevelListDTOS.stream().sorted(Comparator.comparing(UserLevelListDTO::getLevel)).collect(Collectors.toList());
    }

    private UserLevelListDTO covertUserLevelInfo(Integer warmValue, WarmLevelDTO warmLevel, List<LevelEquityDTO> levelEquityList, Map<Integer, UserLevelLog> levelLogMap) {
        UserLevelListDTO userLevelListDTO = new UserLevelListDTO();
        userLevelListDTO.setCurrWarmValue(warmValue);
        userLevelListDTO.setLevel(warmLevel.getLevel());
        int equityNum = (int) levelEquityList.stream().filter(levelEquityDTO -> levelEquityDTO.getMinLevel() <= warmLevel.getLevel()).count();
        //已经是最大级别
        if (Objects.isNull(warmLevel.getMax())) {
            if (warmValue >= warmLevel.getMin()) {
                userLevelListDTO.setLevelStatus(0);
                UserLevelLog userLevelLog = levelLogMap.get(warmLevel.getLevel());
                userLevelListDTO.setFinishDate(Objects.isNull(userLevelLog) ? new Date() : userLevelLog.getCreateTime());
                userLevelListDTO.setEquityNum(equityNum);
            } else {
                userLevelListDTO.setLevelStatus(2);
                userLevelListDTO.setEquityNum(equityNum);
                userLevelListDTO.setNextWarmValue(warmLevel.getMin());
                userLevelListDTO.setNextLevel(warmLevel.getLevel());
            }
            return userLevelListDTO;
        } else {
            //当前等级
            if (warmValue >= warmLevel.getMin() && warmValue < warmLevel.getMax()) {
                userLevelListDTO.setLevelStatus(0);
                userLevelListDTO.setEquityNum(equityNum);
                //下一级的温暖值区间
                userLevelListDTO.setNextWarmValue(warmLevel.getMax());
                //下一级的等级
                userLevelListDTO.setNextLevel((warmLevel.getLevel() + 1));
            }
            //未达到
            if (warmValue < warmLevel.getMin()) {
                userLevelListDTO.setLevelStatus(2);
                userLevelListDTO.setEquityNum(equityNum);
                userLevelListDTO.setNextWarmValue(warmLevel.getMin());
                userLevelListDTO.setNextLevel(warmLevel.getLevel());
            }
            //已达到
            if (warmValue > warmLevel.getMax()) {
                userLevelListDTO.setLevelStatus(1);
                userLevelListDTO.setEquityNum(equityNum);
                userLevelListDTO.setNextWarmValue(warmLevel.getMin());
                userLevelListDTO.setNextLevel(warmLevel.getLevel());
                UserLevelLog userLevelLog = levelLogMap.get(warmLevel.getLevel());
                userLevelListDTO.setFinishDate(Objects.isNull(userLevelLog) ? new Date() : userLevelLog.getCreateTime());
            }
            return userLevelListDTO;
        }
    }


    private List<WarmLevelDTO> getWarmLevelList() {
        KeyGenerator keyGenerator = USER_WARM_LEVEL_LIST.copy().appendKey("list");
        List<WarmLevelDTO> warmLevelDTOList = redisListAdapter.leftIndex(keyGenerator, -1, new TypeReference<WarmLevelDTO>() {
        });
        if (CollectionUtils.isEmpty(warmLevelDTOList)) {
            List<WarmLevel> warmLevelList = warmLevelMapper.getWarmLevelList();
            warmLevelDTOList = warmLevelList.stream().map(warmLevel -> {
                WarmLevelDTO warmLevelDTO = WarmLevelDTO.builder().build();
                warmLevelDTO.setLevel(warmLevel.getLevel());
                warmLevelDTO.setLevelUrl(warmLevel.getLevelUrl());
                warmLevelDTO.setMax(warmLevel.getMaxValue());
                warmLevelDTO.setMin(warmLevel.getMinValue());
                return warmLevelDTO;
            }).collect(Collectors.toList());
            redisListAdapter.leftPush(keyGenerator, warmLevelDTOList);
            return warmLevelDTOList;
        }
        return warmLevelDTOList;
    }

    @Override
    public List<LevelEquityDTO> getLevelEquityList() {
        KeyGenerator keyGenerator = USER_WARM_EQUITY_LIST.copy().appendKey("list");
        List<LevelEquityDTO> levelEquityDTOS = redisListAdapter.leftIndex(keyGenerator, -1, new TypeReference<LevelEquityDTO>() {
        });
        if (CollectionUtils.isEmpty(levelEquityDTOS)) {
            List<WarmEquityDetail> levelEquityList = warmEquityDetailMapper.getLevelEquityList();
            levelEquityDTOS = levelEquityList.stream().map(equityDetail -> {
                LevelEquityDTO levelEquityDTO = LevelEquityDTO.builder().build();
                levelEquityDTO.setContent(equityDetail.getContent());
                levelEquityDTO.setLockImg(equityDetail.getLockUrl());
                levelEquityDTO.setMinLevel(equityDetail.getMinLevel());
                levelEquityDTO.setUnlockImg(equityDetail.getUnlockUrl());
                levelEquityDTO.setEquityImg(equityDetail.getEquityUrl());
                levelEquityDTO.setType(equityDetail.getType());
                levelEquityDTO.setTitle(equityDetail.getTitle());
                return levelEquityDTO;
            }).collect(Collectors.toList());
            redisListAdapter.leftPush(keyGenerator, levelEquityDTOS);
            return levelEquityDTOS.stream().sorted(Comparator.comparing(LevelEquityDTO::getMinLevel)).collect(Collectors.toList());
        }
        return levelEquityDTOS.stream().sorted(Comparator.comparing(LevelEquityDTO::getMinLevel)).collect(Collectors.toList());
    }

    @Override
    public Boolean mockWarmAddOrDec(Long userId, Integer warmValue, WarmRuleEnum warmRuleEnum) {
        UserWarmActionParam warmActionParam = UserWarmActionParam.builder().build();
        warmActionParam.setWarmRuleEnum(Objects.isNull(warmRuleEnum) ? WarmRuleEnum.DEFAULT : warmRuleEnum);
        warmActionParam.setUserId(userId);
        warmActionParam.setWarmValue(warmValue);
        warmHandlerProxy.handle(warmActionParam);
        return true;
    }


    /**
     * 获取温暖值规则
     *
     * @param type 规则类型
     * @return 温暖值规则
     */
    @Override
    public WarmRuleDetail getWarmRule(String type) {
        WarmRuleDetail warmRuleDetail = redisHashMapAdapter.get(RedisConfig.WARM_RULE_DATA, type, WarmRuleDetail.class);
        if (Objects.isNull(warmRuleDetail)) {
            Map<String, WarmRuleDetail> stringWarmRuleDetailMap = loadWarmDataToRedis();
            return stringWarmRuleDetailMap.get(type);
        }
        return warmRuleDetail;
    }

    /**
     * 初始化 温暖值规则到redis
     *
     * @return map
     */
    private Map<String, WarmRuleDetail> loadWarmDataToRedis() {
        List<WarmRuleDetail> warmRuleList = warmRuleDetailMapper.getWarmRuleList();
        Map<String, WarmRuleDetail> map = new HashMap((int) (warmRuleList.size() / 0.75f + 1));
        warmRuleList.forEach(warmRuleDetail -> map.put(warmRuleDetail.getType(), warmRuleDetail));
        redisHashMapAdapter.putAll(RedisConfig.WARM_RULE_DATA, map);
        return map;
    }

    @Override
    public void delWarmCacheWarmLevelUp(WarmActionContext context) {
        //删除缓存
        redisStringAdapter.remove(buildUserCacheKey(context.getUserId()));
        //库里的数量
        Integer userLevelCount = userLevelLogMapper.getUserLevelCount(context.getUserId());
        WarmLevel warmLevel = warmLevelMapper.getWarmLevelByUserId(context.getUserId());
        //级别相同则不处理
        if (userLevelCount >= warmLevel.getLevel()) {
            return;
        }
        //等级变化清空用户缓存信息
        userInfoChangeSender.sendUserChangeMsg(context.getUserId());
        //补充升级信息日志
        List<UserLevelLog> userLevelLogs = new ArrayList<>();
        int logLevel = 1;
        //判断是否是需要补充，或者是降级了
        if (userLevelCount > 0) {
            //比如升级到了11，库里有10条记录 则10+1 =11 ,从11开始插入数据库
            logLevel = userLevelCount + 1;
        }
        for (int i = logLevel; i <= warmLevel.getLevel(); i++) {
            UserLevelLog userLevelLog = new UserLevelLog();
            userLevelLog.setCreateTime(new Date());
            userLevelLog.setId(sequenceCreate.nextLongId());
            userLevelLog.setLevel(i);
            userLevelLog.setUserId(context.getUserId());
            userLevelLogs.add(userLevelLog);
        }
        //降级了列表数据为空不需要插入了
        if (!userLevelLogs.isEmpty()) {
            userLevelLogMapper.batchInsertLog(userLevelLogs);
        }

        Boolean exists = redisSetAdapter.exists(USER_WARM_LEVEL_PUSH.copy().appendKey(warmLevel.getLevel()), context.getUserId());
        //redis没有当前等级缓存，且当前等级大于之前等级才弹窗，降级不用弹窗
        boolean popResult = (Objects.isNull(exists) || !exists) && warmLevel.getLevel() > userLevelCount;
        if (popResult) {
            //弹窗！！！！！！
            addUserLevelPop(context.getUserId(), warmLevel.getLevel(), warmLevel.getLevelPopUrl());
            redisSetAdapter.add(USER_WARM_LEVEL_PUSH.copy().appendKey(warmLevel.getLevel()), context.getUserId());
        }
    }

    /**
     * 增加用户等级弹窗数据
     *
     * @param userId   用户id
     * @param level    等级
     * @param levelUrl 等级图片
     */
    private void addUserLevelPop(Long userId, Integer level, String levelUrl) {
        redisListAdapter.remove(USER_LEVLE_UP_POP.copy().appendKey(userId));
        UserWarmValueUpDTO userWarmValueUpDTO = new UserWarmValueUpDTO();
        userWarmValueUpDTO.setLevel(level);
        userWarmValueUpDTO.setLevelUrl(levelUrl);
        String innerH5BaseUrl = domainIntegrationService.getInnerH5BaseUrl();
        userWarmValueUpDTO.setJumpUrl(innerH5BaseUrl + h5JumpAddressProperties.getLevelJumpUrl());
        redisListAdapter.leftPush(USER_LEVLE_UP_POP.copy().appendKey(userId), userWarmValueUpDTO);
    }

    @Override
    public List<UserWarmValueDetailDTO> getWarmDetailListByUser(UserWarmValueDetailParam userWarmValueDetailParam) {

        IPage<UserWarmValueFlow> page = new Page<>(userWarmValueDetailParam.getPageNum(), userWarmValueDetailParam.getPageSize());
        IPage<UserWarmValueFlow> warmDetailListByUser = userWarmValueFlowMapper.getWarmDetailListByUser(page,
                userWarmValueDetailParam);

        return warmDetailListByUser.getRecords().stream().map(this::covertWarmDetail).collect(Collectors.toList());
    }

    private UserWarmValueDetailDTO covertWarmDetail(UserWarmValueFlow userWarmValueFlow) {
        UserWarmValueDetailDTO userWarmValueDetailDTO = UserWarmValueDetailDTO.builder().build();
        userWarmValueDetailDTO.setCreateTime(userWarmValueFlow.getCreateTime());
        userWarmValueDetailDTO.setDetailStr(userWarmValueFlow.getTypeDesc());
        userWarmValueDetailDTO.setUserId(userWarmValueFlow.getUserId());
        userWarmValueDetailDTO.setWarmValue(userWarmValueFlow.getWarmValue());
        return userWarmValueDetailDTO;
    }

    @Override
    public UserWarmValueUpDTO getUserLevelPopData(Long userId) {
        return redisListAdapter.leftPop(USER_LEVLE_UP_POP.copy().appendKey(userId), UserWarmValueUpDTO.class);
    }

    @Override
    public EquityNotAbleInfo getNotAvailablePopupExplain(Long userId, String type) {
        List<LevelEquityDTO> levelEquityList = this.getLevelEquityList();
        List<LevelEquityDTO> levelEquityDTOS = levelEquityList.stream().filter(levelEquityDTO -> levelEquityDTO.getType().equals(type)).collect(Collectors.toList());
        if (!levelEquityDTOS.isEmpty()) {
            EquityNotAbleInfo equityNotAbleInfo = new EquityNotAbleInfo();
            equityNotAbleInfo.setLevel(levelEquityDTOS.get(0).getMinLevel());
            equityNotAbleInfo.setEquityDesc(levelEquityDTOS.get(0).getTitle());
            return equityNotAbleInfo;
        }
        return null;
    }

    @Override
    public void updateUserWarmInfo(Long userId, Integer warmValue) {
        //获取等级列表信息
        List<WarmLevelDTO> warmLevels = this.getWarmLevelList();
        //获取1级的最小温暖值作为默认温暖值
        List<WarmLevelDTO> warmLevel1Info = warmLevels.stream().filter(warmLevelDTO -> warmLevelDTO.getLevel().equals(1)).collect(Collectors.toList());
        userStatisticsService.updateUserWarmInfo(userId, warmValue, warmLevel1Info.get(0).getMin());
    }
}
