package com.bxm.egg.user.warmlevel.impl;

import com.bxm.egg.user.enums.WarmRuleEnum;
import com.bxm.egg.user.exception.UserAccountException;
import com.bxm.egg.user.param.UserWarmActionParam;
import com.bxm.egg.user.warmlevel.impl.callback.DefaultWarmActionCallback;
import com.bxm.egg.user.warmlevel.impl.callback.WarmActionCallback;
import com.bxm.egg.user.warmlevel.impl.context.WarmActionContext;
import com.bxm.egg.user.warmlevel.impl.handler.DefaultWarmActionHandler;
import com.bxm.egg.user.warmlevel.impl.handler.WarmActionHandler;
import com.bxm.newidea.component.bo.Message;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.retry.RetryException;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 账号处理代理类 封装原始请求，匹配对应的处理器 如非重大改版，此类不需要进行额外的处理.扩展流程如下：
 * <ol>
 * <li>定义温暖值处理动作 {@link WarmRuleEnum}
 * <li>定义处理动作对应的Handler {@link WarmActionHandler}
 * <li>给Handler绑定对应的处理规则 {@link com.bxm.egg.user.warmlevel.impl.rule.WarmRule}
 * <li>如存在流程之外的逻辑，提供callback {@link WarmActionCallback}
 * </ol>
 *
 * @author lowi
 * @date 2021/03/03
 */
@Component
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class WarmHandlerProxy {

    private Map<WarmRuleEnum, WarmActionHandler> handlerMap;

    private WarmActionHandler defaultHandler = new DefaultWarmActionHandler();


    @Autowired
    public WarmHandlerProxy(List<WarmActionHandler> handlerList) {
        handlerMap = Maps.newHashMap();

        if (!CollectionUtils.isEmpty(handlerList)) {
            handlerList.forEach(handler -> handlerMap.put(handler.support(), handler));
        }
    }

    @Retryable(value = RetryException.class)
    public Message handle(UserWarmActionParam param) {
        return handler(param, null);
    }

    @Retryable(value = RetryException.class)
    public Message handler(UserWarmActionParam param, WarmActionCallback callback) {
        log.debug("进行账号温暖值操作，请求参数：{}", param);
        if (Objects.isNull(param.getUserId())) {
            return Message.build(false).setMessage("用户id不能为空");
        }
        //封装上下文
        WarmActionContext context = new WarmActionContext(param);
        if (null == callback) {
            callback = new DefaultWarmActionCallback();
        }
        context.setCallback(callback);
        try {
            //查找匹配的处理类
            WarmActionHandler handler = handlerMap.getOrDefault(context.getAction(), defaultHandler);
            log.debug("账号操作的Handler：{}", handler.getClass().getSimpleName());

            return handler.handle(context);
        } catch (UserAccountException e) {
            log.warn("处理温暖值发生业务异常，请求参数：{}，异常信息：{}:是否重试：{}",
                    context,
                    e.getMessage(),
                    e.isRetryable());

            //如果可以重试则抛出异常，触发重试
            if (e.isRetryable()) {
                callback.preRetry(context);
                throw new RetryException(e.getMessage(), e);
            } else {
                callback.exception(context);
                throw e;
            }
        } catch (Exception e) {
            callback.exception(context);
            log.error("温暖值处理发生未预期的错误,请求参数：{}", context);
            log.error(e.getMessage(), e);

            throw e;
        }
    }
}
