package com.bxm.egg.user.block.impl;

import com.bxm.egg.common.enums.AuthCodeEnum;
import com.bxm.egg.user.auth.UserAuthCodeService;
import com.bxm.egg.user.auth.UserBlockExpiredTaskService;
import com.bxm.egg.user.block.AdminBlockUserService;
import com.bxm.egg.user.constant.AdminBizConfigProperties;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.enums.BlockUserTypeEnum;
import com.bxm.egg.user.integration.IMIntegrationService;
import com.bxm.egg.user.integration.PushMsgIntegrationService;
import com.bxm.egg.user.mapper.UserBlockMapper;
import com.bxm.egg.user.model.dto.block.UserBlockDTO;
import com.bxm.egg.user.model.entity.UserBlock;
import com.bxm.egg.user.param.UserAuthCodeParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Service
public class AdminBlockUserServiceImpl implements AdminBlockUserService {


    @Autowired
    private UserBlockMapper userBlockMapper;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private IMIntegrationService imIntegrationService;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Resource
    private UserAuthCodeService userAuthCodeService;

    @Autowired
    private AdminBizConfigProperties adminBizConfigProperties;

    @Autowired
    private PushMsgIntegrationService pushMsgIntegrationService;

    @Autowired
    private UserBlockExpiredTaskService userBlockExpiredTaskService;

    @Override
    public UserBlockDTO getInfo(Long userId) {
        List<UserBlock> userBlockList = userBlockMapper.selectByUserId(userId);
        if (CollectionUtils.isEmpty(userBlockList)) {
            return null;
        }
        return convert(userId, userBlockList);
    }

    @Override
    public Boolean editUserBlock(UserBlockDTO userBlockDTO) {
        Long userId = userBlockDTO.getUserId();
        Date blockTime = userBlockDTO.getBlockTime();
        List<Integer> index = Lists.newArrayList();
        // 禁止发帖、禁止发评论、禁止私聊、拉黑
        String actions = "";
        //禁止私聊
        if (userBlockDTO.getChatLimitFlag()) {
            limit(userBlockDTO, BlockUserTypeEnum.CHAT_LIMIT, getChatLimitRedisKey(userId));
            imIntegrationService.block(userId, blockTime);
            actions += "禁止私聊、";
        } else {
            cancelLimit(userBlockDTO, BlockUserTypeEnum.CHAT_LIMIT, getChatLimitRedisKey(userId));
            imIntegrationService.unblock(userId);
        }

        //拉黑
        if (userBlockDTO.getBlockFlag()) {
            limit(userBlockDTO, BlockUserTypeEnum.BLOCK_LIMIT, getBlockLimitRedisKey(userId));
            actions += "拉黑、";
        } else {
            cancelLimit(userBlockDTO, BlockUserTypeEnum.BLOCK_LIMIT, getBlockLimitRedisKey(userId));
        }

        //禁止评论
        if (userBlockDTO.getForbidComment()) {
            actions += "禁止评论、";
            this.addForbidUserAuth(AuthCodeEnum.FORBID_COMMENT, userBlockDTO.getUserId());
            limit(userBlockDTO, BlockUserTypeEnum.FORBID_COMMENT, getForbidCommentLimitRedisKey(userId));
            index.add(AuthCodeEnum.FORBID_COMMENT.index);
        } else {
            this.removeForbidUserAuth(AuthCodeEnum.FORBID_COMMENT, userBlockDTO.getUserId());
            this.cancelLimit(userBlockDTO, BlockUserTypeEnum.FORBID_COMMENT, getForbidCommentLimitRedisKey(userId));
        }

        //禁止发帖
        if (userBlockDTO.getForbidPost()) {
            actions += "禁止发帖、";
            this.addForbidUserAuth(AuthCodeEnum.FORBID_POST, userBlockDTO.getUserId());
            limit(userBlockDTO, BlockUserTypeEnum.FORBID_POST, getForbidPostLimitRedisKey(userId));
            index.add(AuthCodeEnum.FORBID_POST.index);
        } else {
            this.removeForbidUserAuth(AuthCodeEnum.FORBID_POST, userBlockDTO.getUserId());
            this.cancelLimit(userBlockDTO, BlockUserTypeEnum.FORBID_POST, getForbidPostLimitRedisKey(userId));
        }

        //删除移除权限定时任务
        userBlockExpiredTaskService.removeTask(userBlockDTO.getUserId());
        if (CollectionUtils.isNotEmpty(index)) {
            userBlockExpiredTaskService.addTask(userBlockDTO.getUserId(),
                    index,
                    userBlockDTO.getBlockTime());
        }

        // 发推送
        if (isNotBlank(actions)) {
            if (actions.lastIndexOf(",") == actions.length() - 1) {
                actions = actions.substring(0, actions.length() - 1);
            }
            pushMsgIntegrationService.pushBlockMsg(actions, userBlockDTO.getBlockTime(), userBlockDTO.getUserId(),
                    adminBizConfigProperties.getOfficialRulePostId());
        }

        return Boolean.TRUE;
    }


    private void addForbidUserAuth(AuthCodeEnum ennum, Long userId) {
        UserAuthCodeParam param = new UserAuthCodeParam();
        param.setAuthCode(ennum.index);
        param.setUserId(userId);

        AuthCodeEnum authCode = AuthCodeEnum.getByIndex(param.getAuthCode());
        userAuthCodeService.addAuthCode(param.getUserId(), authCode);
    }

    private void removeForbidUserAuth(AuthCodeEnum ennum, Long userId) {
        UserAuthCodeParam param = new UserAuthCodeParam();
        param.setAuthCode(ennum.index);
        param.setUserId(userId);

        AuthCodeEnum authCode = AuthCodeEnum.getByIndex(param.getAuthCode());
        userAuthCodeService.removeAuthCode(param.getUserId(), authCode);
    }


    private UserBlockDTO convert(Long userId, List<UserBlock> userBlockList) {
        UserBlockDTO userBlockDTO = UserBlockDTO.builder()
                .userId(userId)
                .blockTime(userBlockList.get(0).getBlockTime())
                .build();
        userBlockList.forEach(userBlock -> {
            if (BlockUserTypeEnum.CHAT_LIMIT.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getChatLimitRedisKey(userId)))) {
                userBlockDTO.setChatLimitFlag(true);
            }
            if (BlockUserTypeEnum.BLOCK_LIMIT.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getBlockLimitRedisKey(userId)))) {
                userBlockDTO.setBlockFlag(true);
            }
            if (BlockUserTypeEnum.FORBID_COMMENT.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getForbidCommentLimitRedisKey(userId)))) {
                userBlockDTO.setForbidComment(true);
            }
            if (BlockUserTypeEnum.FORBID_POST.getBlockType().equals(userBlock.getBlockType()) &&
                    isNotBlank(redisStringAdapter.getString(getForbidPostLimitRedisKey(userId)))) {
                userBlockDTO.setForbidPost(true);
            }
        });

        return userBlockDTO;
    }

    private void cancelLimit(UserBlockDTO userBlockDTO, BlockUserTypeEnum blockUserTypeEnum, KeyGenerator redisKey) {

        Long userId = userBlockDTO.getUserId();
        UserBlock userBlock = new UserBlock();
        userBlock.setUserId(userId);
        userBlock.setBlockType(blockUserTypeEnum.getBlockType());
        userBlockMapper.deleteByUserIdAndType(userBlock);
        if (Objects.nonNull(redisKey)) {
            redisStringAdapter.remove(redisKey);
        }
    }

    /**
     * 添加限制条件
     *
     * @param userBlockDTO      用户限制信息
     * @param blockUserTypeEnum 限制类型
     * @param redisKey          redisKey
     */
    private void limit(UserBlockDTO userBlockDTO, BlockUserTypeEnum blockUserTypeEnum, KeyGenerator redisKey) {
        Date blockTime = userBlockDTO.getBlockTime();

        addRecord(userBlockDTO, blockUserTypeEnum.getBlockType());

        redisStringAdapter.set(redisKey, blockTime);
        redisStringAdapter.expire(redisKey, blockTime);

    }

    /**
     * 添加限制记录
     *
     * @param userBlockDTO 用户显示信息
     * @param blockType    限制类型
     */
    private void addRecord(UserBlockDTO userBlockDTO, Integer blockType) {

        UserBlock userBlock = UserBlock.builder()
                .id(sequenceCreater.nextLongId())
                .userId(userBlockDTO.getUserId())
                .blockTime(userBlockDTO.getBlockTime())
                .blockType(blockType)
                .build();

        userBlockMapper.deleteByUserIdAndType(userBlock);
        userBlockMapper.insertSelective(userBlock);
    }

    /**
     * 构造聊天限制redis key
     *
     * @param userId 用户id
     * @return key
     */
    private KeyGenerator getChatLimitRedisKey(Long userId) {
        return RedisConfig.IM_BLOCK_LIST.copy().appendKey(userId);
    }

    /**
     * 构造拉黑限制redis key
     *
     * @param userId 用户id
     * @return key
     */
    private KeyGenerator getBlockLimitRedisKey(Long userId) {
        return RedisConfig.USER_BLOCK_LIMIT.copy().appendKey(userId);
    }

    /**
     * 构造评论限制redis key
     *
     * @param userId 用户id
     * @return key
     */
    private KeyGenerator getForbidCommentLimitRedisKey(Long userId) {
        return RedisConfig.FORBID_COMMENT_LIMIT.copy().appendKey(userId);
    }

    /**
     * 构造发帖限制redis key
     *
     * @param userId 用户id
     * @return key
     */
    private KeyGenerator getForbidPostLimitRedisKey(Long userId) {
        return RedisConfig.FORBID_POST_LIMIT.copy().appendKey(userId);
    }
}
