package com.bxm.egg.user.invite.processor;

import com.bxm.egg.user.enums.AwardTypeEnum;
import com.bxm.egg.user.enums.InviteBindMethodEnum;
import com.bxm.egg.user.enums.InviteRecordStateEnum;
import com.bxm.egg.user.enums.InviteRecordStatusEnum;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.invite.InviteProcessorContext;
import com.bxm.egg.user.invite.InviteRecordService;
import com.bxm.egg.user.invite.InviteTypeProcessor;
import com.bxm.egg.user.invite.bind.BindInviteManager;
import com.bxm.egg.user.model.dto.UserInfoDTO;
import com.bxm.egg.user.model.dto.UserInviteBindDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.vo.InviteRecord;
import com.bxm.egg.user.model.vo.UserInviteHistoryBean;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;

/**
 * 抽象的邀请类型处理器
 *
 * @author liujia
 * @date 2020/05/11 15:37
 */
@Slf4j
public abstract class AbstractInviteTypeProcessor implements InviteTypeProcessor {
    /**
     * 显示标题的默认长度，防止超长，标题在客户端显示不正确
     */
    int defaultColumnWidth = 80;

    String defaultInviteMsg = "的其他分享";

    @Resource
    InviteRecordService inviteRecordService;

    @Autowired
    private BindInviteManager bindInviteManager;

    @Autowired
    private UserInfoService userInfoService;

    @Override
    public final Message afterInvited(InviteProcessorContext context) {
        log.info("邀请成功，邀请记录信息：{}", context);
        fillContext(context);

        //邀请结果写入到邀请成功记录表--可能存在已经绑定，所以会save返回false
        UserInviteBindDTO bean = convertInviteBind(context);
        Message message = Message.build(true);
        //绑定邀请关系
        if (null == bindInviteManager.bindInvite(bean)) {
            message.setSuccess(false);
        }
        if (!message.isSuccess()) {
            return message;
        }
        //触发不同邀请的后续处理回调逻辑
        message.append(callback(context));

        return message;
    }

    private void fillContext(InviteProcessorContext context) {
        if (context.getInvitedUser() == null) {
            UserInfoDTO inviteUser = new UserInfoDTO();
            UserInfoEntity userInfoEntity = userInfoService.selectUserById(context.getRecord().getInvitedUserId());
            BeanUtils.copyProperties(userInfoEntity, inviteUser);
            context.setInvitedUser(inviteUser);
        }
    }

    /**
     * 更新邀请人的赏金状态
     */
    private void changeInviteAwardState(InviteProcessorContext context) {
        InviteRecord inviteRecord = context.getRecord();

        //邀请关系被解除
        if (!InviteRecordStateEnum.ACCEPT_INVITE.name().equals(inviteRecord.getInviteState())) {
            log.info("当前邀请人[{}]与受邀人[{}]的状态处于[{}]------不予处理",
                    inviteRecord.getUserId(),
                    inviteRecord.getInvitedUserId(),
                    inviteRecord.getInviteState());
        }

        if (context.isEffective()) {

            //更新邀请状态为已登陆
            inviteRecord.setInviteState(InviteRecordStateEnum.LOGIN_APP.getName());
            inviteRecord.setEffectTime(null);
            inviteRecord.setAwardType(AwardTypeEnum.CASH.name());

        } else {

            //更新邀请状态为无效
            inviteRecord.setInviteState(InviteRecordStateEnum.OVERDUE_INVALID.name());
            inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
            inviteRecord.setStatus(InviteRecordStatusEnum.INVALID.getStatus());
        }
        inviteRecordService.updateInviteRecord(inviteRecord);
    }


    private UserInviteBindDTO convertInviteBind(InviteProcessorContext context) {
        UserInviteBindDTO bean = new UserInviteBindDTO();
        bean.setUserId(context.getRecord().getInvitedUserId());
        if (context.getInvitedUser() != null) {
            bean.setInviteUserId(context.getInvitedUser().getInviteUserId());
        } else {
            bean.setInviteUserId(context.getRecord().getUserId());
        }
        bean.setInviteBindMethodEnum(InviteBindMethodEnum.LOGIN_APP);
        return bean;
    }

    private UserInviteHistoryBean convert(InviteProcessorContext context) {
        UserInviteHistoryBean history = new UserInviteHistoryBean();
        history.setUserId(context.getRecord().getInvitedUserId());
        history.setCreateTime(context.getRecord().getCreateTime());

        if (context.getInvitedUser() != null) {
            history.setInviteUserId(context.getInvitedUser().getInviteUserId());
            history.setInviteUserNickname(context.getRecord().getInvitedUserName());
            history.setInviteHeadImg(context.getRecord().getInvitedUserImg());
            history.setRelationId(context.getInvitedUser().getRelationId());
            if (null != context.getInvitedUser().getIsVip()) {
                history.setActiveVip(context.getInvitedUser().getIsVip().byteValue());
            }
            if (null != context.getRecord()) {
                history.setType(context.getRecord().getType());
            }
        } else {
            history.setInviteUserId(context.getRecord().getUserId());
            history.setInviteUserNickname(context.getRecord().getInvitedUserName());
            history.setInviteHeadImg(context.getRecord().getInvitedUserImg());
            history.setType(context.getRecord().getType());
        }

        //从实现类获取具体的标题，不同邀请类型文案和获取方式不同
        String title = obtainTitle(context);
        history.setRelationTitle(StringUtils.substring(title, 0, 200));

        return history;
    }

    /**
     * 不同邀请类型邀请成功后的回调处理逻辑
     *
     * @param context 邀请记录上下文信息
     * @return 邀请结果
     */
    abstract Message callback(InviteProcessorContext context);
}
