package com.bxm.egg.user.login.impl;

import com.bxm.component.bus.config.ComponentEventBus;
import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.dto.LocationDTO;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.integration.LocationIntegrationService;
import com.bxm.egg.user.integration.sync.HuolaUserDataSyncIntegrationService;
import com.bxm.egg.user.location.UserLocationService;
import com.bxm.egg.user.login.LoginAfterHandlerService;
import com.bxm.egg.user.login.event.LoginEvent;
import com.bxm.egg.user.model.dto.login.LoginResultDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.login.LoginParam;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.newidea.component.model.param.PointParam;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.util.PointReportUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Date;

/**
 * @author lowi
 * @date 2021/11/25 20:06
 */
@Service
@AllArgsConstructor
@Slf4j
public class LoginAfterHandlerServiceImpl implements LoginAfterHandlerService {

    private final UserInfoService userInfoService;

    private final HuolaUserDataSyncIntegrationService huolaUserDataSyncIntegrationService;

    private final UserProperties userProperties;

    private final UserExtendService userExtendService;

    private final LocationIntegrationService locationIntegrationService;

    private final UserLocationService userLocationService;

    @Override
    @Async
    public <T extends LoginParam> void handleUserEggMigration(Long userId, UserLoginTypeEnum userLoginTypeEnum,
                                                              T param, LoginResultDTO resultDTO, boolean isNewPeople) {
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(userId);
        //如果是蛋蛋佳的用户并且未迁移，进行迁移
        if (userInfoEntity != null && userInfoEntity.getEggUser() == 1 && userInfoEntity.getMigration() == 0) {
            huolaUserDataSyncIntegrationService.userDataSync(userId);
            //更新用户迁移状态
            updateSyncStatus(userId);
            //初始化默认用户迁移定位
            initMigrationLocation(userId);
            //初始化资料完成度
            userExtendService.initComplete(userId);
            //迁移重新上传用户头像
            uploadOssHeadImg(userId);
        }

        // 发布登录事件，用于处理登录以后的后续流程
        publishLoginEvent(userLoginTypeEnum, param, userId, resultDTO);

        //添加埋点
        PointReportUtils.report(PointParam.build(param)
                .event(isNewPeople ? 1002 : 1003)
                .userId(userId)
                .project(param.getSrcApp())
        );
    }

    /**
     * 更新用户同步状态
     *
     * @param userId 用户id
     */
    private void updateSyncStatus(Long userId) {
        UserInfoEntity updateUserInfoEntity = new UserInfoEntity();

        updateUserInfoEntity.setId(userId);
        updateUserInfoEntity.setMigration(1);
        updateUserInfoEntity.setActiveTime(new Date());

        userInfoService.updateUserInfo(updateUserInfoEntity);
    }


    /**
     * 初始化迁移用户定位
     *
     * @param userId 用户id
     */
    private void initMigrationLocation(Long userId) {
        LocationDTO defaultLocationDTO = locationIntegrationService.getLocationByGeocode(userProperties.getDefaultLocationCode());
        if (defaultLocationDTO != null) {
            userLocationService.updateUserLocationInfo(userId,
                    userProperties.getDefaultLocationCode(),
                    defaultLocationDTO.getName());
        }
    }

    /**
     * 迁移用户重新上传用户头像到OSS
     *
     * @param userId 用户id
     */
    private void uploadOssHeadImg(Long userId) {
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(userId);
        UserInfoEntity updateUserInfo = new UserInfoEntity();

        String newHeadImg = userProperties.getDefaultHeadImageUrl();
        if (StringUtils.isNotBlank(userInfoEntity.getHeadImg())) {
            newHeadImg = userExtendService.uploadWechatImage(userInfoEntity.getHeadImg(), userId);
        }

        updateUserInfo.setId(userId);
        updateUserInfo.setHeadImg(newHeadImg);

        userInfoService.updateUserInfo(updateUserInfo);
    }

    /**
     * 发布登录事件
     */
    protected <T extends LoginParam> void publishLoginEvent(UserLoginTypeEnum userLoginTypeEnum, T param, Long userId, LoginResultDTO resultDTO) {
        LoginEvent<T> event = new LoginEvent<>();
        event.setLoginResult(resultDTO);
        event.setLoginParam(param);
        event.setUserId(userId);
        event.setLoginType(userLoginTypeEnum);
        ComponentEventBus.publishInnerEvent(event);
    }

}
