package com.bxm.egg.user.login.strategy;

import com.bxm.egg.common.enums.UserTypeEnum;
import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.constant.UserStateEnum;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.login.LoginAfterHandlerService;
import com.bxm.egg.user.login.UserLogService;
import com.bxm.egg.user.mapper.UserInfoMapper;
import com.bxm.egg.user.model.dto.login.LoginResultDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.login.LoginParam;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.sync.facade.dto.UserRegisterDTO;
import com.bxm.sync.facade.param.UserRegisterParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Date;

/**
 * 用户登录策略抽象类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/14 2:04 下午
 */
@Slf4j
@Component
public abstract class AbstractUserLoginStrategy<T extends LoginParam> implements IUserLoginStrategy<T> {

    public static final String LOGIN_RESULT_DTO = "loginResultDTO";

    @Resource
    protected DistributedLock distributedLock;

    @Resource
    protected UserInfoService userInfoService;

    @Resource
    protected UserLogService userLogService;

    @Resource
    protected UserInfoMapper userInfoMapper;

    @Resource
    protected UserProperties userProperties;

    @Resource
    private UserExtendService userExtendService;

    @Autowired
    private LoginAfterHandlerService loginAfterHandlerService;

    /**
     * 登录后的处理
     *
     * @param resultDTO  登录结果
     * @param loginParam 登录参数
     */
    protected void afterLogin(LoginResultDTO resultDTO, T loginParam) {
    }

    /**
     * 检查用户是否存在，每种登录方式检查逻辑有所区别
     *
     * @param loginParam 登录参数
     * @return 用户ID，如果为null表示用户不存在
     */
    protected abstract Long checkExists(T loginParam);

    /**
     * 当前登录方式
     *
     * @return 登录方式枚举
     */
    public abstract UserLoginTypeEnum type();

    /**
     * 保存用户的某种登录信息的唯一性记录
     *
     * @param loginParam 登录参数
     * @param userId     用户ID
     * @return 返回true才会进行后续处理
     */
    protected abstract boolean saveIdentity(T loginParam, Long userId);

    /**
     * 填充用户信息
     *
     * @param loginParam 登录参数
     * @param newUser    新用户信息
     */
    protected abstract void fillUserInfo(T loginParam, UserInfoEntity newUser);

    /**
     * 执行登录逻辑
     *
     * @param loginParam 登录参数
     * @return 登录结果
     */
    @Override
    public Message doLogin(T loginParam) {
        String resource = loginParam.hashCode() + "";

        if (!distributedLock.lock(resource)) {
            log.info("出现重复点击，请求参数：{}", JSON.toJSONString(loginParam));
            return Message.build(false, "正在登录，请稍后");
        }

        //前置登录认证
        Message beforeAuthenticationResult = beforeLoginAuthentication(loginParam);

        if (!beforeAuthenticationResult.isSuccess()) {
            return beforeAuthenticationResult;
        }

        // 判断是否已经为既有用户
        Long userId = checkExists(loginParam);

        LoginResultDTO resultDTO = null;
        boolean isNewPeople = false;

        //如果用户存在，则进入处理已存在用户逻辑，如果用户不存在，则进入处理新用户逻辑
        if (null != userId) {
            Message handleExistsUserMsg = handleExistsUser(userId, loginParam);

            if (!handleExistsUserMsg.isSuccess()) {
                return handleExistsUserMsg;
            }
            resultDTO = handleExistsUserMsg.getParam(LOGIN_RESULT_DTO);
            if (null != resultDTO) {
                updateExistUserInfo(userId, loginParam);
            }
        }

        // 历史账号可能也是无效用户（主要是注销测试的情况）
        if (null == resultDTO) {
            Message handleNewUserMsg = handleNewUser(loginParam);
            if (!handleNewUserMsg.isSuccess()) {
                return handleNewUserMsg;
            }
            resultDTO = handleNewUserMsg.getParam(LOGIN_RESULT_DTO);
            isNewPeople = true;
        }

        if (null == resultDTO) {
            return Message.build(false, "您的账户信息不存在");
        }

        //后置登录认证
        userId = resultDTO.getUserId();
        Message afterAuthenticationResult = afterLoginAuthentication(loginParam, resultDTO);

        //如果后置验证不成功
        if (!afterAuthenticationResult.isSuccess()) {
            return afterAuthenticationResult;
        }

        //处理用户迁移数据
        loginAfterHandlerService.handleUserEggMigration(userId, type(), loginParam, resultDTO, isNewPeople);
        distributedLock.unlock(resource);

        afterLogin(resultDTO, loginParam);

        //填充用户手机号码，到这一步肯定有手机号码
        fillUserPhone(resultDTO);

        if (log.isDebugEnabled()) {
            log.debug("用户信息登录：{}", JSON.toJSONString(resultDTO));
        }

        return Message.build(true).addParam(LOGIN_RESULT_DTO, resultDTO);
    }


    /**
     * 填充用户手机号码
     *
     * @param resultDTO 返回登录信息
     */
    protected void fillUserPhone(LoginResultDTO resultDTO) {
        if (StringUtils.isBlank(resultDTO.getPhone())) {
            UserInfoEntity userInfoEntity = userInfoService.selectUserById(resultDTO.getUserId());
            resultDTO.setPhone(userInfoEntity.getPhone());
        }

        if (log.isDebugEnabled()) {
            log.debug("用户返回手机号码为：{}", resultDTO.getPhone());
        }
    }


    /**
     * 对已经存在的用户信息进行更新
     *
     * @param userId     用户id
     * @param loginParam 登录参数
     */
    protected void updateExistUserInfo(Long userId, T loginParam) {

    }

    /**
     * 处理完用户前置的登录认证，如果不通过，则登录失败
     *
     * @return 是否通过登录验证
     */
    protected Message beforeLoginAuthentication(T loginParam) {
        return Message.build(true);
    }

    /**
     * 处理完用户后置的登录认证，如果不通过，则登录失败
     *
     * @param loginParam 登录参数
     * @param resultDTO  登录返回结果
     * @return 结果信息
     */
    protected Message afterLoginAuthentication(T loginParam, LoginResultDTO resultDTO) {
        return Message.build(true);
    }

    /**
     * 处理新用户
     *
     * @param loginParam 登录参数
     * @return 处理返回信息
     */
    protected Message handleNewUser(T loginParam) {

        UserInfoEntity newUser = new UserInfoEntity();

        newUser.setEquipmentId(loginParam.getDevcId());
        newUser.setState(UserStateEnum.NORMAL.getCode());
        newUser.setLastLoginTime(new Date());
        newUser.setSalt(RandomUtils.getRandomStr(6, true));
        newUser.setHeadImg(userProperties.getDefaultHeadImageUrl());

        UserRegisterDTO userRegisterDTO = userInfoService.generateUserId(UserRegisterParam.builder().phone(loginParam.getPhone())
                .salt(newUser.getSalt())
                .build());
        newUser.setId(userRegisterDTO.getUserId());
        newUser.setCreateTime(userRegisterDTO.getCreateTime());
        newUser.setSyncTime(userRegisterDTO.getSyncDate());
        newUser.setEggUser(userRegisterDTO.getEggUser());

        if (StringUtils.isNotBlank(loginParam.getPhone())) {
            newUser.setNickname(StringUtils.hideMobile(loginParam.getPhone()));
            newUser.setUsername(loginParam.getPhone());
        }

        boolean isTestUser = userProperties.getTestUserPhone().contains(loginParam.getPhone());
        //如果是测试用户，则设置用户类型为测试用户类型
        if (isTestUser) {
            newUser.setType(UserTypeEnum.TEST.getCode());
        }

        // 填充用户基础信息，不同的登录方式填充字段有区别
        fillUserInfo(loginParam, newUser);

        if (null == newUser.getHeadImg()) {
            newUser.setHeadImg(userProperties.getDefaultHeadImageUrl());
        }

        // 保存扩展信息
        if (saveIdentity(loginParam, userRegisterDTO.getUserId())) {
            // 创建新用户
            userInfoMapper.insert(newUser);
            //初始化用户相关扩展数据
            userExtendService.initUserExtendData(newUser);
            //记录日志
            userLogService.recordRegisterLog(newUser, loginParam);
        }

        Message message = Message.build(true);

        return message.addParam("loginResultDTO", LoginResultDTO.builder()
                .firstLogin(true)
                .userId(newUser.getId())
                .nickName(newUser.getNickname())
                .headImg(newUser.getHeadImg())
                .build());
    }

    /**
     * 处理已存在用户
     *
     * @param userId     用户id
     * @param loginParam 登录入参
     * @return 处理返回信息
     */
    protected Message handleExistsUser(Long userId, T loginParam) {
        UserInfoEntity userInfo = userInfoService.selectUserById(userId);

        if (userInfo == null) {
            log.error("登录用户ID[{}]在数据库中不存在或已删除，登录时仍然进行传递", userId);
            // 这里是做一个特殊的处理，主要是兼容客户端可能没有删除用户数据的问题
            return Message.build(true);
        }

        LoginResultDTO resultDTO = LoginResultDTO.builder()
                .firstLogin(false)
                .userId(userInfo.getId())
                .headImg(userInfo.getHeadImg())
                .nickName(userInfo.getNickname())
                .build();

        if (UserStateEnum.NOT_ACTIVATED.equalsCode(userInfo.getState())) {
            userInfoService.updateStatus(userId, UserStateEnum.NORMAL);
            // 激活用户流程，激活的用户也算作首次登录
            resultDTO.setFirstLogin(true);
        } else if (UserStateEnum.BLACK.equalsCode(userInfo.getState())) {
            log.info("屏蔽的用户[{}]尝试登录，请求参数：{}", userId, JSON.toJSON(loginParam));
            return Message.build(false, "登录中，请稍后");
        } else {
            if (!UserStateEnum.NORMAL.equalsCode(userInfo.getState())) {
                log.info("用户状态[{}]不匹配，请求参数：{}", userInfo.getState(), JSON.toJSON(loginParam));
                return Message.build(false, "登录中，请稍后");
            }
        }


        Message message = Message.build(true);
        message.addParam("loginResultDTO", resultDTO);

        return message;
    }

}