package com.bxm.egg.user.login.strategy.support;

import com.alibaba.fastjson.JSON;
import com.bxm.egg.user.constant.BizCodeConstant;
import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.login.strategy.AbstractUserLoginStrategy;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.login.UserAccountAppLoginParam;
import com.bxm.newidea.component.bo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Optional;

/**
 * 用户名密码登录策略【这种策略是不可能会去创建新用户的】
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/14 1:57 下午
 */
@Slf4j
@Service
public class AccountLoginStrategy extends AbstractUserLoginStrategy<UserAccountAppLoginParam> {

    @Resource
    private UserInfoService userInfoService;

    @Override
    protected Long checkExists(UserAccountAppLoginParam loginParam) {
        //判断用户名是否存在
        Optional<Long> userIdOptional = userInfoService.userUsernameExist(loginParam.getUserName());

        if (!userIdOptional.isPresent()) {
            log.error("登录的用户名不存在，跳过了前面的登录校验，需排查原因，登录入参：{}", JSON.toJSONString(loginParam));
            return null;
        }

        return userIdOptional.get();
    }

    @Override
    protected Message beforeLoginAuthentication(UserAccountAppLoginParam loginParam) {
        //判断用户名是否存在
        Optional<Long> userIdOptional = userInfoService.userPhoneExist(loginParam.getUserName());

        if (!userIdOptional.isPresent()) {
            log.warn("用户登录，输入的用户名不存在，登录入参：{}", JSON.toJSONString(loginParam));
            return Message.build(false, "您登录的用户名不存在");
        }

        //判断用户是否设置密码
        if (!userInfoService.hasSettingPassword(loginParam.getUserName())) {
            Message errorMsg = Message.build(false, "您的账户未设置密码");
            errorMsg.setBizCode(BizCodeConstant.NEED_SETTING_PASSWORD);

            return errorMsg;
        }


        //验证用户名密码是否正确
        if (!userInfoService.checkUserPassword(userIdOptional.get(),
                loginParam.getUserName(), loginParam.getPassword())) {
            return Message.build(false, "您输入的密码错误");
        }


        return Message.build(true);
    }

    @Override
    public UserLoginTypeEnum type() {
        return UserLoginTypeEnum.ACCOUNT;
    }

    @Override
    protected boolean saveIdentity(UserAccountAppLoginParam loginParam, Long userId) {
        return true;
    }

    @Override
    protected void fillUserInfo(UserAccountAppLoginParam loginParam, UserInfoEntity newUser) {

    }

    @Override
    protected Message handleNewUser(UserAccountAppLoginParam loginParam) {
        log.error("登录的用户名不存在，跳过了前面的登录校验，需排查原因，登录入参：{}", JSON.toJSONString(loginParam));

        return Message.build(false, "您登录的用户名不存在");
    }
}