package com.bxm.egg.user.login.strategy.support;

import com.alibaba.fastjson.JSON;
import com.bxm.egg.mq.common.constant.SmsTemplateEnum;
import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.integration.SmsIntegrationService;
import com.bxm.egg.user.login.strategy.AbstractUserLoginStrategy;
import com.bxm.egg.user.model.dto.login.LoginResultDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.login.UserSettingsPasswordLoginParam;
import com.bxm.newidea.component.bo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Optional;

/**
 * 设置账户密码后登录【这种登录方式也不会去创建新用户】
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/14 8:05 下午
 */
@Slf4j
@Service
public class SettingsPasswordLoginStrategy extends AbstractUserLoginStrategy<UserSettingsPasswordLoginParam> {

    @Resource
    private UserInfoService userInfoService;

    @Resource
    private SmsIntegrationService smsIntegrationService;

    @Override
    protected Message beforeLoginAuthentication(UserSettingsPasswordLoginParam loginParam) {
        //验证账号是否存在
        Optional<Long> userPhoneExist = userInfoService.userPhoneExist(loginParam.getPhone());

        if (!userPhoneExist.isPresent()) {
            return Message.build(false, "手机号码不存在");
        }

        //验证码是否正确
        if (!smsIntegrationService.verifyByType(SmsTemplateEnum.RESET_PWD, loginParam.getPhone(), loginParam.getSmsCode())) {
            return Message.build(false, "验证码错误");
        }

        return Message.build(true);
    }

    @Override
    protected Long checkExists(UserSettingsPasswordLoginParam loginParam) {
        Optional<Long> userPhoneExist = userInfoService.userPhoneExist(loginParam.getPhone());

        if (!userPhoneExist.isPresent()) {
            log.error("设置密码登录的用户不存在， 用户信息：{}", JSON.toJSONString(loginParam));
            return null;
        }
        loginParam.setUserId(userPhoneExist.get());
        return userPhoneExist.get();
    }

    @Override
    public UserLoginTypeEnum type() {
        return UserLoginTypeEnum.SETTINGS_PASSWORD_LOGIN;
    }

    @Override
    protected boolean saveIdentity(UserSettingsPasswordLoginParam loginParam, Long userId) {
        return true;
    }

    @Override
    protected void fillUserInfo(UserSettingsPasswordLoginParam loginParam, UserInfoEntity newUser) {

    }

    @Override
    protected void afterLogin(LoginResultDTO resultDTO, UserSettingsPasswordLoginParam loginParam) {
        //登录成功后，修改密码
        userInfoService.updateUserPassword(loginParam.getUserId(), loginParam.getPassword());
    }

    @Override
    protected Message handleNewUser(UserSettingsPasswordLoginParam loginParam) {
        log.error("修改密码的用户名不存在，跳过了前面的登录校验，需排查原因，登录入参：{}", JSON.toJSONString(loginParam));

        return Message.build(false, "用户名不存在");
    }
}