package com.bxm.egg.user.login.strategy.support;

import com.bxm.egg.mq.common.constant.SmsTemplateEnum;
import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.convert.EggSixParamConverter;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserWechatGrantService;
import com.bxm.egg.user.integration.EggSixEnjoyIntegrationService;
import com.bxm.egg.user.integration.SmsIntegrationService;
import com.bxm.egg.user.login.strategy.AbstractUserLoginStrategy;
import com.bxm.egg.user.model.dto.login.LoginResultDTO;
import com.bxm.egg.user.model.entity.UserGrantWechatAppEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.UserBindWxParam;
import com.bxm.egg.user.model.param.login.WechatBindMobileLoginParam;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Optional;

/**
 * 微信绑定手机号登录策略
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/14 8:03 下午
 */
@Slf4j
@Service
public class WechatBindMobileLoginStrategy extends AbstractUserLoginStrategy<WechatBindMobileLoginParam> {
    @Resource
    private UserInfoService userInfoService;

    @Resource
    private SmsIntegrationService smsIntegrationService;

    @Resource
    private UserExtendService userExtendService;

    @Resource
    private UserWechatGrantService userWechatGrantService;

    @Resource
    private EggSixEnjoyIntegrationService eggSixEnjoyIntegrationService;

    @Override
    protected Message beforeLoginAuthentication(WechatBindMobileLoginParam loginParam) {

        //验证码是否正确
        if (!smsIntegrationService.verifyByType(SmsTemplateEnum.BINDING_PHONE, loginParam.getPhone(), loginParam.getSmsCode())) {
            return Message.build(false, "验证码错误");
        }

        //验证是否授权
        Optional<UserGrantWechatAppEntity> userUnionIdExist = userWechatGrantService.userInfoUnionIdExist(loginParam.getUnionId());

        if (!userUnionIdExist.isPresent()) {
            return Message.build(false, "微信用户未授权");
        }

        return Message.build(true);
    }

    @Override
    protected Long checkExists(WechatBindMobileLoginParam loginParam) {
        //判断用户绑定的手机号是否已经存在
        Optional<Long> userPhoneExist = userInfoService.userPhoneExist(loginParam.getPhone());

        //判断手机号是否有绑定的用户，如果有则不用创建新用户直接登录，如果没有则要创建新用户
        if (!userPhoneExist.isPresent()) {
            return null;
        }
        loginParam.setUserId(userPhoneExist.get());
        return userPhoneExist.get();
    }

    @Override
    public UserLoginTypeEnum type() {
        return UserLoginTypeEnum.WECHAT_BIND_MOBILE_LOGIN;
    }

    @Override
    protected boolean saveIdentity(WechatBindMobileLoginParam loginParam, Long userId) {
        return true;
    }

    @Override
    protected void fillUserInfo(WechatBindMobileLoginParam loginParam, UserInfoEntity newUser) {
        String newHeadImg = userExtendService.uploadWechatImage(loginParam.getHeadImageUrl(),  newUser.getId());

        newUser.setHeadImg(newHeadImg);
        newUser.setNickname(loginParam.getNickName());
        newUser.setSex(loginParam.getSex());
    }

    @Override
    protected void afterLogin(LoginResultDTO resultDTO, WechatBindMobileLoginParam loginParam) {
        //理论登录绑定手机号只会执行一次，所以放到这个方法里面也可以
        //登录后绑定手机号
        userInfoService.bindPhone(resultDTO.getUserId(), loginParam.getPhone());

        //绑定手机号为用户名
        userInfoService.bindUsername(resultDTO.getUserId(), loginParam.getPhone());

        //修改绑定用户
        userWechatGrantService.updateWechatGrantBindUserId(resultDTO.getUserId(), loginParam.getUnionId());

        //微信登录绑定手机号也要初始化一下资料完成度
        userExtendService.initComplete(resultDTO.getUserId());


        if (StringUtils.isGrateOrEqualThan(loginParam.getCurVer(), "2.0.1")) {
            UserBindWxParam userBindWxParam = new UserBindWxParam();
            userBindWxParam.setUserId(resultDTO.getUserId());
            userBindWxParam.setOpenId(loginParam.getOpenId());
            userBindWxParam.setUnionId(loginParam.getUnionId());
            userBindWxParam.setNickName(loginParam.getNickName());
            userBindWxParam.setDevcId(loginParam.getDevcId());
            userBindWxParam.setHeadImageUrl(loginParam.getHeadImageUrl());
            userBindWxParam.setPhone(loginParam.getPhone());
            userWechatGrantService.userBindWxHandler(userBindWxParam);
        } else {
            //调用六享微信绑定奖励
            eggSixEnjoyIntegrationService.bind(EggSixParamConverter.convert(loginParam, resultDTO.getUserId()));
        }
    }

    @Override
    protected void updateExistUserInfo(Long userId, WechatBindMobileLoginParam loginParam) {
        UserInfoEntity updateUserInfo = new UserInfoEntity();
        String newHeadImg = userExtendService.uploadWechatImage(loginParam.getHeadImageUrl(),  userId);

        updateUserInfo.setEquipmentId(loginParam.getDevcId());
        updateUserInfo.setId(userId);
        updateUserInfo.setHeadImg(newHeadImg);
        updateUserInfo.setNickname(loginParam.getNickName());
        updateUserInfo.setSex(loginParam.getSex());

        userInfoService.updateUserInfo(updateUserInfo);
    }
}