package com.bxm.egg.user.medal.impl;

import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.enums.MedalCategoryEnum;
import com.bxm.egg.user.enums.UserAchievementMedalStatusEnum;
import com.bxm.egg.user.enums.WarmValueEquityEnum;
import com.bxm.egg.user.info.UserInfoCacheService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.msg.UserInfoChangeSender;
import com.bxm.egg.user.mapper.medal.UserMedalRelationInfoMapper;
import com.bxm.egg.user.medal.UserMedalService;
import com.bxm.egg.user.medal.cache.AchievementMedalCacheManager;
import com.bxm.egg.user.model.bo.UserCacheInfoBO;
import com.bxm.egg.user.model.dto.medal.MainPageMedalImageDTO;
import com.bxm.egg.user.model.dto.medal.MedalMainPageSimpleDTO;
import com.bxm.egg.user.model.dto.warmlevel.UserWarmLevelInfoDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.entity.medal.UserMedalInfoEntity;
import com.bxm.egg.user.model.entity.medal.UserMedalRelationInfoEntity;
import com.bxm.egg.user.model.param.medal.SaveWearMedalParam;
import com.bxm.egg.user.model.vo.medal.*;
import com.bxm.egg.user.warmlevel.UserWarmLevelService;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisListAdapter;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 用户勋章服务实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/3/4 3:24 下午
 */
@Service
@Slf4j
@AllArgsConstructor
public class UserMedalServiceImpl implements UserMedalService {

    private AchievementMedalCacheManager achievementMedalCacheManager;

    private UserMedalRelationInfoMapper userMedalRelationInfoMapper;

    private UserInfoService userInfoService;

    private UserWarmLevelService userWarmLevelService;

    private UserInfoCacheService userInfoCacheService;

    private UserInfoChangeSender userInfoChangeSender;

    private RedisHashMapAdapter redisHashMapAdapter;

    @Override
    public OthersMedalVO getOthersMedal(Long othersUserId) {

        OthersMedalVO resultMedalVO = new OthersMedalVO();

        //1、查询用户信息
        UserCacheInfoBO userCacheInfoBO = userInfoCacheService.load(othersUserId);

        resultMedalVO.setUserId(userCacheInfoBO.getUserId());
        resultMedalVO.setHeadImg(userCacheInfoBO.getHeadImg());
        resultMedalVO.setNickname(userCacheInfoBO.getNickname());
        resultMedalVO.setMedalNum(getUserMedalNum(othersUserId));

        //2、获取用户各种类别的勋章中等级最高成就勋章信息
        List<UserMedalRelationInfoEntity> userAllAchievementMedalList =
                userMedalRelationInfoMapper.getUserMaxLevelAchievementMedal(othersUserId);

        //其他用户的返回的成就勋章列表
        List<SingleAchievementMedalBaseVO> achievementMedalList = new ArrayList<>();

        //不为空则进行填充
        if (Boolean.FALSE.equals(CollectionUtils.isEmpty(userAllAchievementMedalList))) {
            //对象进行转换
            for (UserMedalRelationInfoEntity userMedalRelationInfoEntity : userAllAchievementMedalList) {
                achievementMedalList.add(convert2SingleAchievementMedalBaseVO(userMedalRelationInfoEntity.getMedalId(), UserAchievementMedalStatusEnum.NOT_UP_LEVEL.getCode()));
            }

            //只取等级最高的勋章并且按照tOrder进行排序
            achievementMedalList = achievementMedalList.stream().sorted(Comparator.comparing(SingleAchievementMedalBaseVO::getTOrder)).collect(Collectors.toList());
        }
        //设置成就勋章信息
        resultMedalVO.setAchievementMedalList(achievementMedalList);

        //3、获取定制勋章信息
        List<SingleCustomMedalBaseVO> otherUserCustomMedalInfoList = userMedalRelationInfoMapper.getOtherUserCustomMedalInfo(othersUserId);
        for (SingleCustomMedalBaseVO singleCustomMedalBaseVO : otherUserCustomMedalInfoList) {
            singleCustomMedalBaseVO.setExpired(false);
        }
        resultMedalVO.setCustomMedalList(otherUserCustomMedalInfoList);

        return resultMedalVO;
    }

    @Override
    public MyMedalVO getMyMedal(Long userId) {

        MyMedalVO myMedalVO = redisHashMapAdapter.get(builderUserMedal(userId), userId.toString(), MyMedalVO.class);
        if (Objects.nonNull(myMedalVO)) {
            return myMedalVO;
        }

        MyMedalVO resultMyMedalVO = new MyMedalVO();

        //1、查询用户信息
        UserCacheInfoBO userInfoEntity = userInfoCacheService.load(userId);

        resultMyMedalVO.setUserId(userId);
        resultMyMedalVO.setHeadImg(userInfoEntity.getHeadImg());
        resultMyMedalVO.setNickname(userInfoEntity.getNickname());

        //2、成就勋章结果信息
        List<SingleAchievementMedalBaseVO> achievementMedalList = new ArrayList<>();

        //系统所有获取父勋章对应子勋章信息map
        Map<Long, List<UserMedalInfoEntity>> medalParentId2InfoListMap = achievementMedalCacheManager.getMedalParentId2InfoListMap();

        //系统所有获取所有成就父勋章信息
        List<Long> allMedalParentIdList = achievementMedalCacheManager.getAllMedalParentIdList();


        int achievementMedalNum = 0;
        //查找当前用户下是否有这些勋章，如果有取一个等级最高的，并判断该勋章是否是最高等级的，是否可升级
        for (Long parentMedalId : allMedalParentIdList) {
            UserMedalRelationInfoEntity userMaxLevelMedalInfo = userMedalRelationInfoMapper.getUserMaxLevelMedalInfo(userId, parentMedalId);
            //如果为空说明用户从来没有获得过该勋章,则填充一个最小等级的勋章信息，并且是上锁状态
            if (Objects.isNull(userMaxLevelMedalInfo)) {
                List<UserMedalInfoEntity> userMedalInfoEntityList = medalParentId2InfoListMap.get(parentMedalId);
                UserMedalInfoEntity minMedalInfoEntity = userMedalInfoEntityList.stream().min(Comparator.comparing(UserMedalInfoEntity::getLevel)).get();

                achievementMedalList.add(convert2SingleAchievementMedalBaseVO(minMedalInfoEntity.getId(), UserAchievementMedalStatusEnum.UN_LOCK.getCode()));
            } else {

                achievementMedalList.add(convert2SingleAchievementMedalBaseVO(userMaxLevelMedalInfo.getMedalId(),
                        achievementMedalCacheManager.isMaxLevel(parentMedalId, userMaxLevelMedalInfo.getMedalLevel()) ?
                                UserAchievementMedalStatusEnum.NOT_UP_LEVEL.getCode() : UserAchievementMedalStatusEnum.WAIT_UP_LEVEL.getCode()));
                achievementMedalNum++;

            }
        }

        //重新按照tOrder进行排序
        achievementMedalList = achievementMedalList.stream().sorted(Comparator.comparing(SingleAchievementMedalBaseVO::getTOrder)).collect(Collectors.toList());
        //设置成就勋章列表
        resultMyMedalVO.setAchievementMedalList(achievementMedalList);

        int customMedalNum = 0;
        //3、获取定制勋章信息
        List<SingleCustomMedalBaseVO> myUserCustomMedalList = userMedalRelationInfoMapper.getMyUserCustomMedalInfo(userId);
        for (SingleCustomMedalBaseVO singleCustomMedalBaseVO : myUserCustomMedalList) {
            //判断是否过期
            singleCustomMedalBaseVO.setExpired(singleCustomMedalBaseVO.getExpiredStatus() == 0);
            if (singleCustomMedalBaseVO.getExpiredStatus() == 1) {
                customMedalNum++;
            }
        }

        //设置定制勋章
        resultMyMedalVO.setCustomMedalList(myUserCustomMedalList);
        resultMyMedalVO.setMedalNum(achievementMedalNum + customMedalNum);
        redisHashMapAdapter.put(builderUserMedal(userId), userId.toString(), resultMyMedalVO);
        return resultMyMedalVO;
    }

    @Override
    public AchievementMedalDetailVO getAchievementMedalDetail(Long medalParentId, Long userId) {

        AchievementMedalDetailVO resultVO = new AchievementMedalDetailVO();

        //获取用户信息
        UserCacheInfoBO userCacheInfoBO = userInfoCacheService.load(userId);

        resultVO.setUserId(userCacheInfoBO.getUserId());
        resultVO.setHeadImg(userCacheInfoBO.getHeadImg());
        resultVO.setNickname(userCacheInfoBO.getNickname());
        resultVO.setMedalParentId(medalParentId);

        //设置等级信息
        UserWarmLevelInfoDTO userWarmValueInfo = userWarmLevelService.getUserWarmValueInfo(userId);
        resultVO.setUserLevel(userWarmValueInfo.getLevel());
        resultVO.setUserLevelImg(userWarmValueInfo.getLevelUrl());

        //设置当前用户勋章的最高等级
        UserMedalRelationInfoEntity maxLevelMedalInfo = userMedalRelationInfoMapper.getUserMaxLevelMedalInfo(userId, medalParentId);

        if (Objects.nonNull(maxLevelMedalInfo)) {
            //设置勋章等级
            resultVO.setCurrentLevel(maxLevelMedalInfo.getMedalLevel());
        } else {
            resultVO.setCurrentLevel(0);
        }


        //设置等级信息
        Map<Long, List<UserMedalInfoEntity>> medalParentId2InfoListMap = achievementMedalCacheManager.getMedalParentId2InfoListMap();

        List<UserMedalInfoEntity> userMedalInfoEntityList = medalParentId2InfoListMap.get(medalParentId);

        List<MedalLevelVO> medalLevelList = new ArrayList<>();

        for (UserMedalInfoEntity userMedalInfoEntity : userMedalInfoEntityList) {
            MedalLevelVO medalLevelVO = new MedalLevelVO();

            medalLevelVO.setMedalParentId(userMedalInfoEntity.getParentId());
            medalLevelVO.setMedalId(userMedalInfoEntity.getId());
            medalLevelVO.setMedalName(userMedalInfoEntity.getName());
            medalLevelVO.setLevel(userMedalInfoEntity.getLevel());
            medalLevelVO.setUnlockDesc(userMedalInfoEntity.getUnLockDesc());
            medalLevelVO.setExplainText(userMedalInfoEntity.getExplainText());
            medalLevelVO.setUpLevelDesc(userMedalInfoEntity.getUpLevelDesc());
            medalLevelVO.setDisplayIconUrl(userMedalInfoEntity.getGifImgUrl());
            medalLevelVO.setDisplayStaticIconUrl(userMedalInfoEntity.getActiveImgUrl());
            medalLevelVO.setDisplayLockIconUrl(userMedalInfoEntity.getLockImgUrl());
            medalLevelVO.setJumpUrl(userMedalInfoEntity.getJumpUrl());
            //领取记录
            UserMedalRelationInfoEntity medalRecordInfo = userMedalRelationInfoMapper.getByUserIdAndMedalId(userId, userMedalInfoEntity.getId());
            if (Objects.nonNull(medalRecordInfo)) {
                //设置是否解锁
                medalLevelVO.setUnlock(true);
                //设置授予时间
                medalLevelVO.setGrantDateTime(medalRecordInfo.getCreateTime());
            } else {
                medalLevelVO.setUnlock(false);
            }
            medalLevelList.add(medalLevelVO);
        }
        //按照等级重新排序
        medalLevelList = medalLevelList.stream().sorted(Comparator.comparing(MedalLevelVO::getLevel)).collect(Collectors.toList());
        //设置等级信息
        resultVO.setMedalLevelList(medalLevelList);

        return resultVO;
    }

    @Override
    public CustomMedalDetailVO getCustomMedalDetail(Long medalId, Long userId) {
        CustomMedalDetailVO customMedalDetail = userMedalRelationInfoMapper.getCustomMedalDetail(userId, medalId);

        UserInfoEntity userInfoEntity = userInfoService.selectAllUserById(userId);

        customMedalDetail.setUserId(userId);
        customMedalDetail.setNickname(userInfoEntity.getNickname());
        customMedalDetail.setHeadImg(userInfoEntity.getHeadImg());

        //设置等级信息
        UserWarmLevelInfoDTO userWarmValueInfo = userWarmLevelService.getUserWarmValueInfo(userId);
        customMedalDetail.setUserLevel(userWarmValueInfo.getLevel());
        customMedalDetail.setUserLevelImg(userWarmValueInfo.getLevelUrl());

        return customMedalDetail;
    }

    @Override
    public PreviewMedalInfoVO getPreviewMedalInfo(Long userId) {

        PreviewMedalInfoVO resultVO = new PreviewMedalInfoVO();
        //1、获取用户信息
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(userId);

        resultVO.setUserId(userInfoEntity.getId());
        resultVO.setHeadImg(userInfoEntity.getHeadImg());
        resultVO.setNickname(userInfoEntity.getNickname());

        //设置等级信息
        UserWarmLevelInfoDTO userWarmValueInfo = userWarmLevelService.getUserWarmValueInfo(userId);
        resultVO.setUserLevel(userWarmValueInfo.getLevel());
        resultVO.setLevelImgUrl(userWarmValueInfo.getLevelUrl());

        Map<Long, UserMedalInfoEntity> medalId2InfoMap = achievementMedalCacheManager.getMedalId2InfoMap();

        //2、所有勋章的集合
        //获取所有成就勋章最高等级的勋章列表
        List<UserMedalRelationInfoEntity> userMaxLevelAchievementMedalList = userMedalRelationInfoMapper.getUserMaxLevelAchievementMedal(userId);

        List<SinglePreviewMedalVO> previewAchievementList = userMaxLevelAchievementMedalList.stream().map(achievementMedalRecord -> {
            SinglePreviewMedalVO item = new SinglePreviewMedalVO();

            UserMedalInfoEntity userMedalInfoEntity = medalId2InfoMap.get(achievementMedalRecord.getMedalId());
            item.setMedalId(achievementMedalRecord.getMedalId());
            //成就勋章永不过期
            item.setLevel(userMedalInfoEntity.getLevel());
            item.setMedalImgUrl(userMedalInfoEntity.getActiveImgUrl());
            item.setMedalName(userMedalInfoEntity.getName());
            item.setThumbnailUrl(userMedalInfoEntity.getThumbnailUrl());
            item.setHasWear(achievementMedalRecord.getHasWear());
            item.setWearOrder(achievementMedalRecord.getWearOrder());
            item.setTOrder(userMedalInfoEntity.getTOrder());

            return item;
        }).sorted(Comparator.comparing(SinglePreviewMedalVO::getTOrder)).collect(Collectors.toList());

        //获取所有定制勋章的列表
        List<SinglePreviewMedalVO> previewCustomList = userMedalRelationInfoMapper.getAllUsableCustomMedalInfo(userId);

        //组合再一起，重新进行排序
        List<SinglePreviewMedalVO> userAllMedalList = new ArrayList<>();

        userAllMedalList.addAll(previewAchievementList);
        userAllMedalList.addAll(previewCustomList);

        //所有有效的勋章
        userAllMedalList = userAllMedalList.stream()
                .sorted(Comparator.comparing(SinglePreviewMedalVO::getHasWear, Comparator.reverseOrder()).thenComparing(SinglePreviewMedalVO::getWearOrder))
                .collect(Collectors.toList());

        //3、已佩戴勋章的集合
        List<SinglePreviewMedalVO> wearMedalInfo = userMedalRelationInfoMapper.getAllWearMedalInfo(userId);

        resultVO.setMyMedalList(userAllMedalList);
        resultVO.setSelectedMedalList(wearMedalInfo);

        return resultVO;
    }

    @Override
    public Message saveWearMedal(SaveWearMedalParam param) {
        UserWarmLevelInfoDTO userWarmInfoCache = userWarmLevelService.getUserWarmInfoCache(param.getUserId());
        String[] equityType = userWarmInfoCache.getEquityDTO().getEquityType();
        List<String> equityCodeList = Collections.EMPTY_LIST;
        if (Objects.nonNull(equityType) && equityType.length > 0) {
            equityCodeList = Arrays.asList(equityType);
        }

        //判断是否有佩戴勋章资格
        if (Boolean.FALSE.equals(equityCodeList.contains(WarmValueEquityEnum.MEDAL_WEAR.name()))) {
            return Message.build(false, "暂时无法佩戴勋章，提升等级吧！");
        }

        List<Long> medalIdList = param.getMedalId();
        if (medalIdList.size() > 2) {
            medalIdList = medalIdList.subList(0, 2);
        }

        //先取消佩戴的勋章，重新佩戴
        userMedalRelationInfoMapper.unWearUserMedal(param.getUserId());

        //为空直接返回
        if (CollectionUtils.isEmpty(medalIdList)) {
            return Message.build(true);
        }

        //重新佩戴
        for (int i = 0; i < medalIdList.size(); i++) {
            userMedalRelationInfoMapper.wearMedal(param.getUserId(), medalIdList.get(i), i + 1);
        }

        userInfoChangeSender.sendUserChangeMsg(param.getUserId());

        return Message.build(true);
    }

    @Override
    public MedalMainPageSimpleDTO getMainPageMedalDetail(Long userId) {
        MedalMainPageSimpleDTO resultDTO = new MedalMainPageSimpleDTO();

        resultDTO.setMedalNum(getUserMedalNum(userId));

        resultDTO.setMedalImgList(getMainPageMedalImgList(userId));

        return resultDTO;
    }

    private List<String> getMainPageMedalImgList(Long userId) {
        //顺序已经从大到小排好
        List<MainPageMedalImageDTO> userRelationList = userMedalRelationInfoMapper.getRelationListByUserId(userId);

        if (CollectionUtils.isEmpty(userRelationList)) {
            return Collections.emptyList();
        }

        Map<Integer, List<MainPageMedalImageDTO>> groupByUserMedalMap = userRelationList.stream().collect(Collectors.groupingBy(MainPageMedalImageDTO::getMedalCategory));

        //成就勋章列表
        List<MainPageMedalImageDTO> achievementMedalList = groupByUserMedalMap.get(MedalCategoryEnum.ACHIEVEMENT_MEDAL.getCode());

        //定制勋章列表
        List<MainPageMedalImageDTO> customMedalList = groupByUserMedalMap.get(MedalCategoryEnum.CUSTOM_MEDAL.getCode());

        //要过滤的结果list
        List<MainPageMedalImageDTO> resultList = new ArrayList<>();

        if (CollectionUtils.isNotEmpty(customMedalList)) {
            resultList.addAll(customMedalList);
        }

        if (CollectionUtils.isNotEmpty(achievementMedalList)) {
            Map<Long, List<MainPageMedalImageDTO>> achievementMedalMap = achievementMedalList.stream().collect(Collectors.groupingBy(MainPageMedalImageDTO::getMedalParentId));

            achievementMedalMap.forEach((medalParentId, medalImageDTOList) -> {
                if (CollectionUtils.isNotEmpty(medalImageDTOList)) {
                    MainPageMedalImageDTO mainPageMedalImageDTO = medalImageDTOList.stream().max(Comparator.comparing(MainPageMedalImageDTO::getMedalLevel)).get();
                    resultList.add(mainPageMedalImageDTO);
                }
            });
        }

        //按照时间倒序返回勋章缩略图
        return resultList.stream().sorted(Comparator.comparing(MainPageMedalImageDTO::getCreateTime)
                .reversed())
                .map(MainPageMedalImageDTO::getDisplayImgUrl).limit(3).collect(Collectors.toList());

    }

    @Override
    public int getUserMedalNum(Long userId) {
        MyMedalVO myMedal = getMyMedal(userId);
        return Objects.nonNull(myMedal) ? myMedal.getMedalNum() : 0;
    }

    private KeyGenerator builderUserMedal(Long userId) {
        return RedisConfig.USER_MEDAL_NUM.copy().appendKey(userId % 10);
    }


    /**
     * 转换为SingleAchievementMedalBaseVO 对象
     *
     * @param medalId     勋章id
     * @param medalStatus 勋章状态
     * @return 转化对象
     */
    private SingleAchievementMedalBaseVO convert2SingleAchievementMedalBaseVO(Long medalId,
                                                                              Integer medalStatus) {
        SingleAchievementMedalBaseVO resultItem = new SingleAchievementMedalBaseVO();

        Map<Long, UserMedalInfoEntity> medalId2InfoMap = achievementMedalCacheManager.getMedalId2InfoMap();

        UserMedalInfoEntity userMedalInfoEntity = medalId2InfoMap.get(medalId);

        resultItem.setMedalId(medalId);
        resultItem.setMedalParentId(userMedalInfoEntity.getParentId());
        resultItem.setLevel(userMedalInfoEntity.getLevel());
        resultItem.setMedalImgUrl(UserAchievementMedalStatusEnum.UN_LOCK.getCode().equals(medalStatus)
                ? userMedalInfoEntity.getLockImgUrl() : userMedalInfoEntity.getActiveImgUrl());
        resultItem.setMedalName(userMedalInfoEntity.getName());
        resultItem.setStatus(medalStatus);
        resultItem.setTOrder(userMedalInfoEntity.getTOrder());

        return resultItem;
    }


}