package com.bxm.egg.user.token.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.jwt.bo.JwtTokenBO;
import com.bxm.component.jwt.builder.TokenBuildParam;
import com.bxm.component.jwt.util.JwtUtil;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.enums.RenewTokenSceneEnum;
import com.bxm.egg.user.facade.bo.UserTokenBO;
import com.bxm.egg.user.model.dto.token.TokenInfoDTO;
import com.bxm.egg.user.model.param.token.RenewTokenParam;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.egg.user.token.TokenService;
import com.bxm.newidea.component.param.BasicParam;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.utils.SystemClock;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.Objects;

import static com.bxm.egg.user.constant.UserConstant.USER_ID_KEY;

/**
 * TOKEN刷新实现类
 *
 * @author wzy
 * @date 2021年09月14日15:38:24
 **/
@Service
@Slf4j
@AllArgsConstructor
public class TokenServiceImpl implements TokenService {

    private RedisStringAdapter redisStringAdapter;

    private UserProperties userProperties;

    private DistributedLock distributedLock;

    @Override
    public TokenInfoDTO renew(RenewTokenParam param) {
        Preconditions.checkNotNull(param.getUserId());
        Preconditions.checkNotNull(param.getRefreshToken());
        String requestRefreshToken = param.getRefreshToken();


        String secretKey = userProperties.getAppTokenSecret();
        JwtTokenBO jwtTokenBO = JwtUtil.parseToken(requestRefreshToken, secretKey);

        // 判断token中的用户ID是否与ID一致
        if (userProperties.getEnableCheckMatch()) {
            String userIdFormToken = jwtTokenBO.getBodyWithString(USER_ID_KEY);
            if (!Objects.equals(userIdFormToken, param.getUserId().toString())) {
                log.error("[{}]使用的refreshToken与用户ID不匹配", param.getUserId());
                return null;
            }
        }

        long expiredTime = jwtTokenBO.getExpirationDate().getTime();
        if (expiredTime < SystemClock.now()) {
            log.info("[{}]refreshToken已过期", param.getUserId());
            return null;
        }

        // 如果启用临界值判断，一定时间内的refreshToken将不再刷新
        if (userProperties.isEnableRefreshTokenCritical()) {
            long diffSeconds = (expiredTime - SystemClock.now()) / 1000;
            long pastSeconds = userProperties.getRefreshTokenExpiredSeconds() - diffSeconds;

            if (pastSeconds < userProperties.getRefreshRebuildCriticalSeconds()) {
                if (log.isDebugEnabled()) {
                    log.debug("用户[{}]活跃时间较短，继续沿用RefreshToken", param.getUserId());
                }
                return TokenInfoDTO.builder()
                        .refreshToken(param.getRefreshToken())
                        .accessToken(doCreateAccessToken(param.getUserId(), secretKey))
                        .build();
            }
        }

        // 获取用户对应的refreshToken
        String cacheRefreshToken = getCacheRefreshToken(param.getUserId());

        if (log.isDebugEnabled()) {
            log.debug("重新请求accessToken，参数：{}, 场景：{}", JSON.toJSONString(param), param.getScene());
        }

        if (null == cacheRefreshToken) {
            log.info("[{}] 缓存中不存在对应的RefreshToken, 场景：{}", param.getUserId(), param.getScene());
            return null;
        }

        if (!requestRefreshToken.equals(cacheRefreshToken)) {
            log.warn("请求提供的RefreshToken与缓存不一致，uid：{},except：{},actual：{},场景：{}",
                    param.getUserId(),
                    cacheRefreshToken,
                    param.getRefreshToken(),
                    param.getScene());
            return null;
        }

        return create(param.getUserId(), param, param.getScene());
    }

    private String getCacheRefreshToken(Long userId) {
        KeyGenerator key = RedisConfig.USER_REFRESH_TOKEN_CACHE_KEY.copy().appendKey(userId);
        return redisStringAdapter.get(key, String.class);
    }

    @Override
    public TokenInfoDTO create(Long userId, BasicParam param, String scene) {
        String secretKey = userProperties.getAppTokenSecret();

        TokenInfoDTO tokenInfoDTO = TokenInfoDTO.builder()
                .refreshToken(createRefreshToken(userId, secretKey))
                .accessToken(doCreateAccessToken(userId, secretKey))
                .build();

        if (log.isDebugEnabled()) {
            log.debug("创建用户Token信息：{}, 场景:{}，用户id:{}", tokenInfoDTO, scene, userId);
        }

        return tokenInfoDTO;
    }

    /**
     * 创建访问的token，默认30分钟过期
     *
     * @param userId 用户ID
     * @param key    系统对应的加密密钥
     * @return 访问token，短期的凭证
     */
    private String doCreateAccessToken(Long userId, String key) {
        Date accessTokenExpiredDate = DateUtils.addField(new Date(), Calendar.SECOND, userProperties.getAccessTokenExpiredSeconds());

        return JwtUtil.generateToken(TokenBuildParam.create()
                .putClaim(USER_ID_KEY, userId.toString())
                .setIssueDate(new Date())
                .setExpirationTime(accessTokenExpiredDate)
                .setSignatureKey(key));
    }

    @Override
    public UserTokenBO renewToken(Long userId) {
        RenewTokenParam param = new RenewTokenParam();

        String cacheRefreshToken = getCacheRefreshToken(userId);
        param.setUserId(userId);
        param.setRefreshToken(cacheRefreshToken);
        param.setScene(RenewTokenSceneEnum.GATEWAY_RENEW_TOKEN.name());

        TokenInfoDTO tokenInfoDTO = renew(param);

        if (tokenInfoDTO == null) {
            return null;
        }

        UserTokenBO userTokenBO = new UserTokenBO();

        userTokenBO.setReNewAccessToken(tokenInfoDTO.getAccessToken());
        userTokenBO.setReNewRefreshToken(tokenInfoDTO.getRefreshToken());

        return userTokenBO;
    }

    /**
     * 创建用于刷新accessToken的RefreshToken，默认提供一个较长的过期时间
     *
     * @param userId 用户ID
     * @param key    用户对应的应用的加密密钥
     * @return 刷新token
     */
    private String createRefreshToken(Long userId, String key) {
        Date refreshTokenExpiredDate = DateUtils.addField(new Date(), Calendar.SECOND,
                userProperties.getRefreshTokenExpiredSeconds());

        String refreshToken = JwtUtil.generateToken(TokenBuildParam.create()
                .putClaim(USER_ID_KEY, userId.toString())
                .setIssueDate(new Date())
                .setExpirationTime(refreshTokenExpiredDate)
                .setSignatureKey(key));

        // RefreshToken需要保存到缓存，后续用于校验
        KeyGenerator cacheKey = RedisConfig.USER_REFRESH_TOKEN_CACHE_KEY.copy().appendKey(userId);
        redisStringAdapter.set(cacheKey, refreshToken, userProperties.getRefreshTokenExpiredSeconds());

        return refreshToken;
    }
}
