package com.bxm.egg.user.attribute.impl;

import com.bxm.egg.user.attribute.UserVisitService;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.mapper.UserVisitMapper;
import com.bxm.egg.user.mapper.VirtualUserMapper;
import com.bxm.egg.user.model.dto.UserVisitDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.UserVisitParam;
import com.bxm.egg.user.model.vo.UserVisit;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.bxm.newidea.component.tools.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

@Slf4j
@Service
public class UserVisitServiceImpl implements UserVisitService {

    private static final String USER_VISIT = "T_USER_VISIT_";

    private VirtualUserMapper userMapper;

    private final UserVisitMapper userVisitMapper;


    private final DistributedLock distributedLock;

    private final RedisStringAdapter redisStringAdapter;

    private final RedisSetAdapter redisSetAdapter;

    private final UserInfoService userInfoService;

    @Autowired(required = false)
    public UserVisitServiceImpl(VirtualUserMapper userMapper,
                                UserVisitMapper userVisitMapper,
                                DistributedLock distributedLock,
                                RedisStringAdapter redisStringAdapter,
                                RedisSetAdapter redisSetAdapter,
                                UserInfoService userInfoService) {
        this.userMapper = userMapper;
        this.userVisitMapper = userVisitMapper;
        this.distributedLock = distributedLock;
        this.redisStringAdapter = redisStringAdapter;
        this.redisSetAdapter = redisSetAdapter;
        this.userInfoService = userInfoService;
    }

    @Override
    public List<UserVisitDTO> listUserVisit(UserVisitParam userVisitParam) {

        int start = 0;
        if (userVisitParam.getPageNum() > 0) {
            start = (userVisitParam.getPageNum() - 1) * userVisitParam.getPageSize();
        }
        //查询原始数据
        List<UserVisit> userVisitList = userVisitMapper
                .queryPageByVisitDate(start, userVisitParam.getPageSize(), userVisitParam.getUserId());

        //将原始数据转为dto
        List<UserVisitDTO.VisitUserInfo> userVisitInfoList = userVisitList.parallelStream().map(this::convertUserVisit).collect(toList());

        //按日期分组（分组内的访客倒序排）
        TreeMap<LocalDate, List<UserVisitDTO.VisitUserInfo>> userVisitMap = userVisitInfoList.stream()
                .sorted(Comparator.comparing(UserVisitDTO.VisitUserInfo::getVisitTime).reversed())
                .collect(
                        Collectors.groupingBy(UserVisitDTO.VisitUserInfo::getVisitDate, () -> new TreeMap<>(Comparator.reverseOrder()), toList()));

        //分组后遍历组装返回数据
        List<UserVisitDTO> userVisitDtoList = new ArrayList<>();
        userVisitMap.forEach((k, v) -> {
            UserVisitDTO userVisitDTO = new UserVisitDTO();
            userVisitDTO.setVisitDate(k);
            userVisitDTO.setVisitUserInfoList(v);
            userVisitDtoList.add(userVisitDTO);

        });
        //对日期倒序排
        List<UserVisitDTO> result = userVisitDtoList.stream().sorted(Comparator.comparing(UserVisitDTO::getVisitDate).reversed()).collect(toList());
        redisSetAdapter.remove(RedisConfig.USER_LATELY_VISIT_NUM.copy().appendKey(userVisitParam.getUserId()));
        return result;
    }

    @Async
    @Override
    public void visit(Long userId, Long targetUserId) {
        if (null == userId || null == targetUserId || userId.equals(targetUserId)) {
            return;
        }

        UserInfoEntity userInfo = userInfoService.selectAllUserById(userId);
        UserInfoEntity targetUserInfo = userInfoService.selectAllUserById(targetUserId);

        if (null == userInfo || null == targetUserInfo) {
            return;
        }

        UserVisit userVisit = new UserVisit();
        userVisit.setUserId(targetUserId);
        userVisit.setUserNickname(targetUserInfo.getNickname());
        userVisit.setUserHeadImg(targetUserInfo.getHeadImg());
        userVisit.setUserSex(targetUserInfo.getSex());
        userVisit.setVisitUserId(userId);
        userVisit.setVisitUserNickname(userInfo.getNickname());
        userVisit.setVisitUserHeadImg(userInfo.getHeadImg());
        userVisit.setVisitUserSex(userInfo.getSex());

        userVisit.setVisitDate(LocalDate.now());
        userVisit.setVisitTime(LocalDateTime.now());

        userVisit.setIsVest(userInfo.getState() == 3 ? (byte) 1 : (byte) 0);

        String lockKey = USER_VISIT + userId + "_" + targetUserId;
        if (distributedLock.lock(lockKey)) {
            int update = userVisitMapper.update(userVisit);
            if (update == 0) {
                userVisitMapper.insert(userVisit);
                this.increaseTodayVisit(targetUserId);
            }
            this.increaseLatelyVisit(targetUserId, userId);
            distributedLock.unlock(lockKey);
        }
    }

    private void increaseLatelyVisit(Long targetUserId, Long userId) {
        redisSetAdapter.add(RedisConfig.USER_LATELY_VISIT_NUM.copy().appendKey(targetUserId), userId);
    }

    /**
     * 每当最近访客增加时，放入redis中
     */
    private void increaseTodayVisit(Long targetUserId) {
        redisStringAdapter.incrementWithDefault(RedisConfig.USER_VISIT_NUM.copy().appendKey("visit").appendKey(targetUserId), 1, 1,
                DateUtils.getCurSeconds());
    }

    @Override
    public void virtualVisit(Long userId, Integer num) {
        List<Long> virtualUserIdList = userMapper.getRandomVirtualUserIdList(num);
        virtualUserIdList.forEach(e -> visit(e, userId));
    }

    @Override
    public List<String> listRecentVisitorHeadImg(Long userId) {
        List<UserVisit> userVisitList = userVisitMapper.selectLastVisitorHeadImg(userId, 3);
        if (!CollectionUtils.isEmpty(userVisitList)) {
            Collections.reverse(userVisitList);
        }
        return userVisitList.stream().map(UserVisit::getVisitUserHeadImg).collect(toList());
    }

    @Override
    public Integer getTodayVisitorNum(Long userId) {
        return redisStringAdapter.getInt(RedisConfig.USER_VISIT_NUM.copy().appendKey("visit").appendKey(userId));
    }

    @Override
    public void doTriggerUpdateInfo(Long userId, String nickname, String headImg, Byte sex) {
        int tableTotal = 10;
        CountDownLatch countDownLatch = new CountDownLatch(tableTotal);

        //由于数量比较固定且数量不是很大，所以使用SynchronousQueue队列来执行任务
        ExecutorService executor = new ThreadPoolExecutor(0,
                tableTotal,
                60L,
                TimeUnit.SECONDS,
                new SynchronousQueue<>(),
                new NamedThreadFactory("thread-trigger_update_user-"));

        //构建10个任务
        for (int i = 0; i < tableTotal; i++) {
            int finalI = i;
            executor.execute(() -> {
                log.debug("user update thread:{} begin work", Thread.currentThread().getId());
                try {
                    this.userVisitMapper.updateUserInfo(USER_VISIT + finalI, userId, nickname, headImg, sex);
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                } finally {
                    countDownLatch.countDown();
                }
                log.debug("user update thread:" + Thread.currentThread().getId() + "end work");
            });
        }

        try {
            //等待所有任务执行完
            countDownLatch.await();
            log.debug("用户冗余信息同步所有线程执行结束");
        } catch (InterruptedException e) {
            log.error("用户冗余信息同步多线程执行出现错误:", e);
            Thread.currentThread().interrupt();
        }

        executor.shutdown();
    }

    @Override
    public Integer getInterviewNumByUserId(Long userId) {
        return userVisitMapper.selectCountByUserId(userId);
    }

    @Override
    public Integer getInterviewNumByTime(Long userId, Date date) {
        return userVisitMapper.selectCountByTime(userId, date);
    }

    /**
     * convert userVisit
     */
    private UserVisitDTO.VisitUserInfo convertUserVisit(UserVisit userVisit) {
        UserVisitDTO.VisitUserInfo userVisitDTO = new UserVisitDTO.VisitUserInfo();
        BeanUtils.copyProperties(userVisit, userVisitDTO);
        return userVisitDTO;
    }

}
