package com.bxm.egg.user.location.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.bxm.egg.facade.dto.LocationDetailDTO;
import com.bxm.egg.user.constant.LogicGroup;
import com.bxm.egg.user.constant.SwitchType;
import com.bxm.egg.user.constant.UserBizConfigProperties;
import com.bxm.egg.user.constant.UserLocationTypeEnum;
import com.bxm.egg.user.dto.LocationDTO;
import com.bxm.egg.user.integration.LocationIntegrationService;
import com.bxm.egg.user.location.UserLocationService;
import com.bxm.egg.user.location.context.LocationSwitchContext;
import com.bxm.egg.user.mapper.UserLocationMapper;
import com.bxm.egg.user.model.dto.location.AreaInfoDTO;
import com.bxm.egg.user.model.dto.location.LocationSwitchDTO;
import com.bxm.egg.user.model.entity.UserLocationEntity;
import com.bxm.egg.user.model.param.UserHomeTownParam;
import com.bxm.egg.user.model.param.location.LocationSwitchParam;
import com.bxm.newidea.component.rule.RuleGroupExecutor;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

import static com.bxm.newidea.component.JSON.toJSONString;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/9/17 11:12 上午
 */
@Slf4j
@Service
@AllArgsConstructor
public class UserLocationServiceImpl implements UserLocationService {

    private final UserLocationMapper userLocationMapper;

    private final RuleGroupExecutor ruleGroupExecutor;

    private final UserBizConfigProperties userBizConfigProperties;

    private final LocationIntegrationService locationIntegrationService;

    @Override
    public UserLocationEntity getUserLocationByUserId(Long userId) {
        //查询资料信息
        LambdaQueryWrapper<UserLocationEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(UserLocationEntity::getUserId, userId);

        UserLocationEntity userLocationEntity = userLocationMapper.selectOne(queryWrapper);

        if (userLocationEntity == null) {
            return buildValidUserLocation(userId);
        }

        return userLocationEntity;
    }

    private UserLocationEntity buildValidUserLocation(Long userId) {
        UserLocationEntity userLocationEntity = new UserLocationEntity();
        userLocationEntity.setId(0L);
        userLocationEntity.setUserId(userId);

        userLocationEntity.setCreateTime(LocalDateTime.now());
        userLocationEntity.setModifyTime(LocalDateTime.now());

        return userLocationEntity;
    }


    @Override
    public int updateUserHometownInfo(UserHomeTownParam userHomeTownParam) {
        UserLocationEntity userLocationEntity = new UserLocationEntity();

        userLocationEntity.setHometownCode(userHomeTownParam.getHometownCode());
        userLocationEntity.setHometownName(userHomeTownParam.getHometownName());

        LambdaQueryWrapper<UserLocationEntity> updateQueryWrapper = new LambdaQueryWrapper<>();

        updateQueryWrapper.eq(UserLocationEntity::getUserId, userHomeTownParam.getUserId());

        return userLocationMapper.update(userLocationEntity, updateQueryWrapper);
    }

    @Override
    public int updateUserLocationInfo(Long userId,
                                      String locationCode,
                                      String locationName,
                                      UserLocationTypeEnum locationType) {
        UserLocationEntity userLocationEntity = new UserLocationEntity();

        userLocationEntity.setLocationCode(locationCode);
        userLocationEntity.setLocationName(locationName);
        userLocationEntity.setUserId(userId);
        userLocationEntity.setLocationType(locationType.getCode());

        LambdaQueryWrapper<UserLocationEntity> updateQueryWrapper = new LambdaQueryWrapper<>();

        updateQueryWrapper.eq(UserLocationEntity::getUserId, userId);

        return userLocationMapper.update(userLocationEntity, updateQueryWrapper);
    }

    @Override
    public int updateUserLocationInfo(Long userId, UserLocationEntity userLocationEntity) {

        LambdaQueryWrapper<UserLocationEntity> updateQueryWrapper = new LambdaQueryWrapper<>();

        updateQueryWrapper.eq(UserLocationEntity::getUserId, userId);

        return userLocationMapper.update(userLocationEntity, updateQueryWrapper);
    }

    @Override
    public LocationSwitchDTO locationSwitch(LocationSwitchParam param) {
        if (log.isDebugEnabled()) {
            log.debug("用户: {} 获取定位信息的请求参数: {}", param.getUserId(), toJSONString(param));
        }

        LocationSwitchContext context = new LocationSwitchContext();
        context.setParam(param);

        // 如果匹配就返回false，不匹配返回true继续处理
        ruleGroupExecutor.apply(LogicGroup.LOCATION_SWITCH_RULE, context);

        String areaCode = context.getFinalAreaCode();
        if (areaCode == null) {
            areaCode = userBizConfigProperties.getDefaultLocation();
        }
        Byte finalSwitchType = context.getFinalSwitchType();
        if (finalSwitchType == null) {
            finalSwitchType = SwitchType.FORCE_SWITCH;
        }

        LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(areaCode);

        AreaInfoDTO areaInfo = new AreaInfoDTO();
        areaInfo.setCode(areaCode);
        areaInfo.setName(locationDTO.getName());

        LocationSwitchDTO switchDTO = new LocationSwitchDTO();
        switchDTO.setSwitchType(finalSwitchType);
        switchDTO.setAreaInfo(areaInfo);

        log.info("用户: {} 请求参数: {} 最终获取到的定位信息为: {},定位方式：{}", param.getUserId(),
                toJSONString(param),
                toJSONString(switchDTO),
                context.getMatchRuleClass() == null ? "" : context.getMatchRuleClass().getSimpleName());
        return switchDTO;
    }
}