package com.bxm.egg.user.login.listener;

import com.alibaba.fastjson.JSON;
import com.bxm.component.bus.annotations.EventBusSubscriber;
import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.convert.EggSixParamConverter;
import com.bxm.egg.user.facade.enums.WarmRuleEnum;
import com.bxm.egg.user.facade.param.UserWarmActionParam;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserWechatGrantService;
import com.bxm.egg.user.integration.EggSixEnjoyIntegrationService;
import com.bxm.egg.user.login.UserLogService;
import com.bxm.egg.user.login.event.LoginEvent;
import com.bxm.egg.user.model.dto.login.LoginResultDTO;
import com.bxm.egg.user.model.entity.UserGrantWechatAppEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.UserBindWxParam;
import com.bxm.egg.user.model.param.WechatBindMobileOneKeyLoginParam;
import com.bxm.egg.user.model.param.login.ChuangLanLoginParam;
import com.bxm.egg.user.model.param.login.LoginParam;
import com.bxm.egg.user.model.param.login.UserSettingsPasswordLoginParam;
import com.bxm.egg.user.model.param.login.WechatBindMobileLoginParam;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.egg.user.warmlevel.impl.WarmHandlerProxy;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/10/19 9:24 下午
 */
@Slf4j
@Component
public class LoginListener {

    @Resource
    private UserLogService userLogService;

    @Resource
    private WarmHandlerProxy warmHandlerProxy;

    @Resource
    private UserProperties userProperties;

    @Resource
    private UserInfoService userInfoService;

    @Resource
    private UserWechatGrantService userWechatGrantService;

    @Resource
    private UserExtendService userExtendService;

    @Resource
    private EggSixEnjoyIntegrationService eggSixEnjoyIntegrationService;

    @EventBusSubscriber
    public void userLoginAfterEvent(LoginEvent event) {

        log.info("监听到用户登录事件，事件内容: {}", JSON.toJSONString(event));

        //记录登录日志
        userLogService.recordLoginLog(event.getLoginParam(), event.getLoginResult(), event.getLoginType());
        //更新用户设备信息
        updateUserDevcId(event);
        //登录发放温暖
        UserWarmActionParam userWarmActionParam = UserWarmActionParam.builder()
                .userId(event.getUserId())
                .warmRuleEnum(WarmRuleEnum.LOGIN)
                .warmValue(userProperties.getLoginWarmValue())
                .build();
        warmHandlerProxy.handle(userWarmActionParam);

        switch (event.getLoginType()) {
            case SETTINGS_PASSWORD_LOGIN:
                setPasswordLogin(event);
                break;
            case WECHAT_BIND_MOBILE_LOGIN:
                wxBindPhoneLogin(event);
                break;
            case WECHAT_BIND_MOBILE_ONE_KEY_LOGIN:
                wxOneKeyLogin(event);
                break;
            default:
                oneKeyLogin(event);
                break;
        }
    }

    private void updateUserDevcId(LoginEvent event) {
        if (Objects.nonNull(event.getLoginParam().getDevcId())) {
            UserInfoEntity userInfoEntity = new UserInfoEntity();
            userInfoEntity.setId(event.getUserId());
            userInfoEntity.setEquipmentId(event.getLoginParam().getDevcId());
            userInfoService.updateUserInfo(userInfoEntity);
        }
    }

    private void setPasswordLogin(LoginEvent event) {
        UserSettingsPasswordLoginParam loginParam = (UserSettingsPasswordLoginParam) event.getLoginParam();
        //登录成功后，修改密码
        userInfoService.updateUserPassword(loginParam.getUserId(), loginParam.getPassword());
    }

    private void oneKeyLogin(LoginEvent event) {
        LoginResultDTO loginResult = event.getLoginResult();
        LoginParam loginParam = event.getLoginParam();
        UserInfoEntity userInfoEntity = userInfoService.selectAllUserById(loginResult.getUserId());
        //异步处理，可能新用户还没插入到数据库，如果是新用户
        //一键登录，入参肯定是没微信信息的，无需处理
        if (Objects.isNull(userInfoEntity)) {
            return;
        }
        UserGrantWechatAppEntity userWxInfoByUserId = userWechatGrantService.getUserWxInfoByUserId(loginResult.getUserId());

        //如果一键登录存在用户，则说明肯定是注册成功了
        //没微信信息的，无需处理
        if (Objects.isNull(userWxInfoByUserId)) {
            return;
        }

        //异步补偿用户少发的奖励
        UserBindWxParam userBindWxParam = new UserBindWxParam();
        userBindWxParam.setUserId(loginResult.getUserId());
        userBindWxParam.setOpenId(userWxInfoByUserId.getOpenId());
        userBindWxParam.setUnionId(userWxInfoByUserId.getUnionId());
        userBindWxParam.setNickName(userWxInfoByUserId.getNickName());
        userBindWxParam.setDevcId(loginParam.getDevcId());
        userBindWxParam.setHeadImageUrl(userWxInfoByUserId.getHeadImg());
        userBindWxParam.setPhone(loginParam.getPhone());
        userWechatGrantService.userBindWxHandler(userBindWxParam);
    }

    private void wxBindPhoneLogin(LoginEvent event) {
        LoginResultDTO resultDTO = event.getLoginResult();
        WechatBindMobileLoginParam loginParam = (WechatBindMobileLoginParam) event.getLoginParam();
        //理论登录绑定手机号只会执行一次，所以放到这个方法里面也可以
        //登录后绑定手机号
        userInfoService.bindPhone(resultDTO.getUserId(), loginParam.getPhone());

        //绑定手机号为用户名
        userInfoService.bindUsername(resultDTO.getUserId(), loginParam.getPhone());

        //修改绑定用户
        userWechatGrantService.updateWechatGrantBindUserId(resultDTO.getUserId(), loginParam.getUnionId());

        //微信登录绑定手机号也要初始化一下资料完成度
        userExtendService.initComplete(resultDTO.getUserId());


        if (StringUtils.isGrateOrEqualThan(loginParam.getCurVer(), "2.0.1")) {
            UserBindWxParam userBindWxParam = new UserBindWxParam();
            userBindWxParam.setUserId(resultDTO.getUserId());
            userBindWxParam.setOpenId(loginParam.getOpenId());
            userBindWxParam.setUnionId(loginParam.getUnionId());
            userBindWxParam.setNickName(loginParam.getNickName());
            userBindWxParam.setDevcId(loginParam.getDevcId());
            userBindWxParam.setHeadImageUrl(loginParam.getHeadImageUrl());
            userBindWxParam.setPhone(loginParam.getPhone());
            userWechatGrantService.userBindWxHandler(userBindWxParam);
        } else {
            //调用六享微信绑定奖励
            eggSixEnjoyIntegrationService.bind(EggSixParamConverter.convert(loginParam, resultDTO.getUserId()));
        }
    }

    private void wxOneKeyLogin(LoginEvent event) {

        LoginResultDTO resultDTO = event.getLoginResult();
        WechatBindMobileOneKeyLoginParam loginParam = (WechatBindMobileOneKeyLoginParam) event.getLoginParam();
        //理论登录绑定手机号只会执行一次，所以放到这个方法里面也可以
        //登录后绑定手机号
        userInfoService.bindPhone(resultDTO.getUserId(), loginParam.getPhone());
        //绑定手机号为用户名
        userInfoService.bindUsername(resultDTO.getUserId(), loginParam.getPhone());
        //修改绑定用户
        userWechatGrantService.updateWechatGrantBindUserId(resultDTO.getUserId(), loginParam.getUnionId());
        //微信登录绑定手机号也要初始化一下资料完成度
        userExtendService.initComplete(resultDTO.getUserId());

        if (StringUtils.isGrateOrEqualThan(loginParam.getCurVer(), "2.0.1")) {
            UserBindWxParam userBindWxParam = new UserBindWxParam();
            userBindWxParam.setPhone(loginParam.getPhone());
            userBindWxParam.setUserId(resultDTO.getUserId());
            userBindWxParam.setOpenId(loginParam.getOpenId());
            userBindWxParam.setUnionId(loginParam.getUnionId());
            userBindWxParam.setNickName(loginParam.getNickName());
            userBindWxParam.setDevcId(loginParam.getDevcId());
            userBindWxParam.setHeadImageUrl(loginParam.getHeadImageUrl());
            userWechatGrantService.userBindWxHandler(userBindWxParam);
        } else {
            //调用六享微信绑定奖励
            eggSixEnjoyIntegrationService.bind(EggSixParamConverter.convert(loginParam, resultDTO.getUserId()));
        }
    }

}