package com.bxm.egg.user.warmlevel.impl.handler;

import com.bxm.egg.user.mapper.warmlevel.UserWarmValueFlowMapper;
import com.bxm.egg.user.model.vo.warmlevel.UserWarmValueFlow;
import com.bxm.egg.user.model.vo.warmlevel.WarmRuleDetail;
import com.bxm.egg.user.warmlevel.UserWarmLevelService;
import com.bxm.egg.user.warmlevel.impl.context.WarmActionContext;
import com.bxm.egg.user.warmlevel.impl.rule.WarmRule;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 抽象的温暖值处理类，对一些公共的行为进行抽象
 *
 * @author lowi
 */
public abstract class AbstractWarmActionHandler implements WarmActionHandler {
    @Autowired
    private UserWarmValueFlowMapper userWarmValueFlowMapper;

    @Autowired
    private SequenceCreater sequenceCreate;

    protected UserWarmLevelService userWarmLevelService;

    private Map<Class<? extends WarmRule>, WarmRule> ruleMap;


    @Autowired
    public void setRules(List<WarmRule> rules) {
        if (CollectionUtils.isNotEmpty(rules)) {
            ruleMap = Maps.newHashMap();
            rules.forEach(rule -> ruleMap.put(rule.getClass(), rule));
        }
    }

    @Override
    public final Message handle(WarmActionContext context) {
        if (null == userWarmLevelService) {
            userWarmLevelService = SpringContextHolder.getBean(UserWarmLevelService.class);
        }
        //前置处理，主要用来做校验
        Message message = preAction(context);
        if (!message.isSuccess()) {
            return message;
        }
        //处理中，进行逻辑处理，数据入库，流水记录等
        postAction(context);
        return Message.build(true);
    }

    /**
     * 前置处理，执行绑定的验证规则，执行通过后调用自定义回调
     *
     * @param context 上下文信息
     */
    private Message preAction(WarmActionContext context) {
        Class<? extends WarmRule> bindRules = bindRules();
        if (bindRules != null) {
            WarmRule rule = ruleMap.get(bindRules);
            Message message = rule.apply(context);
            if (!message.isSuccess()) {
                return message;
            }
        }
        //规则全部校验通过后执行自定义回调
        context.getCallback().preAction(context);
        return Message.build(true);
    }

    private void postAction(WarmActionContext context) {
        //执行具体的处理逻辑，负责将入库账号中的字段进行变更处理
        execAction(context);

        //记录流水信息
        saveFlow(context);

        //数据入库（更新温暖值）
        syncUserWarm(context);

        //处理完成后触发回调
        context.getCallback().postAction(context);

        // 后续动作，redis更新之类的
        afterPost(context);

        //清理用户温暖值缓存触发升级之类的后续操作
        delWarmCacheWarmLevelUp(context);
    }

    /**
     * 温暖值增加的 后续动作
     *
     * @param context 上下文
     */
    protected abstract void afterPost(WarmActionContext context);


    /**
     * 执行温暖值具体逻辑处理
     *
     * @param context 上下文信息
     */
    protected abstract void execAction(WarmActionContext context);

    /**
     * 同步温暖值信息
     *
     * @param context 上下文
     */
    private void syncUserWarm(WarmActionContext context) {
        //更新用户的温暖值
        userWarmLevelService.updateUserWarmInfo(context.getUserId(), context.getWarmRuleDetail().getWarmValue());
    }


    /**
     * 保存操作产生的流水信息
     *
     * @param context 上下文信息
     */
    private void saveFlow(WarmActionContext context) {
        UserWarmValueFlow userWarmValueFlow = new UserWarmValueFlow();
        userWarmValueFlow.setId(sequenceCreate.nextLongId());
        userWarmValueFlow.setCreateTime(new Date());
        userWarmValueFlow.setRemark(context.getOriginParam().getRemake());
        userWarmValueFlow.setType(context.getWarmRuleDetail().getType());
        userWarmValueFlow.setTypeDesc(context.getWarmRuleDetail().getTypeDesc());
        userWarmValueFlow.setUserId(context.getUserId());
        userWarmValueFlow.setWarmValue(context.getWarmRuleDetail().getWarmValue());
        userWarmValueFlowMapper.insertSelective(userWarmValueFlow);
    }

    /**
     * 获取温暖值规则
     *
     * @param type 规则类型
     * @return 温暖值规则
     */
    WarmRuleDetail getWarmRule(String type) {
        return userWarmLevelService.getWarmRule(type);
    }

    /**
     * 清理用户温暖值缓存触发升级之类的后续操作
     *
     * @param context 上下文
     */
    private void delWarmCacheWarmLevelUp(WarmActionContext context) {
        userWarmLevelService.delWarmCacheWarmLevelUp(context);
    }
}
