package com.bxm.localnews.activity.record.convert;

import com.bxm.localnews.activity.record.utils.RecordUtils;
import com.bxm.localnews.activity.vo.RecordInfoBean;
import com.bxm.newidea.component.tools.FileUtils;
import com.google.common.collect.Maps;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import java.io.File;
import java.util.Map;

/**
 * 录音文件类型转换处理器，支持将不同的类型录音转换为baidu aip支持的格式
 * 目前支持的格式有：pcm(推荐)/wav/mp3
 * @author liujia
 */
@Component
public class ConvertProcesser implements ApplicationContextAware, InitializingBean {

    private ApplicationContext applicationContext;

    private Map<String, IConverter> converterMap = Maps.newHashMap();

    /**
     * 将录音文件进行格式转换
     * @param sourceFile 原始路径文件
     * @param duration   客户端提供的录音时长，如果不为-1，则采用传递的值
     * @return 录音文件信息
     */
    public RecordInfoBean convert(File sourceFile, int duration) {
        RecordInfoBean recordInfo = new RecordInfoBean();

        String fileExtend = FileUtils.getFileExtension(sourceFile);

        IConverter converter = converterMap.get(fileExtend);
        File afterFile = converter.convert(sourceFile);

        recordInfo.setConvertExtend(converter.after());
        recordInfo.setConvertFile(afterFile);
        recordInfo.setSourceFile(sourceFile);
        recordInfo.setDuration(getRecordDuration(afterFile, duration));

        return recordInfo;
    }

    /**
     * 获取录音文件时长
     * @param recordFile 录音文件
     * @return 录音文件时长
     */
    private int getRecordDuration(File recordFile, int duration) {
        if (duration != -1) {
            return duration;
        }
        return RecordUtils.getDuration(recordFile);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void afterPropertiesSet() {
        Map<String, IConverter> beanMap = applicationContext.getBeansOfType(IConverter.class);
        for (Map.Entry<String, IConverter> bean : beanMap.entrySet()) {
            converterMap.put(bean.getValue().support(), bean.getValue());
        }
    }
}
