package com.bxm.localnews.activity.record.grade;

import com.bxm.localnews.activity.record.utils.RecordUtils;
import com.bxm.localnews.activity.vo.GradeResultBean;
import com.bxm.localnews.activity.vo.RecordGradeContext;
import com.bxm.newidea.component.tools.NumberUtils;
import com.bxm.newidea.component.tools.PinyinUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Maps;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * 评分处理器，运行设定的评分策略，进行不同比例的分支采纳
 * @author liujia
 */
@Service
@Lazy
public class GradeProcesser implements InitializingBean, ApplicationContextAware {

    private Collection<GradeStrategy> strategies;

    private Map<Integer, String> targetMap = Maps.newConcurrentMap();

    private Map<Integer, String> targetPinyinMap = Maps.newConcurrentMap();

    private Pattern pattern = Pattern.compile("[^0-9]");

    private ApplicationContext applicationContext;

    public GradeResultBean grade(RecordGradeContext context) {
        GradeResultBean result = new GradeResultBean();
        context.setGrade(result);
        //前置处理
        preProcess(context);

        for (GradeStrategy strategy : strategies) {
            strategy.grade(context);
        }
        result.setLength(context.getDuration());
        result.setOssUrl(context.getOssUrl());

        return result;
    }

    /**
     * 对上下文进行前置处理，方便后续处置
     * @param context 录音评分文件上下文
     */
    private void preProcess(RecordGradeContext context) {
        // 将原始口令进行转换
        String target = context.getTarget();

        int targetHashCode = target.hashCode();
        String convertTarget = targetMap.get(targetHashCode);
        if (null == convertTarget) {
            convertTarget = convertNum(target);
            targetMap.put(targetHashCode, convertTarget);
            //简化处理，只保留最近500条数据，可有优化为LRU方式
            if (targetMap.size() > 500) {
                targetMap.clear();
            }
        }
        context.setConverTarget(convertTarget);

        //将原始口令转换为拼音
        String convertPinyinTarget = targetPinyinMap.get(targetHashCode);
        if (null == convertPinyinTarget) {
            convertPinyinTarget = StringUtils.join(PinyinUtils.getPinyinArray(convertTarget), ",");
            targetPinyinMap.put(targetHashCode, convertPinyinTarget);
        }
        context.setPinyinTarget(convertPinyinTarget);

        //转换结果中去除特殊字符
        String result = replaceSpecialChar(context.getResult()).replace(" ", "");
        context.setResult(result);
        //将转换结果转换为拼音
        context.setPinyinResult(StringUtils.join(PinyinUtils.getPinyinArray(result), ","));
    }

    private String convertNum(String sourceStr) {
        //去除特殊字符
        String convertContent = replaceSpecialChar(sourceStr);
        //提取字符串中的数字(不支持小数处理)
        String[] numStrArray = pattern.matcher(convertContent).replaceAll(" ").split(" ");
        //将阿拉伯数字转换为中文并替换字符串
        for (String numStr : numStrArray) {
            if (StringUtils.isNotBlank(numStr)) {
                convertContent = convertContent.replaceFirst(numStr,
                        RecordUtils.formatDecimal(NumberUtils.parseToLong(numStr)));
            }
        }

        return convertContent.replace(" ", "");
    }

    private String replaceSpecialChar(String src) {
        String convertContent = src.replaceAll("[\\p{Punct}\\pP 　]", " ");
        convertContent = convertContent.replaceAll("\n", " ");
        return convertContent.replaceAll("\r", " ");
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        strategies = applicationContext.getBeansOfType(GradeStrategy.class).values();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
