package com.bxm.localnews.activity.service.impl;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import com.bxm.localnews.activity.domain.SignRecordMapper;
import com.bxm.localnews.activity.dto.CalendarSignDTO;
import com.bxm.localnews.activity.dto.SignDTO;
import com.bxm.localnews.activity.service.DailySignService;
import com.bxm.localnews.activity.service.SignConfigService;
import com.bxm.localnews.activity.vo.SignConfig;
import com.bxm.localnews.activity.vo.SignRecord;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.param.AccountGoldParam;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2019/4/4 11:30
 * @desc
 */
@Service
public class DailySignServiceImpl extends BaseService implements DailySignService {

    @Autowired
    private SignConfigService signConfigService;

    @Autowired
    private SignRecordMapper signRecordMapper;

    @Autowired
    private DistributedLock distributedLock;

    @Autowired
    private UserAccountIntegrationService userAccountIntegrationService;

    @Override
    public SignDTO signRecord(Long userId) {

        return generateSign(userId);
    }

    @Override
    public List<CalendarSignDTO> listSignRecord(Long userId) {
        Date now = new Date();
        String signDate = DateUtils.formatDate(now);
        //获取对应月份的上一个月的签到信息
        String beforeDate = DateUtils.formatDate(DateUtils.addField(DateUtils.parseDate(signDate), Calendar.MONTH, -1));
        List<CalendarSignDTO> beforeSignRecords = getMonthSignRecord(userId, beforeDate);

        //获取对应月份的签到信息
        List<CalendarSignDTO> signRecords = getMonthSignRecord(userId, signDate);
        beforeSignRecords.addAll(signRecords);
        return beforeSignRecords;
    }

    @Override
    public Message executeUserSign(Long userId) {
        Message message = Message.build();
        SignDTO signDTO = generateSign(userId);
        if (signDTO.isTodaySignState()) {
            return message.setSuccess(false).setMessage("用户已签到");
        }

        String requestId = String.valueOf(nextId());
        Long id = nextId();
        if (distributedLock.lock(userId + "" + id, requestId)) {
            SignRecord signRecord = new SignRecord(id, userId, new Date(), signDTO.getCount().intValue() + 1,
                    signDTO.getGold().longValue(), false);
            signRecordMapper.insertSelective(signRecord);
            //签到添加金币
            AccountGoldParam param = new AccountGoldParam(userId, "USABLE_GOLD", true,
                    signDTO.getGold(), id, "SIGN_DAILY");
            userAccountIntegrationService.addGold(param);
            distributedLock.unlock(userId + "" + id, requestId);
        } else {
            return message.setSuccess(false).setMessage("用户已签到");
        }
        return message;
    }

    @Override
    public Message executeUserFillSign(Long userId) {
        Message message = Message.build();
        SignDTO signDTO = generateSign(userId);
        if (!signDTO.isYesterdaySignState()) {
            return message.setSuccess(false).setMessage("当前用户已补签");
        }

        String requestId = String.valueOf(nextId());
        Long id = nextId();

        if (distributedLock.lock(userId + "" + id, requestId)) {
            Date yesterday = DateUtils.addField(new Date(), Calendar.DATE, -1);

            //获取前天的签到数据
            SignRecord beforeSignRecord = signRecordMapper.getBeforeYesterDaySignRecord(userId);

            //设置连续签到日期
            int signDay = 1;
            if (beforeSignRecord != null) {
                signDay += beforeSignRecord.getSignDay();
            }

            SignRecord signRecord = new SignRecord(id, userId, yesterday, signDay,
                    0L, true);
            signRecordMapper.insertSelective(signRecord);

            //判断今天是否签到---签到则更新连续签到的日期
            if (signDTO.isTodaySignState()) {
                signRecordMapper.updateTodaySign(userId, DateUtils.formatAtWill(new Date(), DateUtils.DATE_FORMAT),
                        signDay + 1);
            }

            distributedLock.unlock(userId + "" + id, requestId);
        } else {
            return message.setSuccess(false).setMessage("当前用户已补签");
        }

        return message;
    }

    /**
     * 获取对应连续签到天数的配置信息
     * 备注：连续签到超过7天---签到配置信息为第七天
     *
     * @param signDay
     * @return
     */
    private SignConfig getSignConfig(int signDay) {
        List<SignConfig> signConfigList = signConfigService.allSignConfig();

        if (signDay < 7) {
            for (SignConfig travSignConfig : signConfigList) {
                if (travSignConfig.getDate() == signDay) {
                    return travSignConfig;
                }
            }

        }

        return signConfigList.stream().filter(signConfig -> signConfig.getDate() == 7)
                .findFirst().get();
    }

    /**
     * 生成用户签到的相关信息
     *
     * @param userId
     * @return
     */
    private SignDTO generateSign(Long userId) {
        SignDTO signDTO = new SignDTO();
        SignRecord yesterdaySignRecord = signRecordMapper.getYesterdaySignRecord(userId);
        //昨天是否可以补签
        signDTO.setYesterdaySignState(yesterdaySignRecord == null ? true : false);

        //获取最后一天签到的信息
        SignRecord signRecord = signRecordMapper.getLastSignRecord(userId);

        //判断连续签到可获得的金币
        if (signRecord == null) {
            signDTO.setGold(getSignConfig(1).getReward().intValue());
            signDTO.setCount(0L);
            signDTO.setYesterdaySignState(true);
        } else {
            Date today = new Date();
            //判断今天是否签到
            if (org.apache.commons.lang3.time.DateUtils.isSameDay(signRecord.getSignDate(), today)) {
                signDTO.setTodaySignState(true);
                signDTO.setGold(getSignConfig(signRecord.getSignDay() + 1).getReward().intValue());
                signDTO.setCount(signRecord.getSignDay().longValue());
            } else {
                if (yesterdaySignRecord == null) {
                    signDTO.setGold(getSignConfig(1).getReward().intValue());
                    signDTO.setCount(0L);
                    signDTO.setYesterdaySignState(true);
                } else {
                    signDTO.setGold(getSignConfig(yesterdaySignRecord.getSignDay() + 1).getReward().intValue());
                    signDTO.setCount(yesterdaySignRecord.getSignDay().longValue());
                }
            }
        }

        return signDTO;
    }

    /**
     * 获取对应用户的签到日历
     *
     * @param userId
     * @param signDate
     * @return
     */
    private List<CalendarSignDTO> getMonthSignRecord(Long userId, String signDate) {

        //组装签到日历
        List<CalendarSignDTO> signRecords = generateMonthSign(signDate);
        //获取对应月份的签到信息
        List<SignRecord> signRecordList = signRecordMapper.getMonthSignRecord(userId, signDate);

        signRecords.forEach(calendarSignDTO -> {
            signRecordList.forEach(signRecord -> {
                String signRecordDate = DateUtils.formatAtWill(signRecord.getSignDate(), DateUtils.DATE_FORMAT);
                if (signRecordDate.equals(calendarSignDTO.getSignDate())) {
                    calendarSignDTO.setSignFlag(true);
                    calendarSignDTO.setFill(signRecord.getFillSign());
                }
            });
        });

        return signRecords;
    }

    /**
     * 组装签到日历
     *
     * @return
     */
    private List<CalendarSignDTO> generateMonthSign(String signDate) {
        List<String> monthDates = getDayByMonth(signDate);
        List<CalendarSignDTO> calendarSignDTOS = Lists.newArrayListWithCapacity(monthDates.size());
        for (String monthDate : monthDates) {
            CalendarSignDTO calendarSignDTO = new CalendarSignDTO();
            calendarSignDTO.setSignDate(monthDate);
            calendarSignDTOS.add(calendarSignDTO);
        }
        return calendarSignDTOS;
    }

    /**
     * 根据传入日期获取当前月份的日历
     *
     * @param signDate
     * @return
     */
    private List<String> getDayByMonth(String signDate) {
        int yearParam = Integer.valueOf(StringUtils.split(signDate, "-")[0]);
        int monthParam = Integer.valueOf(StringUtils.split(signDate, "-")[1]);
        List list = Lists.newArrayList();

        Calendar calendar = Calendar.getInstance();
        calendar.clear();
        calendar.set(yearParam, monthParam - 1, 1);
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.getActualMaximum(Calendar.DATE);

        for (int i = 1; i <= day; i++) {
            String monthDate = null;
            if (month < 10 && i < 10) {
                monthDate = String.valueOf(year) + "-0" + month + "-0" + i;
            }
            if (month < 10 && i >= 10) {
                monthDate = String.valueOf(year) + "-0" + month + "-" + i;
            }
            if (month >= 10 && i < 10) {
                monthDate = String.valueOf(year) + "-" + month + "-0" + i;
            }
            if (month >= 10 && i >= 10) {
                monthDate = String.valueOf(year) + "-" + month + "-" + i;
            }

            list.add(monthDate);
        }
        return list;
    }
}
