package com.bxm.localnews.activity.service.impl;

import java.util.*;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.activity.common.constant.PrivilegeStatusEnum;
import com.bxm.localnews.activity.domain.PrivilegeAreaMapper;
import com.bxm.localnews.activity.domain.PrivilegeMapper;
import com.bxm.localnews.activity.domain.PrivilegeParticipantMapper;
import com.bxm.localnews.activity.domain.WinnerMapper;
import com.bxm.localnews.activity.dto.*;
import com.bxm.localnews.activity.param.*;
import com.bxm.localnews.activity.service.PrivilegeService;
import com.bxm.localnews.base.dto.LocationFacadeDTO;
import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.base.service.ShortLinkService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.PrivilegeShareShortLinkDTO;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.vo.User;
import com.bxm.newidea.component.geo.dto.Coordinate;
import com.bxm.newidea.component.geo.service.GeoService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import com.fasterxml.jackson.core.type.TypeReference;
import com.github.pagehelper.Page;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

/**
 * Created by Arei on 2018/10/18.
 */
@Service("privilegeService")
public class PrivilegeServiceImpl extends BaseService implements PrivilegeService {

    private final static Logger log = LoggerFactory.getLogger(PrivilegeServiceImpl.class);

    @Resource
    private PrivilegeMapper privilegeMapper;

    @Resource
    private GeoService geoService;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private WinnerMapper winnerMapper;

    @Resource
    private RedisZSetAdapter redisZSetAdapter;

    @Resource
    private PrivilegeAreaMapper privilegeAreaMapper;

    @Resource
    private ShortLinkService shortLinkService;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private LocationFacadeService locationFacadeService;

    @Resource
    private AppVersionSupplyService appVersionSupplyService;

    @Resource
    private PrivilegeParticipantMapper privilegeParticipantMapper;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    private static PrivilegeParticipantDTO parseUserParticipant(String userInfoJson) {
        if (com.bxm.newidea.component.tools.StringUtils.isNotBlank(userInfoJson)) {
            try {
                UserInfoDTO userInfoDTO = JSON.parseObject(userInfoJson, UserInfoDTO.class);
                PrivilegeParticipantDTO privilegeParticipantDTO = new PrivilegeParticipantDTO();
                BeanUtils.copyProperties(userInfoDTO, privilegeParticipantDTO);
                return privilegeParticipantDTO;
            } catch (Exception e) {
                log.error("redis user parseObject error", e);
            }
        }
        return null;
    }

    @Override
    public PageWarper<PrivilegeParticipantDTO> listPrivilegeParticipantByInvite(PrivilegeParticipantParam privilegeParticipantParam) {
        //被邀请人ids
        Page<Long> inviterIds = (Page<Long>) privilegeParticipantMapper.listPrivilegeParticipantByInvite(privilegeParticipantParam);
        PageWarper pageWarper = new PageWarper(inviterIds);
        List<String> inviterStrIds = inviterIds.stream().map(String::valueOf).collect(Collectors.toList());
        //批量获取redis用户信息
        List<String> userInfoJsonList = redisHashMapAdapter.multiGet(RedisConfig.USER_INFO, inviterStrIds, String.class);
        List<PrivilegeParticipantDTO> privilegeParticipantDTOS = userInfoJsonList.stream()
                .map(PrivilegeServiceImpl::parseUserParticipant)
                .filter(privilegeParticipantDTO -> privilegeParticipantDTO != null)
                .collect(Collectors.toList());

        //如果拿到所有被邀请人的信息，做类型转换就行,没有拿全再补
        if (inviterIds.size() > privilegeParticipantDTOS.size()) {
            //获取已经拿到的userId列表
            List<Long> existUserIds = privilegeParticipantDTOS.stream()
                    .mapToLong(PrivilegeParticipantDTO::getId)//long stream
                    .boxed()//long stream需要装包，List<long>-->List<Long>
                    .collect(Collectors.toList());
            //获取没拿到用户信息的userId列表
            List<Long> lackUserIds = inviterIds.stream()
                    .filter(id -> !existUserIds.contains(id))
                    .collect(Collectors.toList());
            //获取用户信息并补全
            for (Long lackUserId : lackUserIds) {
                UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(lackUserId);
                PrivilegeParticipantDTO privilegeParticipantDTO = new PrivilegeParticipantDTO();
                BeanUtils.copyProperties(userInfoDTO, privilegeParticipantDTO);
                privilegeParticipantDTOS.add(privilegeParticipantDTO);
            }
        }
        pageWarper.setList(privilegeParticipantDTOS);
        return pageWarper;
    }

    @Override
    public PageWarper<PrivilegeListDTO> listPrivilege(PrivilegeParam privilegeParam, BasicParam basicParam) {
        logger.debug("获取活动列表,参数:{}", JSONObject.toJSONString(privilegeParam));
        if (privilegeParam.getShowAll() == 0) {
            LocationFacadeDTO locationDTO = locationFacadeService.getLocationByCode(privilegeParam.getAreaCode());
            if (null == locationDTO || 0 == locationDTO.getEnableVip()) {
                logger.warn("当前地区不存在或未开通，请求参数:{}", JSON.toJSON(privilegeParam));
                return new PageWarper<>(new ArrayList<>());
            }

            Date today = new Date();

            PrivilegeOverviewParam param = new PrivilegeOverviewParam();
            BeanUtils.copyProperties(privilegeParam, param);
            List<PrivilegeListDTO> privilegeListDTOS = listPrivilege(param);

            privilegeListDTOS = privilegeListDTOS.stream()
                    .filter(x -> compareDate(today, x.getStartTime(), x.getEndTime()))
                    .collect(Collectors.toList());

            int total = privilegeListDTOS.size();

            Coordinate userLngAndLat = getUserCoordinate(privilegeParam);
            //如果用户有传经纬度，则计算距离并按距离排序
            if (null != userLngAndLat) {
                distanceCalculation(userLngAndLat, privilegeListDTOS);
                privilegeListDTOS = privilegeListDTOS.stream()
                        .sorted(Comparator.comparing(PrivilegeListDTO::getDistanceNum)).collect(Collectors.toList());
            }

            //取分页中的部分
            privilegeListDTOS = privilegeListDTOS.stream()
                    .skip((privilegeParam.getPageNum() - 1) * privilegeParam.getPageSize())
                    .limit(privilegeParam.getPageSize()).collect(Collectors.toList());

            //填补free字段和折扣字段 200公里以上不显示距离
            privilegeListDTOS.forEach(this::fillFreeAndDiscount);

            Page page = new Page(privilegeParam.getPageNum(), privilegeParam.getPageSize());
            page.setTotal(total);
            page.addAll(privilegeListDTOS);

            return new PageWarper<>(page);
        } else {
            if (!getPublishState(basicParam)) {
                logger.error("地区编码为空!!! 用户:{},基础参数:{}", privilegeParam.getUserId(), JSON.toJSON(basicParam));
                return new PageWarper<>(new ArrayList<>());
            } else {
                logger.warn("IOS提包时请求此接口!!! 用户:{}", privilegeParam.getUserId());
                privilegeParam.setAreaCode(null);
                List<PrivilegeListDTO> privilegeListDTOS = privilegeMapper.listPrivilege(privilegeParam);
                Coordinate userLngAndLat = getUserCoordinate(privilegeParam);
                if (null != userLngAndLat) {
                    distanceCalculation(userLngAndLat, privilegeListDTOS);
                }
                fillingPrivilegeCountDown(privilegeListDTOS);
                return new PageWarper<>(privilegeListDTOS);
            }

        }
    }

    private boolean compareDate(Date today, Date startDate, Date endDate) {
        long todayTime = today.getTime();
        long startTime = startDate.getTime();
        long endTime = endDate.getTime();
        if (todayTime > startTime && todayTime < endTime) {
            return true;
        }
        return false;
    }

    /**
     * 得到某个地区所有的活动
     * 先从缓存取，如果没有，则去数据库中取
     *
     * @param privilegeParam
     * @return
     */
    private List<PrivilegeListDTO> listPrivilege(PrivilegeOverviewParam privilegeParam) {
        TypeReference<List<PrivilegeListDTO>> typeReference = new TypeReference<List<PrivilegeListDTO>>() {
        };
        List<PrivilegeListDTO> records = this.redisStringAdapter.get(this.getPriviligeListKey(privilegeParam.getAreaCode()), typeReference);
        if (CollectionUtils.isEmpty(records)) {
            records = privilegeMapper.listAllPrivilege(privilegeParam);
            redisStringAdapter.set(this.getPriviligeListKey(privilegeParam.getAreaCode()), records);
        }
        return records;
    }

    private KeyGenerator getPriviligeListKey(String areaCode) {
        return RedisConfig.PRIVILEGE_lIST.copy().appendKey(areaCode);
    }

    /**
     * 是否处于提包状态
     *
     * @param basicParam
     * @return
     */
    private Boolean getPublishState(BasicParam basicParam) {
        return appVersionSupplyService.getPublishState(basicParam);
    }

    /**
     * 填充活动列表里的免费享和折扣字段
     *
     * @param privilegeListDTO
     */
    private void fillFreeAndDiscount(PrivilegeListDTO privilegeListDTO) {
        if (null != privilegeListDTO.getDistanceNum() && privilegeListDTO.getDistanceNum() > 200000) {
            privilegeListDTO.setDistance(null);
        }
        if (StringUtils.isEmpty(privilegeListDTO.getFreeTitle())) {
            privilegeListDTO.setFree(false);
        } else {
            privilegeListDTO.setFree(true);
        }
        if (StringUtils.isEmpty(privilegeListDTO.getDiscount()) || StringUtils.isBlank(privilegeListDTO.getDiscount())) {
            privilegeListDTO.setDiscount("");
        }
    }

    @Override
    public List<PrivilegeOverviewDTO> listAllPrivilege(PrivilegeOverviewParam privilegeParam) {
        LocationFacadeDTO locationDTO = locationFacadeService.getLocationByCode(privilegeParam.getAreaCode());
        if (null == locationDTO || 0 == locationDTO.getEnableVip()) {
            logger.warn("当前地区不存在或未开通，请求参数:{}", JSON.toJSON(privilegeParam));
            return null;
        }

        List<PrivilegeListDTO> privilegeListDTOS = this.listPrivilege(privilegeParam);
        //筛选折扣不为空的
        Date today = new Date();
        privilegeListDTOS = privilegeListDTOS.stream()
                .filter(x -> x.getDiscount() != null || x.getFreeTitle() != null)
                .filter(x -> compareDate(today, x.getStartTime(), x.getEndTime())).collect(Collectors.toList());

        //如果客户端传了经纬度，则计算距离，否则按默认权重排序
        if (null != privilegeParam.getLatitude() && null != privilegeParam.getLongitude()) {
            Coordinate userLngAndLat = new Coordinate();
            userLngAndLat.setLatitude(privilegeParam.getLatitude());
            userLngAndLat.setLongitude(privilegeParam.getLongitude());
            distanceCalculation(userLngAndLat, privilegeListDTOS);
            privilegeListDTOS = privilegeListDTOS.stream().sorted(Comparator.comparing(PrivilegeListDTO::getDistanceNum)).collect(Collectors.toList());

        }
        privilegeListDTOS.forEach(this::fillFreeAndDiscount);
        List<PrivilegeOverviewDTO> privilegeOverviewDTOList = privilegeListDTOS.stream().map(this::convertPrivilege).collect(Collectors.toList());
//        fillingPrivilegeCountDown(privilegeListDTOS);
        return privilegeOverviewDTOList;
    }

    private PrivilegeOverviewDTO convertPrivilege(PrivilegeListDTO privilegeListDTO) {
        PrivilegeOverviewDTO privilegeOverviewDTO = new PrivilegeOverviewDTO();
        BeanUtils.copyProperties(privilegeListDTO, privilegeOverviewDTO);
        return privilegeOverviewDTO;
    }

    @Override
    public PageWarper<PrivilegeListDTO> listRecommendPrivilege(PrivilegeRecommendParam privilegeRecommendParam) {
        List<PrivilegeListDTO> privilegeListDTOS = privilegeMapper.listRecommendPrivilege(privilegeRecommendParam);
        Coordinate userLngAndLat = getUserCoordinate(privilegeRecommendParam);
        if (null != userLngAndLat) {
            distanceCalculation(userLngAndLat, privilegeListDTOS);
        }
        fillingPrivilegeCountDown(privilegeListDTOS);
        return new PageWarper<>(privilegeListDTOS);
    }

    @Override
    public PrivilegeDTO getPrivilegeDetail(PrivilegeDetailParam privilegeDetailParam) {
        PrivilegeDTO privilegeDTO = privilegeMapper.getPrivilegeDetail(privilegeDetailParam.getId());
        if (null == privilegeDTO) {
            return null;
        }
        PrivilegeParam param = new PrivilegeParam();
        BeanUtils.copyProperties(privilegeDetailParam, param);
        Coordinate userLngAndLat = getUserCoordinate(param);
        distanceCalculation(userLngAndLat, privilegeDTO);
        //设置活动分享短链
        if (privilegeDetailParam.getUserId() != null) {
            User user = userIntegrationService.selectByPrimaryKey(privilegeDetailParam.getUserId());
            PrivilegeShareShortLinkDTO privilegeShareShortLinkDTO = new PrivilegeShareShortLinkDTO();
            privilegeShareShortLinkDTO.setId(privilegeDetailParam.getId());
            privilegeShareShortLinkDTO.setUserId(privilegeDetailParam.getUserId());
            String name = user == null ? "游客" : user.getNickname();
            if (StringUtils.isEmpty(name)) {
                if (!StringUtils.isEmpty(user.getPhone())) {
                    name = user.getPhone().replaceAll("(\\d{3})\\d{4}(\\d{4})", "$1****$2");
                } else {
                    name = "游客";
                }
            }
            privilegeShareShortLinkDTO.setUserName(name);
            privilegeShareShortLinkDTO.setUserUrl(StringUtils.isEmpty((user == null ? null : user.getHeadImg())) ?
                    "https://bxm-news.oss-cn-hangzhou.aliyuncs.com/toutiao_test/common/default_head.png" : user.getHeadImg());
            privilegeDTO.setShortLink(shortLinkService.getPrivilegeShareShortLink(privilegeShareShortLinkDTO));
        }
        if (StringUtils.isEmpty(privilegeDetailParam.getAreaCode())) {
            privilegeDTO.setBarcode("http://mtest.wstong.com/localnews_test/png/20181204/H8UKWHWUZ9Q03IK21BDCQ9W7WS4O2CD2VF9PJQ1.png");
            privilegeDTO.setBarcodeGif("http://mtest.wstong.com/localnews_test/gif/20181204/H8UKWHWUZ9Q03IK21BDCQ9W7WS4O2I80VF9PJP1.gif");
            privilegeDTO.setCountdown((long) (10 * 60 * 60 * 1000));
            Date date = new Date();//取时间
            Calendar calendar = new GregorianCalendar();
            calendar.setTime(date);
            calendar.add(calendar.DATE, +1);//把日期往前减少一天，若想把日期向后推一天则将负数改为正数
            date = calendar.getTime();
            privilegeDTO.setEndTime(date);
            // 设置活动倒计时
//            PrivilegeStatusEnum privilegeStatusEnum = PrivilegeStatusEnum.getEnumByStatus(privilegeDTO.getStatus());
//            if (PrivilegeStatusEnum.WAIT_TO_START == privilegeStatusEnum) {
//                privilegeDTO.setCountdown(privilegeDTO.getStartTime().getTime() - System.currentTimeMillis());
//            }
//            if (PrivilegeStatusEnum.PROCESSING == privilegeStatusEnum) {
//                privilegeDTO.setCountdown(privilegeDTO.getEndTime().getTime() - System.currentTimeMillis());
//            }

            KeyGenerator inviteNumRedisKey = RedisConfig.PRIVILEGE_INVITENUM.copy().appendKey(privilegeDetailParam.getId());
            Set<Long> inviteUserIds =
                    redisZSetAdapter.range(inviteNumRedisKey, 0, 2, true, Long.class);
            List<LeaderboardDTO> leaderboardDTOS = new ArrayList<>();
            //设置活动排行榜
            for (Long inviteUserId : inviteUserIds) {
                LeaderboardDTO leaderboardDTO = new LeaderboardDTO();
                UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(inviteUserId);
                String nickname = userInfoDTO.getNickname();
                if (StringUtils.isEmpty(nickname)) {
                    nickname = userInfoDTO.getPhone().replaceAll("(\\d{3})\\d{4}(\\d{4})", "$1****$2");
                }
                leaderboardDTO.setNickname(nickname);
                leaderboardDTO.setUserId(inviteUserId);
                leaderboardDTO.setHeadImg(StringUtils.isEmpty(userInfoDTO.getHeadImg()) ?
                        "https://bxm-news.oss-cn-hangzhou.aliyuncs.com/toutiao_test/common/default_head.png" : userInfoDTO.getHeadImg());
                leaderboardDTO.setInviteNum(redisZSetAdapter.score(inviteNumRedisKey, inviteUserId).longValue());
                leaderboardDTOS.add(leaderboardDTO);
            }
            privilegeDTO.setLeaderboardDTOS(leaderboardDTOS);
            //设置当前用户是否已经参加过
            if (privilegeDetailParam.getUserId() != null) {
                privilegeDTO.setParticipated(privilegeParticipantMapper.getPrivilegeParticipant(privilegeDetailParam.getUserId(), privilegeDetailParam.getId()) != null);
            }
            //设置活动开放地区
            privilegeDTO.setAreaCodeList(privilegeAreaMapper.listPrivilegeAreaCode(privilegeDetailParam.getId()));
        }
        return privilegeDTO;
    }

    @Override
    public List<WinnerDTO> listWinnerByPrivilegeId(Long privilegeId, Long userId, boolean isDisplayCommodityCode) {
        List<WinnerDTO> winnerDTOS = winnerMapper.listWinnerByPrivilegeId(privilegeId);
        for (WinnerDTO winnerDTO : winnerDTOS) {
            if (StringUtils.isEmpty(winnerDTO.getNickname())) {
                winnerDTO.setNickname(winnerDTO.getPhone());
            }
            if (StringUtils.isEmpty(winnerDTO.getHeadImg())) {
                winnerDTO.setHeadImg("https://bxm-news.oss-cn-hangzhou.aliyuncs.com/toutiao_test/common/default_head.png");
            }

            //是否给前端返回中奖码
            if (!isDisplayCommodityCode) {
                winnerDTO.setCommodityCode(null);
            } else {
                if (userId == null) {
                    winnerDTO.setCommodityCode(null);
                } else if (!winnerDTO.getUserId().equals(userId)) {
                    winnerDTO.setCommodityCode(null);
                }
            }

            //在第一次展示中奖人的中奖码时，中奖弹窗提醒状态更新
            if (winnerDTO.getPopup() == 1 && winnerDTO.getCommodityCode() != null) {
                winnerMapper.updateWinnerPopup(winnerDTO.getId());
            }
        }
        return winnerDTOS;
    }

    @Override
    public PrivilegeDTO getPrivilegeDetail(Long id, Long userId, Long inviteUserId) {
//        PrivilegeDTO privilegeDTO = this.getPrivilegeDetail(id, userId);
//
//        if (privilegeDTO == null) {
//            return null;
//        }
////        if (inviteUserId != null && (privilegeDTO.getParticipated() != null && privilegeDTO.getParticipated())) {
////            privilegeDTO.setHelp(false);
////            PrivilegeParticipantVO privilegeParticipantVO = privilegeParticipantMapper.getPrivilegeParticipant(userId, id);
////            if (privilegeParticipantVO != null) {
////                if (inviteUserId.equals(privilegeParticipantVO.getInviteUserId())) {
////                    privilegeDTO.setHelp(true);
////                }
////            }
////        }
//        return privilegeDTO;
        return null;
    }

    /**
     * 获取用户经纬度定位
     *
     * @param privilegeParam 活动列表查询参数
     * @return
     */
    @Override
    public Coordinate getUserCoordinate(PrivilegeParam privilegeParam) {
        Double longitude = privilegeParam.getLongitude();
        Double latitude = privilegeParam.getLatitude();
        //有真实经纬度则使用真实经纬度
        if (null == longitude || null == latitude) {
            return null;
        }
        Coordinate userLngAndLat = new Coordinate();
        userLngAndLat.setLongitude(longitude);
        userLngAndLat.setLatitude(latitude);
        return userLngAndLat;
    }


    /**
     * 计算用户和活动地点之间的距离，如果拿不到用户或者活动的经纬度则跳过
     *
     * @param userLngAndLat     用户经纬度
     * @param privilegeListDTOS 活动列表
     */
    private void distanceCalculation(Coordinate userLngAndLat, List<PrivilegeListDTO> privilegeListDTOS) {
        if (null != userLngAndLat) {
            for (PrivilegeListDTO privilegeListDTO : privilegeListDTOS) {
                Coordinate merchantLngAndLat = new Coordinate();
                if (privilegeListDTO.getGeoLng() != null && privilegeListDTO.getGeoLat() != null) {
                    merchantLngAndLat.setLongitude(privilegeListDTO.getGeoLng());
                    merchantLngAndLat.setLatitude(privilegeListDTO.getGeoLat());
                    Long distance = geoService.getDistance(userLngAndLat, merchantLngAndLat);
                    String townName = privilegeListDTO.getTownName();
                    String distancePreStr = "";
                    if (!StringUtils.isEmpty(townName)) {
                        distancePreStr = "";
                    }
                    privilegeListDTO.setDistance(distancePreStr + Math.round(distance / 100d) / 10d + "km");
                    privilegeListDTO.setDistanceNum(distance);
                }
            }
        }
    }

    /**
     * 计算用户和活动地点之间的距离，如果拿不到用户或者活动的经纬度则跳过
     *
     * @param userLngAndLat    用户经纬度
     * @param privilegeListDTO 活动
     */
    private void distanceCalculation(Coordinate userLngAndLat, PrivilegeDTO privilegeListDTO) {
        if (null != userLngAndLat) {
            Coordinate merchantLngAndLat = new Coordinate();
            if (privilegeListDTO.getGeoLng() != null && privilegeListDTO.getGeoLat() != null) {
                merchantLngAndLat.setLongitude(privilegeListDTO.getGeoLng());
                merchantLngAndLat.setLatitude(privilegeListDTO.getGeoLat());
                Long distance = geoService.getDistance(userLngAndLat, merchantLngAndLat);
                String distancePreStr = "距您";
                privilegeListDTO.setDistance(distancePreStr + Math.round(distance / 100d) / 10d + "km");
            }

        }
    }

    /**
     * 填充活动开始倒计时文案（活动未开始）
     *
     * @param privilegeListDTOS
     * @return
     */
    private void fillingPrivilegeCountDown(List<PrivilegeListDTO> privilegeListDTOS) {
        Long currentTime = System.currentTimeMillis();
        for (PrivilegeListDTO privilegeListDTO : privilegeListDTOS) {
            if (PrivilegeStatusEnum.WAIT_TO_START
                    == PrivilegeStatusEnum.getEnumByStatus(privilegeListDTO.getStatus())) {
                privilegeListDTO.setCountdown(getPrivilegeUnStartContent(currentTime, privilegeListDTO.getStartTime()));
            }
        }
    }

    /**
     * 判断活动倒计时文案
     *
     * @param currentTime        当前时间
     * @param privilegeStartDate 活动开始时间
     * @return
     */
    private String getPrivilegeUnStartContent(Long currentTime, Date privilegeStartDate) {
        Long waitTime = privilegeStartDate.getTime() - currentTime;
        Long oneDay = 3600 * 24 * 1000L;
        Long oneHour = 3600 * 1000L;
        Long oneMinute = 60 * 1000L;
        Long oneSecond = 1000L;
        if (waitTime.compareTo(oneDay) >= 0) {
            long dayNum = Math.round((double) waitTime / oneDay);
            return dayNum + "天后开始";
        }
        if (waitTime.compareTo(oneDay) == -1 && waitTime.compareTo(oneHour) >= 0) {
            long hourNum = Math.round((double) waitTime / oneHour);
            return hourNum + "小时后开始";
        }
        if (waitTime.compareTo(oneHour) == -1 && waitTime.compareTo(oneMinute) >= 0) {
            long miniuteNum = Math.round((double) waitTime / oneMinute);
            return miniuteNum + "分钟后开始";
        }
        if (waitTime.compareTo(oneMinute) == -1 && waitTime.compareTo(oneSecond) >= 0) {
            long secondNum = Math.round((double) waitTime / oneSecond);
            return secondNum + "秒后开始";
        }
        return null;
    }
}
