package com.bxm.localnews.activity.service.impl;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import com.bxm.localnews.activity.domain.SignConfigMapper;
import com.bxm.localnews.activity.domain.SignRecordMapper;
import com.bxm.localnews.activity.service.SignFacadeService;
import com.bxm.localnews.activity.service.SignService;
import com.bxm.localnews.activity.vo.SignConfig;
import com.bxm.localnews.activity.vo.SignRecord;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.param.AccountGoldParam;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import static com.bxm.localnews.common.constant.RedisConfig.SIGN_CACHE_KEY;
import static com.bxm.newidea.component.tools.DateUtils.addField;

@Service
public class SignServiceImpl extends BaseService implements SignService, SignFacadeService {

    private static final Logger logger = LoggerFactory.getLogger(SignServiceImpl.class);

    /**
     * 用户签到数据过期时间，防止非活跃用户长期占据内存
     */
    private static final int EXPIRE_TIME = 3600 * 24 * 2;

    /**
     * 签到列表长度
     */
    private static final int SIGN_LIST_LENGTH = 8;

    @Resource
    private SignConfigMapper signConfigMapper;

    @Resource
    private SignRecordMapper signRecordMapper;

    @Resource
    private UserAccountIntegrationService userAccountIntegrationService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private DistributedLock distributedLock;

//    @Override
//    public com.bxm.localnews.activity.dto.SignWarper listSignRecord(Long userId, int platform) {
//        com.bxm.localnews.activity.dto.SignWarper warper = new com.bxm.localnews.activity.dto.SignWarper();
//        //获取签到记录
//        List<SignRecord> signRecords = this.listUserSignRecord(userId);
//
//        User user = userService.selectByPrimaryKey(userId);
//        if (user != null && 1 == user.getState()) {
//            Date today = new Date();
//
//            for (SignRecord record : signRecords) {
//                if (DateUtils.isSameDay(record.getSignDate(), today) && record.getSignFlag()) {
//                    warper.setTodaySignState(true);
//                    break;
//                }
//            }
//
//            String requestId = String.valueOf(nextSequence());
//            if (!warper.isTodaySignState()) {
//                if (distributedLock.lock(String.valueOf(userId), requestId)) {
//                    logger.warn("签到逻辑-今日未签到，自动进行签到,user:[{}]", userId);
//                    signRecords = this.getLastSignRecordsByDoDailySign(userId, signRecords);
//                    //用于标记是否弹窗签到成功页面
//                    warper.setAutoSign(false);
//                    warper.setTodaySignState(true);
//                    //解锁
//                    distributedLock.unlock(String.valueOf(userId), requestId);
//                }else{
//                    logger.warn("签到逻辑-获取不到锁，进行短暂休眠，重新走一遍签到逻辑,user:[{}]",userId);
//                    try {
//                        Thread.sleep(200);
//                    } catch (InterruptedException e) {
//                        logger.error("分布式锁休眠时间:200,错误:{}",e.getMessage());
//                    }
//                    return this.listSignRecord(userId,platform);
//                }
//            } else {
//                warper.setAutoSign(true);
//            }
//            //获取最新一天的签到金币数
//            Long signReward = signRecords.stream()
//                    .filter(SignRecord::getSignFlag)
//                    .sorted(Comparator.comparing(SignRecord::getSignDate).reversed())
//                    .findFirst().get().getSignReward();
//
//            warper.setSignReward(signReward);
//
//            warper.setTotalGold(userAmountIntegrationService.selectGoldBalanceByUserId(userId));
//        }
//        warper.setRecords(signRecords);
//        warper.setCount(this.getSignCount(userId));
//        return warper;
//    }

    @Override
    public com.bxm.localnews.activity.dto.SignWarper listSignRecord(Long userId, int platform) {
        com.bxm.localnews.activity.dto.SignWarper warper = new com.bxm.localnews.activity.dto.SignWarper();
        //获取签到记录
        Date today = new Date();
        List<SignRecord> signRecords = this.buildSignList(0, today, userId, true);

        warper.setAutoSign(true);
        warper.setTodaySignState(true);
        warper.setSignReward(0L);
        warper.setTotalGold(BigDecimal.valueOf(userAccountIntegrationService.getUserUsableGold(userId)));
        warper.setRecords(signRecords);
        warper.setCount(this.getSignCount(userId));
        return warper;
    }

    /**
     * 用户今日签到之后最新的签到记录
     *
     * @param userId
     * @param signRecords
     * @return
     */
    private List<SignRecord> getLastSignRecordsByDoDailySign(Long userId, List<SignRecord> signRecords) {
        logger.debug("执行签到:[{}]", userId);

        //判断今日是否已进行过签到
        Date today = new Date();
        for (SignRecord record : signRecords) {
            if (DateUtils.isSameDay(record.getSignDate(), today) && record.getSignFlag()) {
                logger.debug("重复签到:[{}]", userId);
                return signRecords;
            }
        }

        //保存签到信息，增加积分信息
        this.saveSignRecord(userId);

        //更新签到列表
        for (SignRecord cacheRecord : signRecords) {
            if (DateUtils.isSameDay(cacheRecord.getSignDate(), today)) {
                cacheRecord.setSignFlag(true);
                break;
            }
        }

        this.redisStringAdapter.set(this.getSignListKey(userId), signRecords, EXPIRE_TIME);
        return signRecords;
    }

    /**
     * 获取用户签到次数的key
     *
     * @param userId
     * @return
     */
    private KeyGenerator getSignCountKey(Long userId) {
        return RedisConfig.SIGN_COUNT_KEY.copy().appendKey(userId.toString());
    }

    /**
     * 获取用户签到列表的key
     *
     * @param userId
     * @return
     */
    private KeyGenerator getSignListKey(Long userId) {
        return RedisConfig.SIGN_LIST_KEY.copy().appendKey(userId.toString());
    }

    /**
     * 获取用户签到次数
     *
     * @param userId
     * @return
     */
    private int getSignCount(Long userId) {
        Integer count = this.redisStringAdapter.getInt(this.getSignCountKey(userId));

        return count == null ? 1 : count;
    }

    /**
     * 重置用户签到总数
     *
     * @param userId 用户ID
     */
    private void resetSignCount(Long userId) {
        this.redisStringAdapter.set(this.getSignCountKey(userId), 0);
    }

    /**
     * 保存签到记录
     *
     * @param userId
     * @return
     */
    private SignRecord saveSignRecord(Long userId) {

        SignRecord lastRecord = signRecordMapper.getLastSignRecord(userId);

        if (lastRecord != null) {
            if (DateUtils.isSameDay(lastRecord.getSignDate(), new Date())) {
                return lastRecord;
            }
        }

        long reward = this.getSignReward(userId);
        //用户签到天数自增
        Long count = this.redisStringAdapter.increment(this.getSignCountKey(userId));
        logger.debug("签到成功，当前天数：[{}]", userId, count);

        SignRecord record = new SignRecord();
        record.setSignDate(new Date());
        record.setSignReward(reward);
        record.setSignDay(count.intValue());
        record.setUserId(userId);

        //保存签到记录
        this.signRecordMapper.insertSelective(record);

        AccountGoldParam param = new AccountGoldParam(userId, "USABLE_GOLD", true,
                (int) reward, record.getId(), "SIGN_DAILY");
        userAccountIntegrationService.addGold(param);

        return record;
    }

    /**
     * 获取签到天数对应的奖金
     *
     * @param userId
     * @return
     */
    private long getSignReward(Long userId) {
        List<SignConfig> signConfigs = this.getSignconfigInfo();

        int count = this.getSignCount(userId);

        if (count >= SIGN_LIST_LENGTH) {
            return signConfigs.get(SIGN_LIST_LENGTH - 1).getReward();
        }

        return signConfigs.get(count).getReward();

    }

    /**
     * 获取最后一次未签到记录与当天的签到日期对比
     *
     * @param records
     * @return true 表示未断签 false 表示已经断签（断签需要重新构建签到列表）
     */
    private boolean getLastUnSignRecord(List<SignRecord> records) {
        //最后一条签到记录已签到 表示未断签
        if (records.get(records.size() - 1).getSignFlag()) {
            return true;
        }
        SignRecord lastRecord = records.stream().filter(signRecord -> signRecord.getSignFlag() == false).findFirst().get();
        Date today = new Date();
        //最后待签到日期是今日或者明日表示没有断签
        return null != lastRecord && (DateUtils.isSameDay(today, lastRecord.getSignDate())
                || DateUtils.isSameDay(addField(lastRecord.getSignDate(), Calendar.DATE, -1), today));
    }

    /**
     * 获取用户签到记录列表,只返回最近8天的数据
     * 签到记录使用缓存进行存储，设置3天有效期
     *
     * @param userId 用户ID
     * @return 用户签到记录
     */
    private List<SignRecord> listUserSignRecord(Long userId) {
        Date today = new Date();
        TypeReference<List<SignRecord>> typeReference = new TypeReference<List<SignRecord>>() {
        };
        List<SignRecord> records = this.redisStringAdapter.get(this.getSignListKey(userId), typeReference);

        if (CollectionUtils.isNotEmpty(records)) {
            SignRecord lastRecord = records.get(records.size() - 1);

            //如果最后一条签到记录已经是昨天则开启一条新的签到列表(签到到末尾)
            if (DateUtils.isSameDay(addField(lastRecord.getSignDate(), Calendar.DATE, 1), today)) {
                records = this.buildSignList(this.getSignCount(userId) + 1, today, userId, true);
            }
            //如果已断签，则重新开启一个签到列表
            else if (!this.getLastUnSignRecord(records)) {
                records = this.buildSignList(0, today, userId, true);
                this.resetSignCount(userId);
            }

            return records;
        }

        //获取用户的最后一条签到记录
        SignRecord record = this.signRecordMapper.getLastSignRecord(userId);

        //如果签到日期不是今天或者昨天，表示已经断签，则重新开启一条签到list
        boolean isSequential = record != null &&
                (DateUtils.isSameDay(today, record.getSignDate())
                        || DateUtils.isSameDay(addField(record.getSignDate(), Calendar.DATE, 1), today)
                );

        if (isSequential) {
            records = this.buildSignList(this.getSignCount(userId), record.getSignDate(), userId, false);
        } else {
            records = this.buildSignList(0, today, userId, true);
            this.resetSignCount(userId);
        }

        return records;
    }

    /**
     * 构建签到列表
     *
     * @param startIndex 开始签到的天数
     * @param startDate  开始签到的日期
     * @param newList    是否为开始的新列表(新列表的已签到标签全部为false)
     * @return 签到列表
     */
    private List<SignRecord> buildSignList(int startIndex, Date startDate, Long userId, boolean newList) {
        List<SignRecord> records = Lists.newArrayList();

        List<SignConfig> signConfigs = this.getSignconfigInfo();

        int startNum = startIndex > SIGN_LIST_LENGTH ? ((startIndex - 1) / SIGN_LIST_LENGTH) * SIGN_LIST_LENGTH : 0;

        int actulStartIndex = startIndex % SIGN_LIST_LENGTH;
        if (startIndex > 0 && actulStartIndex == 0) {
            actulStartIndex = SIGN_LIST_LENGTH;
        } else if (startIndex == 0) {
            actulStartIndex = 1;
        }

        SignRecord record;
        for (int i = 0; i < SIGN_LIST_LENGTH; i++) {
            record = new SignRecord();
            int signDay = i + 1 + startNum;
            record.setSignDay(signDay);
            if (startIndex == 0) {
                record.setSignDate(addField(startDate, Calendar.DATE, i));
            } else {
                record.setSignDate(addField(startDate, Calendar.DATE, i + 1 - actulStartIndex));
            }
            if (newList) {
                record.setSignFlag(false);
            } else {
                record.setSignFlag(startIndex >= signDay);
            }

            if (signDay >= SIGN_LIST_LENGTH) {
                record.setSignReward(signConfigs.get(SIGN_LIST_LENGTH - 1).getReward());
            } else {
                record.setSignReward(signConfigs.get(i).getReward());
            }

            records.add(record);
        }

        this.redisStringAdapter.set(this.getSignListKey(userId), records, EXPIRE_TIME);

        return records;
    }

    /**
     * 获取签到奖励配置信息
     *
     * @return
     */
    @Override
    public List<SignConfig> getSignconfigInfo() {
        TypeReference<List<SignConfig>> typeReference = new TypeReference<List<SignConfig>>() {
        };
        List<SignConfig> redisRet = this.redisStringAdapter.get(SIGN_CACHE_KEY, typeReference);
        if (CollectionUtils.isNotEmpty(redisRet)) {
            return redisRet;
        }

        List<SignConfig> list = this.signConfigMapper.listAll();
        this.redisStringAdapter.set(SIGN_CACHE_KEY, list, 3600 * 24 * 7);
        return list;
    }


}
