package com.bxm.localnews.activity.service.redpacket.command.impl;

import com.bxm.localnews.activity.config.CommandRedPacketProperties;
import com.bxm.localnews.activity.constant.PackTypeEnum;
import com.bxm.localnews.activity.constant.RedPacketObtainStatesEnum;
import com.bxm.localnews.activity.domain.LotteryDrawMapper;
import com.bxm.localnews.activity.domain.RedPacketMapper;
import com.bxm.localnews.activity.dto.CommandResultDTO;
import com.bxm.localnews.activity.dto.ScoreDto;
import com.bxm.localnews.activity.dto.SubtractDTO;
import com.bxm.localnews.activity.service.redpacket.command.RedPacketAmountService;
import com.bxm.localnews.activity.service.redpacket.command.RedPacketObtainService;
import com.bxm.localnews.activity.service.redpacket.command.RedPacketStateService;
import com.bxm.localnews.activity.util.AmountUtil;
import com.bxm.localnews.activity.vo.RedPacket;
import com.bxm.localnews.base.dto.LocationFacadeDTO;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.vo.User;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.service.BaseService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @ClassName RedPacketObtainServiceImpl
 * @CopyRright (c) 2018-bxm：杭州微财网络科技有限公司
 * @Author zhaox
 * @Date 2019/4/30 15:34
 * @Version 1.0.0
 * @Modifier zhaox
 * @Modify Date 2019/4/30 15:34
 **/
@Service
public class RedPacketObtainServiceImpl extends BaseService implements RedPacketObtainService {
    private final static Logger logger = LoggerFactory.getLogger(RedPacketObtainServiceImpl.class);

    @Autowired
    private LotteryDrawMapper lotteryDrawMapper;
    @Autowired
    private LocationFacadeService locationFacadeService;
    @Autowired
    private RedPacketStateService redPacketStateService;
    @Autowired
    private CommandRedPacketProperties commandRedPacketProperties;
    @Autowired
    private RedPacketMapper redPacketMapper;
    @Autowired
    private DistributedLock distributedLock;
    @Autowired
    private RedPacketAmountService redPacketAmountService;

    @Override
    public CommandResultDTO obtainRedPacket(Long rpId, ScoreDto scoreDto, User user, Integer platform,Integer retryTime) {
        if (retryTime == 0) {
            logger.error("retry time is 0, rpid:[{}],userId:[{}]", rpId, user.getId());
            return new CommandResultDTO(RedPacketObtainStatesEnum.LESS40.getType());
        }
        String requestId = nextSequence().toString();
        if (!distributedLock.lock(rpId.toString(), requestId)) {
            logger.warn("获取分布式锁失败重试, rpid:[{}], retry time:[{}]",rpId,retryTime);
            return obtainRedPacket(rpId, scoreDto, user, platform, --retryTime);
        }
        // 红包状态检查
        RedPacket redPacket = redPacketMapper.queryOne(rpId);
        RedPacketObtainStatesEnum rpObtainState = redPacketStateService.rpStatesCheck(redPacket, user.getId());
        if (rpObtainState != null) {
            distributedLock.unlock(rpId.toString(), requestId);
            return new CommandResultDTO(rpObtainState.getType());
        }

        // 减扣红包金额
        SubtractDTO subtractDTO =null;
        try {
            subtractDTO = redPacketAmountService.setSurplusAmount(redPacket, scoreDto, user,platform);
        }catch (Exception e){
            e.printStackTrace();
        }

        distributedLock.unlock(rpId.toString(), requestId);

        // 减扣失败重试
        if (null ==subtractDTO || !subtractDTO.getSubtractSuccess()) {
            logger.warn("红包减扣失败重试, rpid:[{}], retry time:[{}], version:[{}]",rpId,retryTime,redPacket.getVersion());
            return obtainRedPacket(rpId, scoreDto, user, platform, --retryTime);
        }

        return convertToCommandResultDTO(scoreDto, user, subtractDTO);
    }

    @Override
    public CommandResultDTO publicObtainRedPacket(Long rpId, Integer packetType, String city, String district, ScoreDto scoreDto, User user,Integer platform, Integer retryTime) {
        CommandResultDTO commandResultDTO = obtainRedPacket(rpId, scoreDto, user, platform, retryTime);
        // 设置新用户红包金额
        commandResultDTO.setNewUserAmount(getNewUserRpAmount(city, district, packetType, user));
        return commandResultDTO;
    }




    @Override
    public String getNewUserRpAmount(String city,String district,Integer packetType,User user) {
        logger.info("getNewUserRpAmount ,city:[{}],district:[{}],packetType:[{}],receive:[{}],user state:[{}]", city, district, packetType, user.getReceiveRedPacket(), user.getState());
        boolean isUnusable = !packetType.equals(PackTypeEnum.WALL.getType()) // 非红包墙红包
                || user.getReceiveRedPacket() == (byte) 0 // 领取过新人红包
                || user.getState() != (byte) 2;// 非未激活用户
        // 红包类型为普通红包且未领取过新人红包的非激活用户才可领取红包
        logger.info("isUnusable :[{}]", isUnusable);
        if (isUnusable) {
            return null;
        }
        String areaCode = getAreaCodeByCityName(city, district);
        if (StringUtils.isBlank(areaCode)) {
            return null;
        }
        LocationFacadeDTO locationFacadeDTO = locationFacadeService.getLocationByCode(areaCode);
        if (null == locationFacadeDTO) {
            return null;
        }

        return AmountUtil.halfUpScale2StringValue(locationFacadeDTO.getNewRedPacket());
    }

    /**
     * 获取地区编码
     */
    private String getAreaCodeByCityName(String city, String district) {
        String areaCode = "";
        if (StringUtils.isNotBlank(district)){
            areaCode = lotteryDrawMapper.getCodeByDistrict(district);
        }
        if (StringUtils.isBlank(areaCode) && StringUtils.isNotBlank(city)) {
            areaCode = lotteryDrawMapper.getCodeByCity(city);
        }
        return areaCode;
    }




    private CommandResultDTO convertToCommandResultDTO(ScoreDto scoreDto, User user, SubtractDTO subtractDTO) {
        CommandResultDTO commandResultDTO = new CommandResultDTO();
        commandResultDTO.setStatus(RedPacketObtainStatesEnum.SUCCESS.getType());
        commandResultDTO.setUserName(user.getNickname());
        commandResultDTO.setUserImg(user.getHeadImg());
        commandResultDTO.setScore(scoreDto.getScore());
        commandResultDTO.setStar(scoreDto.getStar());
        commandResultDTO.setUserScale(scoreDto.getUserScale());
        commandResultDTO.setMatchScale(scoreDto.getMatchScale());
        commandResultDTO.setUserObtainAmount(AmountUtil.halfUpScale2StringValue(subtractDTO.getActualAmount()));
        commandResultDTO.setNewUserAmount(null);
        commandResultDTO.setMaxLength(commandRedPacketProperties.getMaxLength());
        commandResultDTO.setLength(scoreDto.getLength());
        commandResultDTO.setCommandUrl(scoreDto.getCommandUrl());
        return commandResultDTO;
    }

}
