package com.bxm.localnews.activity.service.sign;

import com.alibaba.fastjson.JSONArray;
import com.bxm.localnews.activity.common.config.SignProperties;
import com.bxm.localnews.activity.common.constant.sign.SignGiftEnum;
import com.bxm.localnews.activity.common.content.SignGiftContent;
import com.bxm.localnews.activity.domain.SignRecordMapper;
import com.bxm.localnews.activity.vo.SignGiftReceiveContent;
import com.bxm.localnews.activity.vo.SignRecord;
import com.bxm.localnews.common.config.ClientConfigProperties;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.AllArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author pf.w
 * @date 2020/10/26 14:49
 **/
@Component
@AllArgsConstructor
public class DailySignNewCacheAndLogic {

    private final SignProperties signProperties;

    private final UserIntegrationService userIntegrationService;

    private final RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    protected SignRecordMapper signRecordMapper;

    private ClientConfigProperties clientConfigProperties;

    /**
     * 能否领取签到礼包
     *
     * @return true : 能
     */
    public boolean canReceiveGift(Long userId, SignGiftEnum signGiftEnum){
        //是否有领签到红包的资格
        if(!hasReceiveGiftInterest(userId,signGiftEnum)){
            return false;
        }

        SignRecord signRecord = signRecordMapper.getLastSignRecord(userId);
        if(Objects.isNull(signRecord)){
            return false;
        }
        // 签到天数是否符合领奖规则
        return Objects.nonNull(this.getGiftPackage(signRecord.getSignDay()));
    }

    /**
     * 是否拥有领取签到红包的资格
     *
     * @param userId 用户ID
     * @param signGiftEnum 红包类型
     * @return true : 有
     */
    public boolean hasReceiveGiftInterest(Long userId, SignGiftEnum signGiftEnum){
        //只有新人才能领
        if(!isNewUser(userId)){
            return false;
        }
        //是否领取过
        if(hasReceiveGift(userId,signGiftEnum)){
            return false;
        }
        return true;
    }

    /**
     * 判断某个区域是否配置vip
     *
     * @param areaCode 地区码
     * @return true : 开通
     */
    public boolean hasVipForAreaCode(String areaCode){
        List<String> hasVipList = JSONArray.parseArray(clientConfigProperties.getCommonKeys()
                .get("specialTabbarAreaCode"),String.class);

        if(CollectionUtils.isEmpty(hasVipList)){
            return false;
        }
        return hasVipList.contains(areaCode);
    }

    /**
     * 开通签到提醒
     *
     * @param userId 用户ID
     */
    public void openSignWarningToRedis(Long userId){
        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
        if(Objects.isNull(userInfoDTO)){
            return;
        }
        redisHashMapAdapter.put(signWarningKey(),userId.toString(),userInfoDTO.getNickname());
    }

    /**
     * 判断签到提醒开启状态
     *
     * @param userId 用户ID
     * @return true : 开启
     */
    public boolean isOpenSignWarning(Long userId){
        return redisHashMapAdapter.exists(signWarningKey(),userId.toString());
    }

    /**
     * 关闭签到提醒
     *
     * @param userId 用户ID
     */
    public void closeSignWarningFromRedis(Long userId){
        redisHashMapAdapter.remove(signWarningKey(),userId.toString());
    }


    /**
     * 签到提醒开关key
     *
     * @return 签到key
     */
    private KeyGenerator signWarningKey(){
        return RedisConfig.SIGN_WARNING_OPERATE_KEY.copy();
    }

    /**
     * 判断是否是新用户  通过该版本上线时间区分新用户
     * 说明：
     * 1,只适用3.8.0版本- 新用户签到领红包逻辑  ；
     * 2,不通用
     *
     * @return true : 是新用户  false : 不是新用户
     */
    private boolean isNewUser(Long userId){
        UserInfoDTO user = userIntegrationService.getUserFromRedisDB(userId);

        if(Objects.isNull(user)){
            return false;
        }
        return user.getCreateTime().after(DateUtils.parse(signProperties.getSignTimeFlag()));
    }

    /**
     *连续签到天数对应礼包实体详情
     *
     * @param signDay 签到天数
     * @return 对应天数礼包实体
     */
    public SignGiftContent getGiftPackage(Integer signDay){
        if(signDay <= 0 ){
            return null;
        }

        //获取礼包配置信息
        String giftPackage = signProperties.getGiftPackage();

        List<SignGiftContent> giftPackages = JSONArray.parseArray(giftPackage,SignGiftContent.class);

        for (SignGiftContent content : giftPackages) {
            //比对 取模
            if(Objects.equals(content.getSignDay(),signDay) || signDay%content.getSignDay() == 0){
                return content;
            }
        }
        return null;
    }

    /**
     *礼包类型对应礼包实体详情
     *
     * @param giftEnum 签到天数
     * @return 对应天数礼包实体
     */
    private SignGiftContent getGiftPackage(SignGiftEnum giftEnum){

        //获取礼包配置信息
        String giftPackage = signProperties.getGiftPackage();

        List<SignGiftContent> giftPackages = JSONArray.parseArray(giftPackage,SignGiftContent.class);

        for (SignGiftContent content : giftPackages) {
            if(Objects.equals(content.getGiftType(),giftEnum.getCode())){
                return content;
            }
        }
        return null;
    }

    /**
     * 判断用户是否领取过新人7天礼包
     *
     * @param userId 用户ID
     * @param signGiftEnum 礼包类型 {@link SignGiftEnum}
     * @return true : 领取过
     */
    private boolean hasReceiveGift(Long userId,SignGiftEnum signGiftEnum){
        List<SignGiftReceiveContent> giftList = userGiftList(userId);
        if(CollectionUtils.isEmpty(giftList)){
            return false;
        }
        for (SignGiftReceiveContent giftContent : giftList) {
            if(Objects.equals(giftContent.getGiftType(),signGiftEnum.getCode())){
                return true;
            }
        }

        return false;
    }

    /**
     * 从缓存获取用户领取过的礼包集合
     *
     * @param userId 用户ID
     * @return 礼包集合
     */
    public List<SignGiftReceiveContent> userGiftList(Long userId){
        TypeReference<List<SignGiftReceiveContent>> typeReference = new TypeReference<List<SignGiftReceiveContent>>(){

        };
        return redisHashMapAdapter.get(signUserGiftKey(),userId.toString(),typeReference);
    }

    /**
     * 用户签到礼包领取 集合key
     *
     * @return 领取礼包集合 key
     */
    public KeyGenerator signUserGiftKey(){
        return RedisConfig.SIGN_GIFT_RECEIVE_KEY.copy();
    }


    /**
     * 缓存 获取7日礼包红花随机数
     *
     * 目前是20-100之间随机
     *
     * @param userId 用户id
     * @return 红花数
     */
    public Long goldNumTodayRandom(Long userId){
        Long result = redisHashMapAdapter.get(RedisConfig.SIGN_GIFT_GOLD_NUM_RANDOM,userId.toString(),Long.class);
        if(Objects.nonNull(result) && result >= 0){
            return result;
        }

        //缓存没有 重新计算7日礼包随机数 并入缓存
        SignGiftContent giftContent = getGiftPackage(SignGiftEnum.SIGN_GIFT_NEW_USER);
        List<Integer> numList = Arrays.stream(giftContent.getReward().split("-"))
                .map(Integer::parseInt).collect(Collectors.toList());

        Random ran = new Random();
        int goldNumRandom = ran.nextInt(numList.get(1)-numList.get(0) + 1) + numList.get(0);

        redisHashMapAdapter.put(RedisConfig.SIGN_GIFT_GOLD_NUM_RANDOM,userId.toString(),(long)goldNumRandom);
        //有效期 物理一天时间
        redisHashMapAdapter.expire(RedisConfig.SIGN_GIFT_GOLD_NUM_RANDOM,DateUtils.getTodayEndMillis()/1000);
        return (long)goldNumRandom;
    }

    /**
     * 领取过后删除 随机红花数缓存
     * @param userId 用户ID
     */
    public void removeGoldNumTodayRandom(Long userId){
        redisHashMapAdapter.remove(RedisConfig.SIGN_GIFT_GOLD_NUM_RANDOM,userId.toString());
    }

    /**
     * 获取某天后的intervals天日期
     *
     * @param day 日期
     * @param intervals 向后几天
     * @return 返回日期集合
     */
    public ArrayList<Date> getDays(Date day,int intervals) {
        ArrayList<Date> pastDaysList = new ArrayList<>();
        for(int i = 0; i < intervals ; i ++){
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(day);
            calendar.set(Calendar.DATE, calendar.get(Calendar.DATE) + i);
            Date today = calendar.getTime();
            pastDaysList.add(today);
        }
        return pastDaysList;
    }

    /**
     * 获取某日向前几天
     *
     * @param day 日期
     * @param intervals 向前天数
     * @return 向前intervals天后的日期
     */
    public Date getDay(Date day,int intervals){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(day);
        calendar.set(Calendar.DATE, calendar.get(Calendar.DATE) -(intervals - 1));
        return calendar.getTime();
    }
}
