package com.bxm.localnews.activity.service.sign.impl;

import java.util.*;

import com.bxm.localnews.activity.common.config.SignProperties;
import com.bxm.localnews.activity.domain.SignRecordMapper;
import com.bxm.localnews.activity.dto.CalendarSignDTO;
import com.bxm.localnews.activity.dto.SignDTO;
import com.bxm.localnews.activity.dto.SignLeaderBoard;
import com.bxm.localnews.activity.service.sign.DailySignService;
import com.bxm.localnews.activity.service.SignConfigService;
import com.bxm.localnews.activity.vo.SignConfig;
import com.bxm.localnews.activity.vo.SignRecord;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.param.AccountGoldParam;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

/**
 * @author zhaoyadong 2019/4/4 11:30
 * @desc
 */
@Service
@Component
public class DailySignServiceImpl extends BaseService implements DailySignService {

    @Autowired
    private SignConfigService signConfigService;

    @Resource
    protected SignRecordMapper signRecordMapper;

    @Autowired
    private DistributedLock distributedLock;

    @Autowired
    private UserAccountIntegrationService userAccountIntegrationService;

    @Autowired
    private RedisZSetAdapter redisZSetAdapter;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    private RedisSetAdapter redisSetAdapter;

    @Autowired
    private LocationFacadeService locationFacadeService;

    @Autowired
    private SignProperties signProperties;

    @Override
    public SignDTO signRecord(Long userId,String areaCode) {
        SignDTO signDTO = generateSign(userId);
        //判断该地区是否开通了签到排行榜
        signDTO.setEnableSignBoard((byte) 0);
        LocationDetailDTO locationDetailDTO = locationFacadeService.getLocationDetailByCode(areaCode);
        if (null != locationDetailDTO && locationDetailDTO.getEnableSignBoard()==1) {
            signDTO.setEnableSignBoard((byte) 1);
        }
        return signDTO;
    }

    @Override
    public List<CalendarSignDTO> listSignRecord(Long userId) {
        Date now = new Date();
        String signDate = DateUtils.formatDate(now);
        //获取对应月份的上一个月的签到信息
        String beforeDate = DateUtils.formatDate(DateUtils.addField(DateUtils.parseDate(signDate), Calendar.MONTH, -1));
        List<CalendarSignDTO> beforeSignRecords = getMonthSignRecord(userId, beforeDate);

        //获取对应月份的签到信息
        List<CalendarSignDTO> signRecords = getMonthSignRecord(userId, signDate);
        beforeSignRecords.addAll(signRecords);
        return beforeSignRecords;
    }

    @Override
    public Message executeUserSign(Long userId,String areaCode) {
        Message message = Message.build();
        SignDTO signDTO = generateSign(userId);
        if (signDTO.isTodaySignState()) {
            return message.setSuccess(false).setMessage("用户已签到");
        }

        String requestId = String.valueOf(nextId());
        Long id = nextId();
        if (distributedLock.lock(userId + "" + id, requestId)) {
            int signDay = signDTO.getCount().intValue() + 1;
            Date date = new Date();
            SignRecord signRecord = new SignRecord(
                    id,
                    userId,
                    date,
                    signDay,
                    signDTO.getGold().longValue(),
                    false,
                    areaCode);
            signRecordMapper.insertSelective(signRecord);
            //签到添加金币
            AccountGoldParam param = new AccountGoldParam(userId, "USABLE_GOLD", true,
                    signDTO.getGold(), id, "SIGN_DAILY");
            userAccountIntegrationService.addGold(param);
            this.record2redis(userId, date);
            this.rankingSignRecord(userId);
            distributedLock.unlock(userId + "" + id, requestId);
        } else {
            return message.setSuccess(false).setMessage("用户已签到");
        }

        return message;
    }

    /**
     * 将用户签到的记录保存到redis中
     * @param userId
     * @param date
     */
    private void record2redis(Long userId, Date date){
        KeyGenerator signRecordKey = RedisConfig.SIGN_NOTIFICATION_KEY.copy().appendKey(DateUtils.formatDate(date));
        redisSetAdapter.add(signRecordKey,userId);
        redisSetAdapter.expire(signRecordKey,60*60*24*2);
    }

    /**
     * 用户在签到或者补签之后对签到排行榜进行更新
     * 如果用户是第一天签到，则不用对之前签到的排行榜进行处理
     * 如果用户是第二天签到，则需要判断前一天签到是在哪个地区，对那个地区的排行榜进行更新
     * @param userId
     */
    private void rankingSignRecord(Long userId) {
        List<SignRecord> signRecordList = signRecordMapper.listUserSignRecord(userId);
        if (CollectionUtils.isNotEmpty(signRecordList)) {
            UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
            if (null == userInfoDTO) {
                return;
            }
            //获得当天用户签到的信息
            SignRecord todaySign = signRecordList.get(0);
            String areaCode = todaySign.getAreaCode();
            if (StringUtils.isNotEmpty(areaCode)) {
                SignLeaderBoard.LeaderBoard leaderBoard = new SignLeaderBoard.LeaderBoard();
                leaderBoard.setUserId(todaySign.getUserId());
                leaderBoard.setCount(todaySign.getSignDay());
                leaderBoard.setNickname(userInfoDTO.getNickname() == null ? "" : userInfoDTO.getNickname());
                leaderBoard.setHeadImg(StringUtils.isEmpty(userInfoDTO.getHeadImg()) ? userIntegrationService.getDefaultHeadImgUrl() : userInfoDTO.getHeadImg());

                //获得当前地区的签到排行榜记录
                KeyGenerator signRecordKey = RedisConfig.SIGN_RANKING_KEY.copy().appendKey(areaCode);
                Set<SignLeaderBoard.LeaderBoard> leaderBoardSet =
                        redisZSetAdapter.range(signRecordKey, 0, 100, true, SignLeaderBoard.LeaderBoard.class);

                //如果在当前地区签到排行榜有用户的信息则将其删除重新加
                Optional<SignLeaderBoard.LeaderBoard> boardOptional = leaderBoardSet.stream().filter(e -> e.getUserId().equals(leaderBoard.getUserId())).findFirst();
                if (boardOptional.isPresent()) {
                    redisZSetAdapter.remove(signRecordKey, boardOptional.get());
                    redisZSetAdapter.add(signRecordKey, leaderBoard, leaderBoard.getCount());
                } else {
                    redisZSetAdapter.add(signRecordKey, leaderBoard, leaderBoard.getCount());
                }
            }
            //如果判断当天签到时连续第二天签到，那么他一定有前一天的签到记录
            if (todaySign.getSignDay() >= 2 && signRecordList.size() >= 2) {
                SignRecord yesterdaySign = signRecordList.get(1);
                String lastAreaCode = yesterdaySign.getAreaCode();
                if (StringUtils.isNotEmpty(lastAreaCode) && !areaCode.equals(lastAreaCode)) {
                    SignLeaderBoard.LeaderBoard leaderBoard = new SignLeaderBoard.LeaderBoard();
                    leaderBoard.setUserId(todaySign.getUserId());

                    //获得前一天签到地区的签到排行榜记录
                    KeyGenerator signRecordKey = RedisConfig.SIGN_RANKING_KEY.copy().appendKey(lastAreaCode);
                    Set<SignLeaderBoard.LeaderBoard> leaderBoardSet =
                            redisZSetAdapter.range(signRecordKey, 0, 100, true, SignLeaderBoard.LeaderBoard.class);

                    Optional<SignLeaderBoard.LeaderBoard> boardOptional = leaderBoardSet.stream().filter(e -> e.getUserId().equals(leaderBoard.getUserId())).findFirst();
                    boardOptional.ifPresent(e -> redisZSetAdapter.remove(signRecordKey, e));
                }
            }
        }
    }

    @Override
    public Message executeUserFillSign(Long userId,String areaCode) {
        Message message = Message.build();
        SignDTO signDTO = generateSign(userId);
        if (!signDTO.isYesterdaySignState()) {
            return message.setSuccess(false).setMessage("当前用户已补签");
        }

        String requestId = String.valueOf(nextId());
        Long id = nextId();

        if (distributedLock.lock(userId + "" + id, requestId)) {
            Date yesterday = DateUtils.addField(new Date(), Calendar.DATE, -1);
            Date date = new Date();
            //获取前天的签到数据
            SignRecord beforeSignRecord = signRecordMapper.getBeforeYesterDaySignRecord(userId);

            //设置连续签到日期
            int signDay = 1;
            if (beforeSignRecord != null) {
                signDay += beforeSignRecord.getSignDay();
            }

            SignRecord signRecord = new SignRecord(id, userId, yesterday, signDay,
                    0L, true,areaCode);
            signRecordMapper.insertSelective(signRecord);

            //判断今天是否签到---签到则更新连续签到的日期
            if (signDTO.isTodaySignState()) {
                signRecordMapper.updateTodaySign(userId, DateUtils.formatDate(date),
                        signDay + 1);
            }
            this.record2redis(userId, date);
            this.rankingSignRecord(userId);
            distributedLock.unlock(userId + "" + id, requestId);
        } else {
            return message.setSuccess(false).setMessage("当前用户已补签");
        }
        return message;
    }

    @Override
    public SignLeaderBoard getSignLeaderBoard(Long userId,String areaCode) {
        if (userId==null || StringUtils.isEmpty(areaCode)) {
            logger.warn("请求签到排行榜参数错误,userId:[{}],areaCode:[{}]", userId,areaCode);
            return null;
        }
        //判断这个地区是否开通了签到排行榜[沈涛]
        LocationDetailDTO locationDetailDTO = locationFacadeService.getLocationDetailByCode(areaCode);
        if (null == locationDetailDTO || null == locationDetailDTO.getEnableSignBoard() || 0 == locationDetailDTO.getEnableSignBoard()) {
            return null;
        }

        KeyGenerator signRecordKey = RedisConfig.SIGN_RANKING_KEY.copy().appendKey(areaCode);
        Set<SignLeaderBoard.LeaderBoard> leaderBoardSet =
                redisZSetAdapter.range(signRecordKey, 0, signProperties.getRankSize()-1, true, SignLeaderBoard.LeaderBoard.class);
        SignLeaderBoard signLeaderBoard = new SignLeaderBoard();
        signLeaderBoard.setUserId(userId);
        int index = 1;
        for (SignLeaderBoard.LeaderBoard e : leaderBoardSet) {
            e.setRanking(index);
            if (signLeaderBoard.getUserId().equals(e.getUserId())) {
                BeanUtils.copyProperties(e, signLeaderBoard);
            }
            index++;
        }
        signLeaderBoard.setLeaderBoardList(leaderBoardSet);
        if (null == signLeaderBoard.getRanking()) {
            UserInfoDTO user = userIntegrationService.getUserFromRedisDB(signLeaderBoard.getUserId());
            if (null != user) {
                signLeaderBoard.setHeadImg(user.getHeadImg());
                signLeaderBoard.setNickname(user.getNickname());
                //用户签到不在排行榜以内则从数据库中取得用户的签到天数(是否是今天且是否在当前地区)
                SignRecord signRecord = signRecordMapper.getTodaySignRecord(userId);
                //今天没签到
                if (null != signRecord) {
                    if (areaCode.equals(signRecord.getAreaCode() == null ? "" : signRecord.getAreaCode())) {
                        signLeaderBoard.setCount(signRecord.getSignDay());
                    }
                } else {
                    SignRecord yesterdaySignRecord = signRecordMapper.getYesterdaySignRecord(userId);
                    if (null != yesterdaySignRecord) {
                        if (areaCode.equals(yesterdaySignRecord.getAreaCode() == null ? "" : yesterdaySignRecord.getAreaCode())) {
                            signLeaderBoard.setCount(yesterdaySignRecord.getSignDay());
                        }
                    }
                }
            }else{
                signLeaderBoard.setHeadImg(userIntegrationService.getDefaultHeadImgUrl());
                signLeaderBoard.setNickname("");
            }
        }

        signLeaderBoard.setLeaderBoardList(leaderBoardSet);
        return signLeaderBoard;
    }

    /**
     * 获取对应连续签到天数的配置信息
     * 备注：连续签到超过7天---签到配置信息为第七天
     *
     * @param signDay
     * @return
     */
    protected SignConfig getSignConfig(int signDay) {
        List<SignConfig> signConfigList = signConfigService.allSignConfig();

        if (signDay < 7) {
            for (SignConfig travSignConfig : signConfigList) {
                if (travSignConfig.getDate() == signDay) {
                    return travSignConfig;
                }
            }

        }

        return signConfigList.stream().filter(signConfig -> signConfig.getDate() == 7)
                .findFirst().get();
    }

    /**
     * 生成用户签到的相关信息
     *
     * @param userId
     * @return
     */
    protected SignDTO generateSign(Long userId) {
        SignDTO signDTO = new SignDTO();
        SignRecord yesterdaySignRecord = signRecordMapper.getYesterdaySignRecord(userId);
        //昨天是否可以补签
        signDTO.setYesterdaySignState(yesterdaySignRecord == null);

        //获取最后一天签到的信息
        SignRecord signRecord = signRecordMapper.getLastSignRecord(userId);

        //判断连续签到可获得的金币
        if (signRecord == null) {
            signDTO.setGold(getSignConfig(1).getReward().intValue());
            signDTO.setCount(0L);
            signDTO.setYesterdaySignState(true);
        } else {
            Date today = new Date();
            //判断今天是否签到
            if (org.apache.commons.lang3.time.DateUtils.isSameDay(signRecord.getSignDate(), today)) {
                signDTO.setTodaySignState(true);
                signDTO.setGold(getSignConfig(signRecord.getSignDay() + 1).getReward().intValue());
                signDTO.setCount(signRecord.getSignDay().longValue());
            } else {
                if (yesterdaySignRecord == null) {
                    signDTO.setGold(getSignConfig(1).getReward().intValue());
                    signDTO.setCount(0L);
                    signDTO.setYesterdaySignState(true);
                } else {
                    signDTO.setGold(getSignConfig(yesterdaySignRecord.getSignDay() + 1).getReward().intValue());
                    signDTO.setCount(yesterdaySignRecord.getSignDay().longValue());
                }
            }
        }

        return signDTO;
    }

    /**
     * 获取对应用户的签到日历
     *
     * @param userId
     * @param signDate
     * @return
     */
    private List<CalendarSignDTO> getMonthSignRecord(Long userId, String signDate) {

        //组装签到日历
        List<CalendarSignDTO> signRecords = generateMonthSign(signDate);
        //获取对应月份的签到信息
        List<SignRecord> signRecordList = signRecordMapper.getMonthSignRecord(userId, signDate);

        signRecords.forEach(calendarSignDTO -> {
            signRecordList.forEach(signRecord -> {
                String signRecordDate = DateUtils.formatAtWill(signRecord.getSignDate(), DateUtils.DATE_FORMAT);
                if (signRecordDate.equals(calendarSignDTO.getSignDate())) {
                    calendarSignDTO.setSignFlag(true);
                    calendarSignDTO.setFill(signRecord.getFillSign());
                }
            });
        });

        return signRecords;
    }

    /**
     * 组装签到日历
     *
     * @return
     */
    private List<CalendarSignDTO> generateMonthSign(String signDate) {
        List<String> monthDates = getDayByMonth(signDate);
        List<CalendarSignDTO> calendarSignDTOS = Lists.newArrayListWithCapacity(monthDates.size());
        for (String monthDate : monthDates) {
            CalendarSignDTO calendarSignDTO = new CalendarSignDTO();
            calendarSignDTO.setSignDate(monthDate);
            calendarSignDTOS.add(calendarSignDTO);
        }
        return calendarSignDTOS;
    }

    /**
     * 根据传入日期获取当前月份的日历
     *
     * @param signDate
     * @return
     */
    private List<String> getDayByMonth(String signDate) {
        int yearParam = Integer.parseInt(StringUtils.split(signDate, "-")[0]);
        int monthParam = Integer.parseInt(StringUtils.split(signDate, "-")[1]);
        List list = Lists.newArrayList();

        Calendar calendar = Calendar.getInstance();
        calendar.clear();
        calendar.set(yearParam, monthParam - 1, 1);
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.getActualMaximum(Calendar.DATE);

        for (int i = 1; i <= day; i++) {
            String monthDate = null;
            if (month < 10 && i < 10) {
                monthDate = String.valueOf(year) + "-0" + month + "-0" + i;
            }
            if (month < 10 && i >= 10) {
                monthDate = String.valueOf(year) + "-0" + month + "-" + i;
            }
            if (month >= 10 && i < 10) {
                monthDate = String.valueOf(year) + "-" + month + "-0" + i;
            }
            if (month >= 10 && i >= 10) {
                monthDate = String.valueOf(year) + "-" + month + "-" + i;
            }

            list.add(monthDate);
        }
        return list;
    }
}
