package com.bxm.localnews.activity.strategy.dailyConcreteStrategy;

import com.bxm.localnews.activity.command.TransferAddGoldCommand;
import com.bxm.localnews.activity.domain.BrowseRecordMapper;
import com.bxm.localnews.activity.dto.NewsMissionRewardDto;
import com.bxm.localnews.activity.param.TaskContext;
import com.bxm.localnews.activity.strategy.AbstractTaskStrategy;
import com.bxm.localnews.activity.vo.DailyTask;
import com.bxm.localnews.common.config.NewsProperties;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.TaskEnum;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.param.AccountGoldParam;
import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.util.Date;

import static com.bxm.localnews.activity.vo.UserMissionModel.TASK_STATE_YES;
import static com.bxm.localnews.common.constant.RedisConfig.*;

/**
 * 日常任务抽象处理类
 *
 * @author shentao
 */
public abstract class AbstractDailyTaskStrategy extends AbstractTaskStrategy {

    protected RedisStringAdapter redisStringAdapter;

    protected RedisHashMapAdapter redisHashMapAdapter;

    private NewsProperties newsProperties;

    protected UserIntegrationService userIntegrationService;

    protected BrowseRecordMapper browseRecordMapper;

    @Autowired(required = false)
    public AbstractDailyTaskStrategy(RedisStringAdapter redisStringAdapter, RedisHashMapAdapter redisHashMapAdapter, NewsProperties newsProperties, UserIntegrationService userIntegrationService, BrowseRecordMapper browseRecordMapper) {
        this.redisStringAdapter = redisStringAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.newsProperties = newsProperties;
        this.userIntegrationService = userIntegrationService;
        this.browseRecordMapper = browseRecordMapper;
    }

    @Override
    protected Message preProcessing(TaskContext taskContext) {
        //每个任务都要进行的预判断，判断是否今日金额已超出
        Message msg = judgeIsOverRewardSum(taskContext);
        if (!msg.isSuccess()) {
            return msg;
        }
        //每个人任务各自的判断
        return customJudge(taskContext);
    }

    /**
     * 每个人任务各自需要的预处理，比如阅读转发文章任务，需要验证参数中用户是否有效等
     */
    protected abstract Message customJudge(TaskContext taskContext);

    /**
     * 判断今日已获得的红花数是否超出限额，3000
     *
     * @param taskContext 任务上下文
     * @return 判断结果
     */
    private Message judgeIsOverRewardSum(TaskContext taskContext) {
        //查询今日红花数是否已达上限
        KeyGenerator userRewardSumKey = RedisConfig.USER_MISSION_REWARD_SUM_PER_DAY.copy().setKey(taskContext.getUserId() + ":" + DateUtils.formatDate(new Date()));
        int totalGoldDay = redisStringAdapter.getInt(userRewardSumKey);
        if (totalGoldDay >= newsProperties.getGoldPerDay()) {
            return Message.build(false, "今日已获得红花数上限!");
        }
        taskContext.addParam("userRewardSumKey", userRewardSumKey);
        return Message.build(true);
    }

    /**
     * 判断比较平常的每日任务是否达到最大次数
     */
    protected Message judgeIsOverRewardCount(TaskContext taskContext) {
        //查到该任务所完成的进度
        KeyGenerator taskCompleteNumKey = getTaskCompleteNumKey(taskContext.getTaskEnum(), taskContext.getUserId());
        int completeNum = redisStringAdapter.getInt(taskCompleteNumKey);

        //如果已经到达最大完成度则返回
        if (completeNum >= taskContext.getDailyTask().getNumber()) {
            return Message.build(false, "今日任务已获得最大次数限制!");
        }

        taskContext.addParam("taskCompleteNumKey", taskCompleteNumKey);
        taskContext.addParam("completeNum", completeNum);
        return Message.build(true);
    }

    /**
     * 日常任务默认处理
     *
     * @param taskContext
     * @return
     */
    protected NewsMissionRewardDto defaultProcess(TaskContext taskContext) {
        long reward;

        Long userId = taskContext.getUserId();
        DailyTask dailyTask = taskContext.getDailyTask();

        //获得上下文的参数
        KeyGenerator taskCompleteNumKey = taskContext.getParam("taskCompleteNumKey");
        KeyGenerator userRewardSumKey = taskContext.getParam("userRewardSumKey");
        int completeNum = taskContext.getParam("completeNum");

        //累加完成次数
        redisStringAdapter.set(taskCompleteNumKey, ++completeNum);
        redisStringAdapter.expire(taskCompleteNumKey, DateUtils.getCurSeconds());
        if (completeNum == dailyTask.getNumber()) {
            //今天的正好按次数完成了任务
            completeDailyTask(userId, taskContext.getTaskEnum().getType());

            //当前任务完成后是否需要推送
            if (isSupportPushMessage()) {
                taskContext.setPushMsg(true);
            }
        }

        //获得每个任务所能奖励的红花数，大多任务都是固定的，部分任务需要通过计算获得奖励
        reward = getTaskReward(taskContext);

        //将实际奖励红花数放入上下文中
        dailyTask.setReward(BigDecimal.valueOf(reward));

        //统计当天完成任务获得的小红花总数，加上任务获得的小红花总数
        redisStringAdapter.incrementWithDefault(userRewardSumKey, reward, Math.toIntExact(reward));
        redisStringAdapter.expire(userRewardSumKey, DateUtils.getCurSeconds());

        //任务支持增加金币命令
        taskContext.setAddGold(true);

        return convertTask(taskContext.getTaskEnum(), reward, userId);
    }

    @Override
    protected void postProcessing(TaskContext taskContext) {

        // 金币增加
        if (taskContext.getAddGold()) {
            afterCompleteToAddGold(taskContext);
        }

        // 推送消息
        if (taskContext.getPushMsg() || taskContext.getPersistenceMsg()) {
            afterCompleteToPushMsg(taskContext);
        }
    }

    /**
     * 获得任务完成的数量，在redis中记录
     *
     * @param taskEnum
     * @param userId
     * @return
     */
    private KeyGenerator getTaskCompleteNumKey(TaskEnum taskEnum, Long userId) {
        KeyGenerator taskCompleteNumKey = null;
        switch (taskEnum) {
            case TASK_NEWS_READ:
                taskCompleteNumKey = NEWS_READ.copy().setKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                break;
            case TASK_NEWS_SHARE:
                taskCompleteNumKey = TASK_SHARE_NEWS_NUM.copy().appendKey(userId.toString()).appendKey(DateUtils.formatDate(new Date()));
                break;
            case TASK_VIDEO_READ:
                taskCompleteNumKey = VIDEO_READ_NUM.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                break;
            case TASK_FIRST_POST_INTIVATION:
            case TASK_POST_INTIVATION:
                taskCompleteNumKey = POST_FORUM.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                break;
            case TASK_COMMENT_NEWS:
                taskCompleteNumKey = NEWS_COMMENT.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                break;
            case TASK_EVERYDAY_SHARE:
                taskCompleteNumKey = NEWS_SHARE.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                break;
            case TASK_SIGN_SHARE:
                taskCompleteNumKey = SIGN_SHARE_NUM.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                break;
            case TASK_WATCH_VIDEO:
                taskCompleteNumKey = WATCH_VIDEO.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                break;
            default:
                logger.error("TASK-ERR--userId:" + userId + ",taskName:" + taskEnum.name());
                break;
        }
        return taskCompleteNumKey;
    }

    /**
     * 完成每日任务，记录在redis中
     *
     * @param userId
     * @param taskId
     */
    private void completeDailyTask(Long userId, Byte taskId) {
        Assert.notNull(taskId, "任务ID不能为空");
        logger.info(LogMarker.BIZ, "[{}]完成了任务[{}]", userId, taskId);

        KeyGenerator key = getDailyTaskCacheKey(userId);

        redisHashMapAdapter.put(key, taskId.toString(), TASK_STATE_YES);
        redisHashMapAdapter.expire(key, DateUtils.getCurSeconds());
    }

    private KeyGenerator getDailyTaskCacheKey(Long userId) {
        return DAILY_TASK_COMPELE_STATUS.copy().setKey(DateUtils.formatDate(new Date()) + userId);
    }

    /**
     * 增加小红花
     */
    private void afterCompleteToAddGold(TaskContext taskContext) {
        //添加金币
        AccountGoldParam param = AccountGoldParam.buildAccountGoldParam(taskContext.getUserId(), taskContext.getDailyTask().getReward().intValue(),
                taskContext.getRelationId() == null ? null : Long.valueOf(taskContext.getRelationId()), taskContext.getTaskEnum().name(), taskContext.getContent());
        TransferAddGoldCommand transferAddGoldCommand = TransferAddGoldCommand.buildTransferAddGoldCommand(param);
        taskInvokeThreadLocal.get().setCommand(transferAddGoldCommand);
    }

    /**
     * 是否支持推送消息
     * 其中分享内容不推送消息
     *
     * @return
     */
    protected abstract boolean isSupportPushMessage();

    /**
     * 得到每个任务的奖励金额
     *
     * @return
     */
    protected long getTaskReward(TaskContext taskContext) {
        return 0;
    }

}
