package com.bxm.localnews.activity.stream;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.action.stream.MessageProcessor;
import com.bxm.localnews.activity.common.constant.RankEnum;
import com.bxm.localnews.activity.rank.RankSelfService;
import com.bxm.localnews.activity.vo.InteractRankInfo;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.dto.UserInviteOrderVO;
import com.bxm.localnews.user.event.RedundancyActionEvent;
import com.bxm.localnews.vo.ForumRankVo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.stream.annotation.StreamListener;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 *
 *  监听y用户信息变更 更新榜单头像和昵称
 *
 * @author pf.w
 * @date 2020/9/3 14:53
 **/
@Component
@Slf4j
public class MessageSubscribe {

    @Autowired
    RankSelfService rankSelfService;

    @Resource
    RedisHashMapAdapter redisHashMapAdapter;

    @StreamListener(MessageProcessor.REDUNDANCY_ACTION)
    public void syncRankForHeadImgAndNickname(RedundancyActionEvent event){
        if (log.isDebugEnabled()) {
            log.debug("接受到用户冗余信息变更 ： {}", JSON.toJSONString(event));
        }

        if (Objects.nonNull(event)) {
            this.syncUserInfo(event.getUserId(), event.getHeadImg(), event.getNickName());
        }
    }

    private void syncUserInfo(Long userId,String headImg,String nickName){
        //更新发帖榜
        updateForumRank(userId,headImg,nickName);

        //更新互动榜
        updateInteractRank(userId,headImg,nickName);

        //更新邀请榜
        updateInviteRank(userId,headImg,nickName);
    }

    /**
     * 更新发帖榜
     * @param userId 用户id
     * @param headImg 头像
     * @param nickName 昵称
     */
    private void updateForumRank(Long userId,String headImg,String nickName){
        KeyGenerator forumKey =  RedisConfig.RANK_LIST.copy().appendKey(RankEnum.FORUM.name().toLowerCase());
        Set<String> areaCodes = redisHashMapAdapter.keys(forumKey);

        TypeReference<List<ForumRankVo>> typeReference = new TypeReference<List<ForumRankVo>>() {
        };
        for (String areaCode : areaCodes) {
            List<ForumRankVo> forumRankVoList = redisHashMapAdapter.get(buildKey(RankEnum.FORUM), areaCode, typeReference);
            if(CollectionUtils.isEmpty(forumRankVoList)){
                continue;
            }

            boolean flag = false;
            for (ForumRankVo forumRankVo : forumRankVoList) {
                if(Objects.equals(forumRankVo.getUserId(),userId)){
                    forumRankVo.setUserImg(headImg);
                    forumRankVo.setUserName(nickName);
                    flag = true;
                }
            }

            if(flag){
                redisHashMapAdapter.put(buildKey(RankEnum.FORUM), areaCode, forumRankVoList);
                redisHashMapAdapter.expire(buildKey(RankEnum.FORUM), DateUtils.DAY_MILLISECOND / 1000);
            }
        }
    }

    /**
     * 更新互动榜
     * @param userId 用户id
     * @param headImg 头像
     * @param nickName 昵称
     */
    private void updateInteractRank(Long userId,String headImg,String nickName){
        //更新互动榜
        KeyGenerator interactKey =  RedisConfig.RANK_LIST.copy().appendKey(RankEnum.INTERACT.name().toLowerCase());
        Set<String> areaCodes = redisHashMapAdapter.keys(interactKey);

        TypeReference<List<InteractRankInfo>> typeReference = new TypeReference<List<InteractRankInfo>>() {
        };
        for (String areaCode : areaCodes) {
            List<InteractRankInfo> interactRankInfos = redisHashMapAdapter.get(buildKey(RankEnum.INTERACT), areaCode, typeReference);
            if(CollectionUtils.isEmpty(interactRankInfos)){
                continue;
            }

            boolean flag = false;
            for (InteractRankInfo interactRankInfo : interactRankInfos) {
                if(Objects.equals(interactRankInfo.getId(),userId)){
                    interactRankInfo.setHeadImg(headImg);
                    interactRankInfo.setNickName(nickName);
                    flag = true;
                }
            }

            if(flag){
                redisHashMapAdapter.put(buildKey(RankEnum.INTERACT), areaCode, interactRankInfos);
                redisHashMapAdapter.expire(buildKey(RankEnum.INTERACT), DateUtils.DAY_MILLISECOND / 1000);
            }
        }

    }
    /**
     * 更新邀请榜
     * @param userId 用户id
     * @param headImg 头像
     * @param nickName 昵称
     */
    private void updateInviteRank(Long userId,String headImg,String nickName){
        KeyGenerator inviteKey =  RedisConfig.RANK_LIST.copy().appendKey(RankEnum.INVITE.name().toLowerCase());
        Set<String> areaCodes = redisHashMapAdapter.keys(inviteKey);

        TypeReference<List<UserInviteOrderVO>> typeReference = new TypeReference<List<UserInviteOrderVO>>() {
        };
        for (String areaCode : areaCodes) {
            List<UserInviteOrderVO> inviteRankInfos = redisHashMapAdapter.get(buildKey(RankEnum.INVITE), areaCode,
                    typeReference);
            if(CollectionUtils.isEmpty(inviteRankInfos)){
                continue;
            }
            boolean flag = false;
            for (UserInviteOrderVO inviteRankInfo : inviteRankInfos) {
                if(Objects.equals(inviteRankInfo.getUserId(),userId)){
                    inviteRankInfo.setHeadImg(headImg);
                    inviteRankInfo.setNickname(nickName);
                    flag = true;
                }
            }

            if(flag){
                redisHashMapAdapter.put(buildKey(RankEnum.INVITE), areaCode, inviteRankInfos);
                redisHashMapAdapter.expire(buildKey(RankEnum.INVITE), DateUtils.DAY_MILLISECOND / 1000);
            }
        }

    }

    /**
     * 缓存key值
     *
     * @param rankEnum 榜单类型
     * @return key
     */
    private KeyGenerator buildKey(RankEnum rankEnum) {
        return RedisConfig.RANK_LIST.copy().appendKey(rankEnum.name().toLowerCase());
    }
}
