package com.bxm.localnews.activity.vote.strategy.impl;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.activity.domain.VoteChoiceCountMapper;
import com.bxm.localnews.activity.domain.VoteChoiceRecordMapper;
import com.bxm.localnews.activity.dto.VoteDetailDTO;
import com.bxm.localnews.activity.param.VoteParam;
import com.bxm.localnews.activity.param.VotePinParam;
import com.bxm.localnews.activity.vo.VoteChoiceCountBean;
import com.bxm.localnews.activity.vo.VoteChoiceRecordBean;
import com.bxm.localnews.activity.vote.strategy.IVoteStrategy;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.List;
import java.util.Map;

import static com.bxm.localnews.common.constant.RedisConfig.LAST_VOTE_KEY;
import static com.bxm.localnews.common.constant.RedisConfig.VOTE_OPTIONS_KEY;

@Slf4j
public abstract class AbstractVoteStrategy implements IVoteStrategy {

    /**
     * 今日投票总数
     */
    static final String VOTE_TOTAL = "vt";

    /**
     * 剩余投票次数
     */
    static final String TIMES = "t";

    @Autowired
    protected RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    protected RedisStringAdapter redisStringAdapter;

    @Autowired
    private SequenceCreater sequenceCreater;

    /**
     * 用户投票相关缓存键值生成
     *
     * @param userId 用户ID
     * @return 用户投票缓存键值
     */
    abstract KeyGenerator buildKey(Long userId, Long voteId);

    /**
     * 用户投票相关缓存过期时间设置
     *
     * @param key     缓存key
     * @param endTime 投票活动截止时间
     */
    abstract void expired(KeyGenerator key, Date endTime);

    @Override
    public Message addTime(VoteDetailDTO detail, VotePinParam param) {
        KeyGenerator key = buildKey(param.getUserId(), param.getVoteId());

        Map<String, Long> userVoteMap = redisHashMapAdapter.entries(key, Long.class);

        Long voteTotal = userVoteMap.get(VOTE_TOTAL);
        Long times = userVoteMap.get(TIMES);

        //初始有一票
        long max = detail.getMax() + 1;

        if (null != voteTotal && voteTotal >= max) {
            log.info("新增投票次数，但已达到总数上限，业务逻辑有问题，需要排查。参数：[{}]", param);
            return Message.build(false, "已达到投票总数上限");
        }

        if (null != times && times >= max) {
            return Message.build(false, "今日可获得的投票数量达到上限");
        }

        //增加可投票次数，到达失效时间后自动失效
        redisHashMapAdapter.increment(key, TIMES, 1);
        expired(key, detail.getEndTime());
        return Message.build();
    }

    @Override
    public Message vote(VoteDetailDTO detail, VoteParam param) {
        KeyGenerator key = buildKey(param.getUserId(), param.getVoteId());

        Map<String, Long> userVoteMap = redisHashMapAdapter.entries(key, Long.class);

        //判断用户是否还有可投票次数
        Long times = userVoteMap.get(TIMES);
        if (null != times && times == 0) {
            log.info("用户已无投票次数，投票动作仍被调用。参数：[{}]", param);
            return Message.build(false, "用户没有投票次数");
        }

        if (times == null) {
            redisHashMapAdapter.put(key, TIMES, 0);
        } else {
            redisHashMapAdapter.increment(key, TIMES, -1);
        }
        redisHashMapAdapter.increment(key, VOTE_TOTAL, 1);
        expired(key, detail.getEndTime());

        //保存投票结果
        saveVote(param);
        return Message.build();
    }

    /**
     * 更新缓存数据
     * 将投票结果序列化到数据库
     *
     * @param param 投票参数
     */
    private void saveVote(VoteParam param) {
        String[] optionIdArray = StringUtils.split(param.getOptionIds(), ",");

        //保存用户最后的投票记录
        KeyGenerator lastVoteKey = LAST_VOTE_KEY.copy().appendKey(param.getVoteId()).appendKey(param.getUserId());
        redisStringAdapter.set(lastVoteKey, param.getOptionIds());

        KeyGenerator voteOptionsKey = VOTE_OPTIONS_KEY.copy().appendKey(param.getVoteId());
        List<VoteChoiceCountBean> choiceCountList = Lists.newArrayList();
        List<VoteChoiceRecordBean> choiceRecordList = Lists.newArrayList();

        for (String optionId : optionIdArray) {
            //缓存中的选项增加投票数量
            redisHashMapAdapter.increment(voteOptionsKey, optionId, 1);

            choiceCountList.add(VoteChoiceCountBean.builder()
                    .voteId(param.getVoteId())
                    .relationId(param.getRelationId())
                    .optionId(Long.valueOf(optionId))
                    .build());

            choiceRecordList.add(VoteChoiceRecordBean.builder()
                    .createTime(new Date())
                    .id(sequenceCreater.nextLongId())
                    .optionId(Long.valueOf(optionId))
                    .relationId(param.getRelationId())
                    .userId(param.getUserId())
                    .voteId(param.getVoteId())
                    .build());
        }

        MybatisBatchBuilder.create(VoteChoiceCountMapper.class, choiceCountList).run(VoteChoiceCountMapper::updateCount);
        MybatisBatchBuilder.create(VoteChoiceRecordMapper.class, choiceRecordList).run(VoteChoiceRecordMapper::insert);
    }

    void setNotStart(VoteDetailDTO detail) {
        detail.setStatus(ButtonStatus.DISABLE.name());
        detail.setSubTitle(null);
        detail.setAction(Action.VOTE.name());
    }

    void setExpired(VoteDetailDTO detail) {
        detail.setStatus(ButtonStatus.DISABLE.name());
        detail.setStatusLabel("投票活动已结束");
        detail.setSubTitle(null);
        detail.setAction(Action.SHARE.name());
    }

    void setVote(VoteDetailDTO detail) {
        detail.setStatus(ButtonStatus.ENABLE.name());
        detail.setStatusLabel("投票");
        detail.setSubTitle(null);
        detail.setAction(Action.VOTE.name());

        //清楚用户最后选择的结果
        detail.getOptions().forEach(item -> item.setChecked(false));
    }

    protected enum Action {
        /**
         * 发起投票
         */
        VOTE,
        /**
         * 发起分享
         */
        SHARE
    }

    protected enum ButtonStatus {
        /**
         * 可点击
         */
        ENABLE,
        /**
         * 不可点击
         */
        DISABLE
    }
}
