package com.bxm.localnews.admin.controller;

import java.util.List;

import javax.annotation.Resource;

import com.bxm.localnews.admin.domain.AdminRoleMenuMapper;
import com.bxm.localnews.admin.dto.AdminMenuDTO;
import com.bxm.localnews.admin.service.AdminMenuService;
import com.bxm.localnews.admin.service.AdminRoleMenuService;
import com.bxm.localnews.admin.vo.AdminMenu;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.tools.StringUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

@Api(tags = "9-11 [管理]菜单管理", description = "菜单栏相关操作")
@Controller
@RequestMapping("api/admin/menu")
public class AdminMenuController {

    @Resource
    private AdminMenuService adminMenuService;

    @Resource
    private AdminRoleMenuService adminRoleMenuService;

    @ApiOperation(value = "9-11-1 根据获取用户角色拥有的菜单", notes = "")
    @ApiImplicitParam(name = "roleId", value = "角色id")
    @GetMapping("getRoleMenu")
    @ResponseBody
    public Json<List<AdminMenuDTO>> getRoleMenu(Integer roleId) {
        if (roleId == null) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数为空");
        }
        return ResultUtil.genSuccessResult(this.adminMenuService.getMenusByRoleId(roleId));
    }

    @RequiresPermissions(value = "admin:menu")
    @ApiOperation(value = "9-11-2 根据角色获取菜单列表", notes = "")
    @ApiImplicitParam(name = "roleId", value = "角色id")
    @GetMapping("getUserMenu")
    @ResponseBody
    public Json<List<AdminMenuDTO>> getMenuByUserId(Long roleId) {
        List<AdminMenuDTO> allMenus = this.adminMenuService.getMenusByRole(roleId);
        return ResultUtil.genSuccessResult(allMenus);
    }

    /**
     * 获取所有菜单列表
     * @return
     */
    @RequiresPermissions(value = "admin:menu")
    @ApiOperation(value = "9-11-3 获取所有菜单列表", notes = "")
    @GetMapping("getAllMenu")
    @ResponseBody
    public Json<List<AdminMenuDTO>> getAllMenu() {
        List<AdminMenuDTO> allMenus = this.adminMenuService.getAllMenu();
        return ResultUtil.genSuccessResult(allMenus);
    }

    /**
     * 根据id获取菜单信息
     */
    @ApiOperation(value = "9-11-4 获取所有菜单列表", notes = "")
    @ApiImplicitParam(name = "menuId", value = "菜单id")
    @ResponseBody
    @GetMapping("getMenuInfo")
    // @RequiresPermissions(value = "admin:menu")
    public Json<AdminMenu> getMenuInfo(@RequestParam(value = "menuId", required = true) Integer menuId) {
        AdminMenu menu = this.adminMenuService.selectByPrimaryKey(menuId);
        return ResultUtil.genSuccessResult(menu);
    }

    /**
     * 获取父节点菜单
     * @return
     */
    @ApiOperation(value = "9-11-5 获取父节点菜单", notes = "")
    @GetMapping("getParentMenuList")
    @ResponseBody
    public Json<List<AdminMenu>> getParentMenuList() {
        return ResultUtil.genSuccessResult(this.adminMenuService.getParentMenus());
    }

    @RequiresPermissions(value = "admin:menu")
    @ApiOperation(value = "9-11-6 保存菜单", notes = "")
    @RequestMapping(value = "save", method = RequestMethod.POST)
    @ResponseBody
    public Json saveMenu(@RequestBody AdminMenu adminMenu) {
        if (StringUtils.isEmpty(adminMenu.getName())
                || (adminMenu.getLevel() == 1 && StringUtils.isEmpty(adminMenu.getPath()))) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "菜单名称为空或子菜单路径为空");
        }
        int flag = this.adminMenuService.upsert(adminMenu);
        if (flag == 1 || flag > 1) {
            return ResultUtil.genSuccessMsg();
        }
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "数据库异常");
    }

    @RequiresPermissions(value = "admin:menu")
    @ApiOperation(value = "删除菜单", notes = "")
    @ApiImplicitParam(name = "menuId", value = "菜单id", required = true)
    @PostMapping(value = "delete")
    @ResponseBody
    public Json delete(Integer menuId) {
        //TODO 删除菜单的时候需确认用户是否具有权限
        Integer userNum = this.adminRoleMenuService.countByMenuId(menuId);
        if (userNum > 0) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "该菜单关联其他角色，不能删除");
        }
        this.adminMenuService.deleteByPrimaryKey(menuId);
        return ResultUtil.genSuccessMsg();
    }

}
