package com.bxm.localnews.admin.controller;

import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.quartz.service.ScheduleJobService;
import com.bxm.newidea.component.quartz.vo.ScheduleJobBean;
import com.bxm.newidea.component.quartz.vo.ScheduleJobPageParam;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import static com.bxm.localnews.common.constant.AppConst.STATE_DISABLE;
import static com.bxm.localnews.common.constant.AppConst.STATE_USE;

@Api(tags = "1-12 [管理]定时任务", description = "定时任务的启动、暂停和执行周期修改")
@RestController
@RequestMapping("api/admin/quartz")
public class AdminQuartzController extends BaseController {

    private final ScheduleJobService scheduleJobService;

    @Autowired
    public AdminQuartzController(ScheduleJobService scheduleJobService) {
        this.scheduleJobService = scheduleJobService;
    }

    @ApiOperation(value = "分页获取定时任务列表")
    @GetMapping("page")
    @RequiresPermissions("admin:quartz:list")
    public Json<PageWarper<ScheduleJobBean>> page(ScheduleJobPageParam param) {
        return Json.build(scheduleJobService.selectByPageNumSize(param));
    }

    @ApiOperation(value = "修改定时任务状态,启用、禁用、执行周期修改")
    @RequiresPermissions("admin:quartz:list")
    @PostMapping("editState")
    public Json editState(ScheduleJobBean jobBean) {
        if (STATE_DISABLE.toString().equals(jobBean.getStatus())) {
            scheduleJobService.pauseScheduleJob(jobBean.getJobName(), jobBean.getJobGroup());
        } else if (STATE_USE.toString().equals(jobBean.getStatus())) {
            scheduleJobService.resumeScheduleJob(jobBean.getJobName(), jobBean.getJobGroup());
        }

        return ResultUtil.genSuccessMsg();
    }

    @ApiOperation("修改定时任务执行周期")
    @RequiresPermissions("admin:quartz:list")
    @PostMapping("edit")
    public Json edit(ScheduleJobBean jobBean) {
        if (StringUtils.isNotBlank(jobBean.getCronExpression())) {
            scheduleJobService.updateScheduleJob(jobBean);
        }

        return ResultUtil.genSuccessMsg();
    }

    @ApiOperation(value = "立即运行定时任务")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "jobName", value = "任务名称"),
            @ApiImplicitParam(name = "jobGroup", value = "任务分组")
    })
    @PostMapping("run")
    @RequiresPermissions("admin:quartz:list")
    public Json run(@RequestParam String jobName, @RequestParam String jobGroup) {
        scheduleJobService.runOnce(jobName, jobGroup);
        return ResultUtil.genSuccessMsg();
    }

    @ApiOperation(value = "获取定时任务详细信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "jobName", value = "任务名称"),
            @ApiImplicitParam(name = "jobGroup", value = "任务分组")
    })
    @GetMapping("findById")
    @RequiresPermissions("admin:quartz:list")
    public ScheduleJobBean findById(@RequestParam String jobName, @RequestParam String jobGroup) {
        return scheduleJobService.selectByJobNameAndJobGroup(jobName, jobGroup);
    }

}
