package com.bxm.localnews.admin.controller.activity;

import com.bxm.localnews.admin.common.BizConfigProperties;
import com.bxm.localnews.admin.dto.CodeTemplate;
import com.bxm.localnews.admin.param.PushCodeParam;
import com.bxm.localnews.admin.service.activity.OfflinePushCodeService;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.Date;
import java.util.List;

@Api(tags = "1-43 [管理]线下地推码生成", description = "线下推广生成对应地区的二维码")
@RestController
@RequestMapping("api/admin/push/code")
@Slf4j
@RefreshScope
public class OfflinePushCodeController {

    @Autowired
    private OfflinePushCodeService offlinePushCodeService;

    @Autowired
    private BizConfigProperties bizConfigProperties;

    @ApiOperation(value = "1-43-01 生成线下地推码", notes = "根据模板类型生成对应的线下地推码")
    @GetMapping
    public void generatePushCode(PushCodeParam pushCodeParam, HttpServletResponse response) {
        byte[] data = offlinePushCodeService.generatePushCode(pushCodeParam);
        try {
            String zipName = DateUtils.DATE_TIME_SECOND_FORMAT_THREAD_LOCAL.get().format(new Date()) + ".zip";
            response.setContentType("application/octet-stream;charset=utf-8");
            response.setHeader("Content-Disposition", "attachment; filename=" +
                    new String(zipName.getBytes("GB2312"), "ISO-8859-1"));
            OutputStream outputStream = response.getOutputStream();
            InputStream inputStream = new ByteArrayInputStream(data);
            int len;
            byte[] buf = new byte[1024];
            while ((len = inputStream.read(buf)) != -1) {
                outputStream.write(buf, 0, len);
            }
            outputStream.flush();

            outputStream.close();
            deleteTempDir();
        } catch (IOException e) {
            log.error("zip包输出到客户端异常:", e);
        }
    }

    @ApiOperation(value = "1-43-02 获取线下地推码模板", notes = "返回模板列表")
    @GetMapping("/template")
    public Json<List<CodeTemplate>> getCodeTemplate() {
        return ResultUtil.genSuccessResult(this.offlinePushCodeService.getCodeTemplates());
    }

    @ApiOperation(value = "1-43-03 绑定线下推广码", notes = "用户绑定线下推广码")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", required = true),
            @ApiImplicitParam(name = "remark", value = "备注"),
            @ApiImplicitParam(name = "code", value = "推广码", required = true),
            @ApiImplicitParam(name = "association", value = "0: 否 1：是", required = true)
    })
    @PostMapping("/bind")
    public Json<Boolean> bindPushCode(@RequestParam("userId") Long userId, @RequestParam("code") Long code,
                                      @RequestParam("association") Byte association,
                                      @RequestParam(value = "remark", required = false) String remark) {
        if (!offlinePushCodeService.userIsExist(userId)) {
            return ResultUtil.genFailedResult("用户不存在");
        }
        Message message = this.offlinePushCodeService.bindPushCode(userId, association, remark, code);
        return ResultUtil.genSuccessResult(message.isSuccess());
    }

    @ApiOperation(value = "1-43-04 查看用户是否存在", notes = "绑定地推码的时候判断用户是否存在")
    @GetMapping("/exist")
    public Json<String> userIsExist(Long userId) {
        return ResultUtil.genSuccessResult(this.offlinePushCodeService.getUserNickname(userId));
    }

    /**
     * 删除文件或者目录
     */
    private void deleteTempDir() throws IOException {
        File dirFile = new File(bizConfigProperties.getTempDir());
        FileUtils.deleteDirectory(dirFile);
    }
}
