package com.bxm.localnews.admin.controller.security;

import com.bxm.localnews.admin.dto.AdminMenuDTO;
import com.bxm.localnews.admin.service.security.AdminMenuService;
import com.bxm.localnews.admin.service.security.AdminRoleMenuService;
import com.bxm.localnews.admin.vo.AdminMenu;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

@Api(tags = "9-05 [管理]菜单管理", description = "菜单栏相关操作", hidden = true)
@RestController
@RequestMapping("api/admin/menu")
public class AdminMenuController {

    @Resource
    private AdminMenuService adminMenuService;

    @Resource
    private AdminRoleMenuService adminRoleMenuService;

    @ApiOperation(value = "9-5-1 根据用户角色获取拥有的菜单", hidden = true)
    @ApiImplicitParam(name = "roleId", value = "角色id")
    @GetMapping("getRoleMenu")
    public Json<List<AdminMenuDTO>> getRoleMenu(Integer roleId) {
        if (roleId == null) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数为空");
        }
        return ResultUtil.genSuccessResult(this.adminMenuService.getMenusByRoleId(roleId));
    }

    @ApiOperation(value = "9-5-2 根据角色获取菜单列表", hidden = true)
    @ApiImplicitParam(name = "roleId", value = "角色id")
    @GetMapping("getUserMenu")
    public Json<List<AdminMenuDTO>> getMenuByUserId(Long roleId) {
        List<AdminMenuDTO> allMenus = this.adminMenuService.getMenusByRole(roleId);
        return ResultUtil.genSuccessResult(allMenus);
    }

    /**
     * 获取所有菜单列表
     *
     * @return
     */
    @ApiOperation(value = "9-5-3 获取所有菜单列表")
    @GetMapping("getAllMenu")
    public Json<List<AdminMenuDTO>> getAllMenu() {
        List<AdminMenuDTO> allMenus = this.adminMenuService.getAllMenu();
        return ResultUtil.genSuccessResult(allMenus);
    }

    /**
     * 根据id获取菜单信息
     */
    @ApiOperation(value = "9-5-4 获取菜单详情")
    @ApiImplicitParam(name = "menuId", value = "菜单id")
    @GetMapping("getMenuInfo")
    public Json<AdminMenu> getMenuInfo(@RequestParam(value = "menuId") Integer menuId) {
        AdminMenu menu = this.adminMenuService.selectByPrimaryKey(menuId);
        return ResultUtil.genSuccessResult(menu);
    }

    @ApiOperation(value = "9-5-5 获取父节点菜单")
    @GetMapping("getParentMenuList")
    public Json<List<AdminMenu>> getParentMenuList() {
        return ResultUtil.genSuccessResult(this.adminMenuService.getParentMenus());
    }

    @ApiOperation(value = "9-5-6 保存菜单")
    @PostMapping(value = "save")
    public Json saveMenu(@RequestBody AdminMenu adminMenu) {
        int num = 0;

        if (adminMenu.getParentId() == null || adminMenu.getParentId() == 0) {
            adminMenu.setLevel(0);
        }

        if (adminMenu.getMenuId() == null || adminMenu.getMenuId() == 0) {
            num = adminMenuService.insert(adminMenu);
        } else {
            num = this.adminMenuService.update(adminMenu);
        }

        if (num > 0) {
            return ResultUtil.genSuccessMsg();
        }
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "数据不存在或发生异常");
    }

    @ApiOperation(value = "9-5-7 删除菜单")
    @ApiImplicitParam(name = "menuId", value = "菜单id", required = true)
    @PostMapping(value = "delete")
    public Json delete(Integer menuId) {
        Integer userNum = this.adminRoleMenuService.countByMenuId(menuId);
        if (userNum > 0) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "该菜单关联其他角色，不能删除");
        }
        this.adminMenuService.deleteByPrimaryKey(menuId);
        return ResultUtil.genSuccessMsg();
    }

}
