package com.bxm.localnews.admin.controller.security;

import com.bxm.localnews.admin.controller.CommonController;
import com.bxm.localnews.admin.dto.AdminUserDTO;
import com.bxm.localnews.admin.dto.CurrentUserVo;
import com.bxm.localnews.admin.dto.LoginSuccessWarper;
import com.bxm.localnews.admin.param.AdminUserParam;
import com.bxm.localnews.admin.service.security.*;
import com.bxm.localnews.admin.vo.security.AdminRole;
import com.bxm.localnews.admin.vo.security.AdminUser;
import com.bxm.localnews.admin.vo.security.AdminUserRole;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.List;
import java.util.Optional;

import static com.bxm.newidea.component.tools.MD5Util.hgmd5;
import static org.apache.commons.lang3.StringUtils.isEmpty;

/**
 * Created by hanshuaiqi 13:33 2017/9/22
 */
@Api(tags = "1-16 [管理]后台用户管理接口", description = "包含用户登录、登出等功能")
@RestController
@RequestMapping("api/admin")
public class AdminUserController extends CommonController {

    @Resource
    private AdminUserService adminUserService;

    @Resource
    private AdminRoleService adminRoleService;

    @Resource
    private AdminUserRoleService adminUserRoleService;

    @Resource
    private AdminMenuService adminMenuService;

    @Resource
    private JwtTokenService jwtTokenService;


    @ApiOperation(value = "1-16-1 登出", notes = "")
    @ApiImplicitParam(name = "userName", value = "登出的用户账号")
    @PostMapping(value = "/logout")
    public Json logout(@RequestParam String userName, HttpServletRequest request, HttpServletResponse response) {
        //清除缓存中的用户信息
        AdminUser user = adminUserService.getUser(userName);
        if (user != null) {
            adminUserService.removeToken(user.getId());
        }
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation(value = "1-16-2 用户登录", notes = "后续此token必须在请求中携带")
    @PostMapping(value = "toLoginNew")
    public Json<LoginSuccessWarper> loginAjax(AdminUserDTO sysUser) {
        if (isEmpty(sysUser.getUsername()) || isEmpty(sysUser.getPassword())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误,用户名、密码必须输入");
        }

        AdminUser adminUser = this.adminUserService.getUser(sysUser.getUsername());
        if (null == adminUser || adminUser.getStatus() != 0) {
            return ResultUtil.genFailedResult("该用户已被禁用");
        }

        //判断用户名、密码
        String encodePwd = hgmd5(sysUser.getPassword());
        if (StringUtils.notEquals(encodePwd, adminUser.getPassword())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "用户名、密码错误");
        }

        //创建token
        String token = this.jwtTokenService.generateToken(adminUser);

        adminUserService.addTokenCache(adminUser.getId(), token);

        LoginSuccessWarper warper = new LoginSuccessWarper();
        warper.setToken(token);
        warper.setUserName(adminUser.getUsername());
        warper.setRealName(adminUser.getName());
        //返回用户id
        warper.setUserId(adminUser.getId());

        Optional<Integer> roleId = adminRoleService.selectByUserId(adminUser.getId()).stream().map(AdminRole::getId).findFirst();

        if (roleId.isPresent()) {
            warper.setRoleList(adminMenuService.getMenuAdapter(roleId.get()));
        } else {
            logger.error("用户[{}]没有配置对应的角色，菜单为空", adminUser.getId());
        }

        return ResultUtil.genSuccessResult(warper).setErrorMsg("登陆成功");
    }

    @ApiOperation(value = "1-16-3 修改密码", notes = "")
    @PostMapping(value = "modifyPassword")
    public Json modifyPassword(AdminUserDTO sysUser) {
        return this.adminUserService.modifyPassword(sysUser);
    }

    @ApiOperation(value = "1-16-4 管理后台用户列表", notes = "")
    @GetMapping("list")
    @ResponseBody
    public Json<PageWarper<AdminUserDTO>> getAdminUserList(AdminUserParam adminUserParam) {

        return ResultUtil.genSuccessResult(this.adminUserService.queryAdminUsers(adminUserParam));
    }

    @ApiOperation(value = "1-16-5 保存用户", notes = "")
    @PostMapping(value = "save")
    public Json saveAdminUser(AdminUserDTO adminUserDTO) {
        if (isEmpty(adminUserDTO.getName())
                || isEmpty(adminUserDTO.getUsername())
                || isEmpty(adminUserDTO.getPhone())
                || isEmpty(adminUserDTO.getRoleId())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数为空");
        }

        //判断用户名是否存在
        AdminUser adminUser = this.adminUserService.selectByUsername(adminUserDTO.getUsername().trim());
        if (adminUserDTO.getId() != null) {
            if (adminUser != null) {
                if (!adminUser.getId().equals(adminUserDTO.getId())) {
                    return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "用户名重复");
                }
            }
            adminUserDTO.setUpdateTime(new Date());
            adminUserDTO.setUpdateUser("system");
            this.adminUserService.updateByPrimaryKeySelective(adminUserDTO);
        } else {
            if (adminUser != null) {
                return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "用户名重复");
            }
            this.adminUserService.upsert(adminUserDTO);
        }

        //增加角色
        String[] roleIds = adminUserDTO.getRoleId().split(",");
        this.adminUserRoleService.deleteByUserId(adminUserDTO.getId());

        for (String roleId : roleIds) {
            Integer intRoleId = Integer.parseInt(roleId);
            AdminRole adminRole = this.adminRoleService.selectByPrimaryKey(intRoleId);
            if (adminRole != null) {
                AdminUserRole adminUserRole = new AdminUserRole();
                adminUserRole.setRoleId(intRoleId);
                adminUserRole.setUserId(adminUserDTO.getId());
                this.adminUserRoleService.insertSelective(adminUserRole);
            }
        }
        return ResultUtil.genSuccessMsg();
    }

    @ApiOperation(value = "1-16-6 锁定或解锁用户", notes = "")
    @PostMapping(value = "updateUserStatus")
    public Json updateUserStatus(AdminUser user) {

        if (user.getId() == null || user.getStatus() == null) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数为空");
        }
        AdminUser adminUser = new AdminUser();
        adminUser.setId(user.getId());
        adminUser.setStatus(user.getStatus());
        this.adminUserService.updateByPrimaryKeySelective(adminUser);

        return ResultUtil.genSuccessMsg();
    }

    @ApiOperation(value = "1-16-7 获取当前用户信息和权限", notes = "")
    @GetMapping(value = "getCurrentUserInfo")
    public Json<CurrentUserVo> getCurrentUserInfo() {

        AdminUser adminUser = this.getLoginUser();
        if (adminUser != null) {
            CurrentUserVo currentUserVo = new CurrentUserVo(this.adminRoleService.selectByUserId(adminUser.getId()),
                    adminUser.getId(), adminUser.getUsername(),
                    adminUser.getPhone());
            return ResultUtil.genSuccessResult(currentUserVo);
        }
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "请重新登录");
    }

    @ApiOperation(value = "1-16-8 修改用户密码（管理员专用）", notes = "")
    @PostMapping(value = "/password")
    public Json modifyPasswordByAdmin(AdminUserDTO sysUser) {
        return this.adminUserService.modifyPasswordByAdmin(sysUser);
    }


    @ApiOperation(value = "1-16-9 修改用户角色", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "用户id", value = "userId"),
            @ApiImplicitParam(name = "角色id", value = "roleId")
    })
    @PostMapping(value = "/updateUserRole")
    public Json updateUserRole(Long userId, Integer roleId) {
        this.adminUserService.updateUserRole(userId, roleId);
        return ResultUtil.genSuccessResult();
    }


    @ApiOperation(value = "1-16-10 获取管理后台所有可用账号列表", notes = "")
    @GetMapping("user/list")
    public Json<List<AdminUserDTO>> getUserList() {
        return ResultUtil.genSuccessResult(adminUserService.getAvailableAdminUser());
    }
}








