package com.bxm.localnews.admin.service.activity.impl;

import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.localnews.admin.common.BizConfigProperties;
import com.bxm.localnews.admin.common.OfflinePushCodeProperties;
import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.AreaPushCodeMapper;
import com.bxm.localnews.admin.domain.UserMapper;
import com.bxm.localnews.admin.dto.CodeTemplate;
import com.bxm.localnews.admin.param.PushCodeParam;
import com.bxm.localnews.admin.service.activity.OfflinePushCodeService;
import com.bxm.localnews.admin.service.activity.strategy.PushCodeDispatcher;
import com.bxm.localnews.admin.utils.CompressUtil;
import com.bxm.localnews.admin.vo.AreaPushCode;
import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.base.model.dto.AvailableDomainInfo;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.ShortUrlGenerator;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.Optional;

/**
 * @author zhaoyadong 2019/4/2 15:10
 * @desc
 */
@Service
@RefreshScope
public class OfflinePushCodeServiceImpl extends BaseService implements OfflinePushCodeService {

    @Autowired
    private OfflinePushCodeProperties offlinePushCodeProperties;

    @Autowired
    private PushCodeDispatcher pushCodeDispatcher;

    @Autowired
    private BizConfigProperties bizConfigProperties;

    @Autowired
    private AreaPushCodeMapper areaPushCodeMapper;

    @Autowired
    private UserMapper userMapper;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private DomainIntegrationService domainIntegrationService;

    private List<CodeTemplate> codeTemplates = Lists.newArrayList();

    @PostConstruct
    public void init() {
        List<String> templateCodes = offlinePushCodeProperties.getTemplateCode();
        List<String> templateNames = offlinePushCodeProperties.getTemplateName();
        List<String> templateImgs = offlinePushCodeProperties.getTemplateImg();
        for (int i = 0; i < templateCodes.size(); i++) {
            CodeTemplate codeTemplate = new CodeTemplate();
            codeTemplate.setTemplateCode(templateCodes.get(i));
            codeTemplate.setTemplateName(templateNames.get(i));
            codeTemplate.setTemplateImg(templateImgs.get(i));
            codeTemplates.add(codeTemplate);
        }
    }

    @Override
    public byte[] generatePushCode(PushCodeParam pushCodeParam) {
        CodeTemplate codeTemplate = getCodeTemplate(pushCodeParam.getTemplateCode());

        List<AreaPushCode> areaPushCodes = Lists.newArrayListWithCapacity(pushCodeParam.getGenerateNum());
        for (int i = 0; i < pushCodeParam.getGenerateNum(); i++) {
            if (null != codeTemplate) {
                AreaPushCode areaPushCode = generatePushCode(codeTemplate, pushCodeParam);
                Message message = pushCodeDispatcher.execPushCode(areaPushCode, codeTemplate.getTemplateImg());
                areaPushCodes.add(areaPushCode);
            }
        }
        saveAreaPushCodes(areaPushCodes);

        try {
            byte[] bytes = CompressUtil.createZip(bizConfigProperties.getTempDir());


            return bytes;
        } catch (Exception e) {
            logger.error("生成二维码打成zip包出错");
            logger.error("异常信息：", e);
        }
        return null;
    }

    @Override
    public List<CodeTemplate> getCodeTemplates() {

        return codeTemplates;
    }

    @Override
    public Message bindPushCode(Long userId, Byte association, String remark, Long code) {
        Message message = Message.build();
        AreaPushCode areaPushCode = areaPushCodeMapper.selectById(code);

        if (areaPushCode == null || areaPushCode.getUserId() != null) {
            message.setSuccess(Boolean.FALSE);
            return message;
        }

        areaPushCode.setUserId(userId);
        areaPushCode.setAssociation(association);
        areaPushCode.setRemark(remark);

        areaPushCodeMapper.update(areaPushCode);
        redisStringAdapter.set(RedisConfig.SHORT_URL.copy().appendKey(generateShortCode(code.toString())),
                getDownloadUrl(userId));
        return message;
    }

    @Override
    public String getUserNickname(Long userId) {
        return userMapper.getNicknameById(userId);
    }

    @Override
    public Boolean userIsExist(Long userId) {
        return userMapper.countByUserId(userId) > 0;
    }

    /**
     * 根据模板code获取对应的模板信息
     *
     * @param templateCode
     * @return
     */
    private CodeTemplate getCodeTemplate(String templateCode) {
        Optional<CodeTemplate> optional = codeTemplates.stream().filter(codeTemplate ->
                codeTemplate.getTemplateCode().equals(templateCode)).findFirst();

        if (!optional.isPresent()) {
            return null;
        }

        return optional.get();
    }

    /**
     * 生成短链接
     *
     * @param code
     * @return
     */
    private String getUrl(String code) {
        // 获取短链的base url
        Optional<AvailableDomainInfo> baseUrlInfoByScene = domainIntegrationService.getBaseUrlInfoByScene(DomainScene.SHORT,
                null, null);
        String shortCode = generateShortCode(code);
        String h5ServerHost = domainIntegrationService.getInnerH5BaseUrl();
        redisStringAdapter.set(RedisConfig.SHORT_URL.copy().appendKey(shortCode),
                h5ServerHost + "/bindAccount.html?code=" + code);
        return baseUrlInfoByScene.isPresent() ? baseUrlInfoByScene.get().getBaseUrl() : "" + "/t/" + shortCode;
    }

    /**
     * 生成短码
     *
     * @param code
     * @return
     */
    private String generateShortCode(String code) {
        return ShortUrlGenerator.getShortUrl(code)[0];
    }

    /**
     * 生成下载链接
     *
     * @param userId
     * @return
     */
    private String getDownloadUrl(Long userId) {
        return domainIntegrationService.getInnerH5BaseUrl() + "pre/h5/offlineDownload.html?userId=" + userId;
    }

    /**
     * 生成推广码对象
     *
     * @param codeTemplate
     * @param pushCodeParam
     * @return
     */
    private AreaPushCode generatePushCode(CodeTemplate codeTemplate, PushCodeParam pushCodeParam) {
        Long id = nextId();
        return new AreaPushCode(id, pushCodeParam.getAreaCode(), pushCodeParam.getAreaName(),
                codeTemplate.getTemplateCode(), getUrl(id.toString()));
    }

    /**
     * 保存生成的二维码信息
     *
     * @param areaPushCodes
     */
    private void saveAreaPushCodes(List<AreaPushCode> areaPushCodes) {
        if (areaPushCodes.size() > 0) {
            new BatchHelper<AreaPushCodeMapper, AreaPushCode>(AreaPushCodeMapper.class, areaPushCodes,
                    "primarySessionTemplate") {
                @Override
                protected int invoke(AreaPushCode element) {
                    return mapper.insert(element);
                }
            };
        }
    }
}
