package com.bxm.localnews.admin.service.base.impl;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.AreaDivisionMapper;
import com.bxm.localnews.admin.domain.LocationMapper;
import com.bxm.localnews.admin.domain.LocationRelationMapper;
import com.bxm.localnews.admin.dto.LocationDTO;
import com.bxm.localnews.admin.enums.QrCodeEnum;
import com.bxm.localnews.admin.param.*;
import com.bxm.localnews.admin.service.base.LocationService;
import com.bxm.localnews.admin.vo.*;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.PinyinUtils;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.ResponseJson;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 地区相关接口实现类
 */
@Service
public class LocationServiceImpl extends BaseService implements LocationService {


    private final LocationMapper locationMapper;

    private final AreaDivisionMapper areaDivisionMapper;

    private final RedisStringAdapter redisStringAdapter;

    private LocationRelationMapper locationRelationMapper;

    /**
     * redis Hash服务类
     */
    private final RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    public LocationServiceImpl(LocationMapper locationMapper, AreaDivisionMapper areaDivisionMapper, RedisStringAdapter redisStringAdapter,
                               LocationRelationMapper locationRelationMapper, RedisHashMapAdapter redisHashMapAdapter) {
        this.locationMapper = locationMapper;
        this.areaDivisionMapper = areaDivisionMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.locationRelationMapper = locationRelationMapper;

        this.redisHashMapAdapter = redisHashMapAdapter;
    }

    @Override
    public List<LocationDTO> listLocationByPage(LocationPageParam param) {
        Page<Location> page = PageHelper.startPage(param).doSelectPage(() -> locationMapper.selectByPage(param));

        return page.stream()
                .map(this::convertLocation)
                .collect(Collectors.toList());
    }

    @Override
    public List<LocationDTO> listLocation(LocationParam locationParam) {
        Location location = new Location();
        BeanUtils.copyProperties(locationParam, location);

        if (Objects.equals("全国", locationParam.getName())) {
            // 如果查询的全国 伪造一个location
            LocationDTO locationDTO = new LocationDTO();
            // 查询code为0的时候，就是查询为全国的帖子
            locationDTO.setCode("0");
            locationDTO.setName("全国");
            return Collections.singletonList(locationDTO);
        }

        List<Location> locationList = locationMapper.selectByModel(location);
        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        return locationDTOList;
    }

    @Override
    public ResponseJson modifyLocation(ModifyLocationParam param) {
        if (Objects.isNull(param.getId())) {
            return ResponseJson.badReqeuset("要编辑的id为空").build();
        }

        Location location = new Location();
        location.setId(param.getId());
        // 编辑最新的行政区域码
        location.setLastCode(param.getLastCode());

        locationMapper.updateByPrimaryKeySelective(location);

        return ResponseJson.ok().build();
    }

    @Override
    public void saveHotLocation(String code) {
        Location location = locationMapper.selectByCode(code);
        if (location.getHot().equals((byte) 0)) {
            location.setHot((byte) 1);
            location.setShow((byte) 1);
            int count = locationMapper.updateByCodeSelective(location);
            if (count == 1) {
                List<LocationDTO> locationDTOList = listHotLocation();
                redisStringAdapter.set(RedisConfig.HOT_LOCATION, locationDTOList);
            }
        }
    }

    @Override
    public List<LocationDTO> listHotLocation() {
        Location locationParam = new Location();
        locationParam.setHot((byte) 1);
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);
        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        return locationDTOList;
    }

    private List<LocationDTO> listOpenedLocation() {
        Location locationParam = new Location();
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);


        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        //填充首字母
        locationDTOList.forEach(x -> x.setPinyin(PinyinUtils.getPinYinHeadChar(x.getName()).substring(0, 1).toUpperCase()));

        return locationDTOList;
    }

    @Override
    public void saveOpenedLocation(String code) {
        Location location = locationMapper.selectByCode(code);
        if (location.getShow().equals((byte) 0)) {
            location.setShow((byte) 1);
            int count = locationMapper.updateByCodeSelective(location);
            if (count == 1) {
                List<LocationDTO> locationDTOList = listOpenedLocation();
                redisStringAdapter.set(RedisConfig.OPENED_LOCATION, locationDTOList);
            }
        }
    }

    @Override
    public List<LocationDTO> listOpenedLocationByArea(String code, Byte show) {
        List<Location> locationList = new ArrayList<>();
        Location location = locationMapper.selectByCode(code);
        List<Location> locationListByParent = locationMapper.selectByParentCode(code, show);
        if (null != location) {
            locationList.add(location);
        }
        if (CollectionUtils.isNotEmpty(locationListByParent)) {
            locationList.addAll(locationListByParent);
        }
        return locationList.stream().map(this::convertLocation).collect(Collectors.toList());
    }

    @Override
    public List<LocationDTO> listOpenedLocationByAreaWithTwoLevels(String code, Byte show) {
        List<LocationDTO> locationDTOList = new ArrayList<>();

        //       根据省份信息得到所有已开通城市
        AreaDivision areaDivisionParam = new AreaDivision();
        areaDivisionParam.setParentCode(code);
        List<AreaDivision> areaDivisionList = areaDivisionMapper.selectByModel(areaDivisionParam);
        for (AreaDivision areaDivision : areaDivisionList) {
            List<LocationDTO> temLocationDTOList = listOpenedLocationByArea(areaDivision.getCode(), show);
            locationDTOList.addAll(temLocationDTOList);
        }

        locationDTOList = locationDTOList.stream().sorted(Comparator.comparing(x -> x.getPinyin().substring(0, 1))).collect(Collectors.toList());
        //        额外添加省份信息
//        AreaDivision provinceAreaDivision = areaDivisionMapper.selectByCode(code);
//        if (provinceAreaDivision != null) {
//            LocationDTO locationDTO = new LocationDTO();
//            locationDTO.setCode(code);
//            locationDTO.setName(provinceAreaDivision.getName());
//            locationDTOList.add(0,locationDTO);
//        }
        return locationDTOList;
    }

    private LoadingCache<String, Map<String, Location>> cache;

    @Override
    public Map<String, Location> listLocation() {
        if (cache == null) {
            cache = CacheBuilder.newBuilder()
                    .expireAfterWrite(10, TimeUnit.MINUTES)
                    .build(new CacheLoader<String, Map<String, Location>>() {
                        @Override
                        public Map<String, Location> load(String key) throws Exception {
                            Location location = new Location();
                            List<Location> locationList = locationMapper.selectByModel(location);
                            return locationList.stream().collect(Collectors.toMap(Location::getCode, Function.identity(), (key1, key2) -> key2));
                        }
                    });
        }

        return cache.getUnchecked("all");
    }

    @Override
    public String getAreaDetail(String[] areaDetailArray) {
        Map<String, Location> locationMap = listLocation();
        List<AreaCode> areaCodeList = new ArrayList<>();

        Arrays.stream(areaDetailArray).forEach(e -> {
            if (locationMap.get(e) != null) {
                AreaCode areaCode = new AreaCode();
                areaCode.setAreaCode(e);
                areaCode.setAreaName(locationMap.get(e).getName());
                areaCodeList.add(areaCode);
            }
        });
        return JSONObject.toJSONString(areaCodeList);
    }

    @Override
    public List<Location> getLocationsByAreaCodes(List<String> areaCodes) {
        if (CollectionUtils.isNotEmpty(areaCodes)) {
            List<Location> areas = new ArrayList<>();
            for (String areaCode : areaCodes) {
                Location location = locationMapper.selectByCode(areaCode);
                if (null != location) {
                    areas.add(location);
                }
            }
            return areas;
        }

        return null;
    }

    @Override
    public Boolean saveLocationRelation(LocationRelationParam param) {
        LocationRelationBean bean = new LocationRelationBean();
        BeanUtils.copyProperties(param, bean);
        bean.setId(nextId());
        bean.setCreateTime(new Date());
        return locationRelationMapper.insertAll(bean) > 0;
    }

    @Override
    public Boolean updateLocationRelation(LocationRelationParam param) {
        if (null == param.getId()) {
            return false;
        }
        LocationRelationBean bean = new LocationRelationBean();
        BeanUtils.copyProperties(param, bean);
        return locationRelationMapper.updateBySelective(bean) > 0;
    }

    @Override
    public Boolean updateLocationPhone(AreaCustomerPhoneParam param) {
        // 如果code为空则返回false
        if (StringUtils.isBlank(param.getCode())) {
            return false;
        }
        Location location = new Location();
        location.setEnableGroupCustomerServicePhone(param.getEnableGroupCustomerServicePhone());
        location.setGroupCustomerServicePhone(param.getGroupCustomerServicePhone());
        location.setCode(param.getCode());
        //如果缓存存在则清除缓存
        if (redisHashMapAdapter.exists(RedisConfig.GROUP_CUSTOMER_PHONE, param.getCode())) {
            redisHashMapAdapter.remove(RedisConfig.GROUP_CUSTOMER_PHONE, param.getCode());
        }
        return locationMapper.updateByCodeSelective(location) > 0;
    }

    @Override
    public LocationPhoneVO getLocationPhoneByCode(String code) {
        LocationPhoneVO phoneVO = new LocationPhoneVO();
        // 如果code为空则返回空对象
        if (StringUtils.isBlank(code)) {
            return phoneVO;
        }

        Location location = locationMapper.selectByCode(code);
        phoneVO.setCode(code);
        phoneVO.setName(location.getName());
        phoneVO.setEnableGroupCustomerServicePhone(location.getEnableGroupCustomerServicePhone());
        phoneVO.setGroupCustomerServicePhone(location.getGroupCustomerServicePhone());
        return phoneVO;
    }

    @Override
    public Message updateOfficialAccount(AreaOfficialAccountParam param) {
        // 如果code为空则返回false
        if (StringUtils.isBlank(param.getAreaCode())) {
            return Message.build(false).setMessage("地区编码不能为空");
        }

        Location location = locationMapper.selectByCode(param.getAreaCode());
        if (Objects.isNull(location)) {
            return Message.build(false).setMessage("地区信息不存在");
        }
        List<QrCodeInfoParam> qrCodeInfoList = param.getQrCodeInfoList();

        LocationOfficialAccountVO officialAccountVO = new LocationOfficialAccountVO();
        officialAccountVO.setName(location.getName());
        officialAccountVO.setCode(param.getAreaCode());
        qrCodeInfoList.removeIf(qrCodeInfo -> Objects.isNull(qrCodeInfo.getQrCodeUrl()));
        if (qrCodeInfoList.isEmpty()) {
            return Message.build(false).setMessage("二维码至少一个");
        }

        location.setOfficialAccountQrcode(JSON.toJSONString(qrCodeInfoList));
        List<QrCodeInfoVO> qrCodeInfoVOS = qrCodeInfoList.stream().map(qrCodeInfoParam -> {
            QrCodeInfoVO qrCodeInfoVO = new QrCodeInfoVO();
            BeanUtils.copyProperties(qrCodeInfoParam, qrCodeInfoVO);
            return qrCodeInfoVO;
        }).collect(Collectors.toList());
        officialAccountVO.setQrCodeInfoList(qrCodeInfoVOS);
        //如果缓存存在则清除缓存
        if (redisHashMapAdapter.exists(RedisConfig.QRCODE_LIST_INFO, param.getAreaCode())) {
            redisHashMapAdapter.remove(RedisConfig.QRCODE_LIST_INFO, param.getAreaCode());
        }
        redisHashMapAdapter.put(RedisConfig.QRCODE_LIST_INFO, param.getAreaCode(), officialAccountVO);
        if (locationMapper.updateByCodeSelective(location) > 0) {
            return Message.build(true);
        } else {
            return Message.build(false).setMessage("请稍后重试");
        }
    }

    @Override
    public LocationOfficialAccountVO getOfficialAccountByCode(String code) {
        LocationOfficialAccountVO officialAccountVO = redisHashMapAdapter.get(RedisConfig.QRCODE_LIST_INFO, code, LocationOfficialAccountVO.class);
        if (Objects.isNull(officialAccountVO)) {
            Location location = locationMapper.selectByCode(code);
            if (Objects.isNull(location)) {
                return null;
            }
            officialAccountVO = new LocationOfficialAccountVO();
            officialAccountVO.setCode(location.getCode());
            officialAccountVO.setName(location.getName());
            String officialAccountQrcode = location.getOfficialAccountQrcode();
            if (!JSONUtil.isJson(officialAccountQrcode) && StringUtils.isNoneBlank(officialAccountQrcode)) {
                List<QrCodeInfoVO> qrCodeInfoList = new ArrayList<>();
                QrCodeInfoVO qrCodeInfoVO = new QrCodeInfoVO();
                qrCodeInfoVO.setType(QrCodeEnum.TYPE_1.getType());
                qrCodeInfoVO.setTitle(QrCodeEnum.TYPE_1.getTitle());
                qrCodeInfoVO.setQrCodeUrl(officialAccountQrcode);
                qrCodeInfoList.add(qrCodeInfoVO);
                officialAccountVO.setQrCodeInfoList(qrCodeInfoList);
                redisHashMapAdapter.put(RedisConfig.QRCODE_LIST_INFO, code, officialAccountVO);

                //为了给前端回显用
                qrCodeInfoVO = new QrCodeInfoVO();
                qrCodeInfoVO.setType(QrCodeEnum.TYPE_2.getType());
                qrCodeInfoVO.setTitle(QrCodeEnum.TYPE_2.getTitle());
                qrCodeInfoVO.setQrCodeUrl(officialAccountQrcode);
                qrCodeInfoList.add(qrCodeInfoVO);
                qrCodeInfoVO = new QrCodeInfoVO();
                qrCodeInfoVO.setType(QrCodeEnum.TYPE_3.getType());
                qrCodeInfoVO.setTitle(QrCodeEnum.TYPE_3.getTitle());
                qrCodeInfoVO.setQrCodeUrl(officialAccountQrcode);
                qrCodeInfoList.add(qrCodeInfoVO);
                officialAccountVO.setQrCodeInfoList(qrCodeInfoList);
                return officialAccountVO;
            } else {
                List<QrCodeInfoVO> qrCodeInfos = JSONUtil.toList(officialAccountQrcode, QrCodeInfoVO.class);
                officialAccountVO.setQrCodeInfoList(qrCodeInfos);
                redisHashMapAdapter.put(RedisConfig.QRCODE_LIST_INFO, code, officialAccountVO);
                fillQrCodeInfo(qrCodeInfos);
                return officialAccountVO;
            }
        }
        fillQrCodeInfo(officialAccountVO.getQrCodeInfoList());
        return officialAccountVO;
    }

    private void fillQrCodeInfo(List<QrCodeInfoVO> qrCodeInfos) {
        List<Integer> qrCodeInfoTypeList = new ArrayList<>();
        qrCodeInfoTypeList = qrCodeInfos.stream().map(QrCodeInfoVO::getType).collect(Collectors.toList());
        List<Integer> typeList = QrCodeEnum.getTypeList();
        typeList.removeAll(qrCodeInfoTypeList);
        if (!typeList.isEmpty()) {
            for (Integer type : typeList) {
                String titleByType = QrCodeEnum.getTitleByType(type);
                QrCodeInfoVO qrCodeInfoVO = new QrCodeInfoVO();
                qrCodeInfoVO.setType(type);
                qrCodeInfoVO.setTitle(titleByType);
                qrCodeInfos.add(qrCodeInfoVO);
            }
        }
    }


    private LocationDTO convertLocation(Location location) {
        LocationDTO locationDTO = new LocationDTO();
        locationDTO.setCode(location.getCode());
        locationDTO.setId(location.getId());
        locationDTO.setHot(location.getHot());
        locationDTO.setShow(location.getShow());
        locationDTO.setLevel(location.getLevel());
        locationDTO.setName(location.getName());
        locationDTO.setPinyin(location.getPinyin());
        locationDTO.setLastCode(location.getLastCode());
        return locationDTO;
    }

    public static void main(String[] args) {
        List<QrCodeInfoParam> qrCodeInfoList = new ArrayList<>();
        QrCodeInfoParam qrCodeInfoParam = new QrCodeInfoParam();
        qrCodeInfoParam.setQrCodeUrl("12321");
        qrCodeInfoList.add(qrCodeInfoParam);
        qrCodeInfoList.removeIf(qrCodeInfo -> {
            return Objects.isNull(qrCodeInfo.getQrCodeUrl());
        });

        if (qrCodeInfoList.isEmpty()) {
            System.out.println("二维码至少一个");
        }
    }
}
