package com.bxm.localnews.admin.service.base.impl;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.UserMapper;
import com.bxm.localnews.admin.domain.talent.UserTalentInviteHistoryMapper;
import com.bxm.localnews.admin.entity.talent.UserTalentInviteHistoryEntity;
import com.bxm.localnews.admin.enums.BecomeTalentType;
import com.bxm.localnews.admin.integration.OrderIntegrationService;
import com.bxm.localnews.admin.integration.PushMsgIntegrationService;
import com.bxm.localnews.admin.param.RebateUpParam;
import com.bxm.localnews.admin.param.UserBecomeTalentParam;
import com.bxm.localnews.admin.param.talent.TalentListQueryParam;
import com.bxm.localnews.admin.param.talent.UpgradeTalentParam;
import com.bxm.localnews.admin.service.base.TalentMangeService;
import com.bxm.localnews.admin.vo.User;
import com.bxm.localnews.admin.vo.talent.TalentListVO;
import com.bxm.localnews.common.constant.TalentLevelEnum;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import static com.github.pagehelper.page.PageMethod.startPage;

/**
 * 达人后台管理service实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2020/12/14 2:47 下午
 */
@Slf4j
@Service
public class TalentManageServiceImpl implements TalentMangeService {

    private final UserTalentInviteHistoryMapper userTalentInviteHistoryMapper;

    private final SequenceCreater sequenceCreater;

    private final DistributedLock lock;

    private final UserMapper userMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final OrderIntegrationService orderIntegrationService;

    private final PushMsgIntegrationService pushMsgIntegrationService;

    public TalentManageServiceImpl(UserTalentInviteHistoryMapper userTalentInviteHistoryMapper,
                                   SequenceCreater sequenceCreater,
                                   DistributedLock lock,
                                   UserMapper userMapper,
                                   RedisStringAdapter redisStringAdapter,
                                   OrderIntegrationService orderIntegrationService,
                                   PushMsgIntegrationService pushMsgIntegrationService) {
        this.userTalentInviteHistoryMapper = userTalentInviteHistoryMapper;
        this.sequenceCreater = sequenceCreater;
        this.lock = lock;
        this.userMapper = userMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.orderIntegrationService = orderIntegrationService;
        this.pushMsgIntegrationService = pushMsgIntegrationService;
    }

    @Override
    public PageWarper<TalentListVO> getTalentList(TalentListQueryParam param) {
        startPage(param.getPageNum(), param.getPageSize());
        return new PageWarper<>(userTalentInviteHistoryMapper.getTalentList(param));
    }

    @Override
    public Message upgradeTalent(UpgradeTalentParam param) {

        // 获取升级达人等级分布式锁，锁粒度，同一个用户
        String resourceKey = genUpgradeTalentLockKey(param.getUserId());
        String requestId = sequenceCreater.nextStringId();

        //重复提交时间限制2s
        long repeatCommitTimeLimit = 2;
        if (Boolean.FALSE.equals(lock.lock(resourceKey, requestId, repeatCommitTimeLimit, TimeUnit.SECONDS))) {
            return Message.build(false, "您的请求过于频繁");
        }
        //当前用户信息
        User currentUserInfo = userMapper.selectByUserId(param.getUserId());

        //校验达人当前能否被升级
        Message checkMessage = checkUserUpgrade(currentUserInfo);
        if (Boolean.FALSE.equals(checkMessage.isSuccess())) {
            return checkMessage;
        }
        //如果当前是中达人，提升为大达人
        if (Objects.equals(TalentLevelEnum.MIDDLE_TALENT.getCode(), currentUserInfo.getTalentLevel())) {
            //发送升级为大达人推送
            pushMsgIntegrationService.pushBecomeBigTalent(param.getUserId());
            //进行收益升级
            RebateUpParam rebateUpParam = new RebateUpParam();
            rebateUpParam.setUserId(currentUserInfo.getId());
            Message rebateUpMsg = orderIntegrationService.rebateUp(rebateUpParam);
            if (Boolean.FALSE.equals(rebateUpMsg.isSuccess())) {
                log.error("达人收益升级失败，用户id：{}", currentUserInfo.getId());
            }
        }

        //如果当前是小达人，提升为中达人
        if (Objects.equals(TalentLevelEnum.SMALL_TALENT.getCode(), currentUserInfo.getTalentLevel())) {
            //发送升级为中达人的推送
            pushMsgIntegrationService.pushBecomeMiddleTalent(param.getUserId());
        }
        //进行达人升级
        upgradeTalentExecute(currentUserInfo, currentUserInfo.getTalentLevel() + 1);

        //释放锁
        lock.unlock(resourceKey, requestId);

        return Message.build(true);
    }

    @Override
    public Message becomeTalent(UserBecomeTalentParam param) {
        // 获取升级达人分布式锁，锁粒度，同一个用户
        String resourceKey = genUpgradeTalentLockKey(param.getUserId());
        String requestId = sequenceCreater.nextStringId();

        //重复提交时间限制2s
        long repeatCommitTimeLimit = 2;
        if (Boolean.FALSE.equals(lock.lock(resourceKey, requestId, repeatCommitTimeLimit, TimeUnit.SECONDS))) {
            return Message.build(false, "您的请求过于频繁");
        }

        //当前用户信息
        User currentUserInfo = userMapper.selectByUserId(param.getUserId());
        if (Objects.isNull(currentUserInfo)) {
            return Message.build(false, "用户不存在");
        }
        //校验达人当前能否被升级
        Message checkMessage = checkBecomeTalent(currentUserInfo);
        if (Boolean.FALSE.equals(checkMessage.isSuccess())) {
            return checkMessage;
        }

        //将普通用户升级为小达人
        upgradeTalentExecute(currentUserInfo, TalentLevelEnum.SMALL_TALENT.getCode());
        //添加到达人表中
        addTalentInfo(currentUserInfo);

        //发送成为达人的推送
        pushMsgIntegrationService.pushRegisterTalent(param.getUserId());

        //发送成为达人的公众号推送
        pushMsgIntegrationService.pushRegisterTalentOfficialAccount(param.getUserId(), currentUserInfo.getNickname());

        lock.unlock(resourceKey, requestId);

        return Message.build(true);
    }

    /**
     * 添加达人相关信息
     *
     * @param currentUserInfo 用户信息
     */
    private void addTalentInfo(User currentUserInfo) {
        UserTalentInviteHistoryEntity addEntity = new UserTalentInviteHistoryEntity();

        addEntity.setId(sequenceCreater.nextLongId());
        addEntity.setCreateTime(new Date());
        //设置成为达人方式，1表示后台管理直接升级
        addEntity.setType(BecomeTalentType.ADMIN.getCode());
        addEntity.setUserId(currentUserInfo.getId());
        addEntity.setPhone(currentUserInfo.getPhone());
        addEntity.setRealName(currentUserInfo.getNickname());

        userTalentInviteHistoryMapper.insertSelective(addEntity);
    }

    /**
     * 执行升级用户达人等级
     *
     * @param currentUserInfo 用户信息
     */
    private void upgradeTalentExecute(User currentUserInfo, Integer targetLevel) {

        User updateUser = new User();
        updateUser.setId(currentUserInfo.getId());
        // 对达人等级进行升级
        updateUser.setTalentLevel(targetLevel);

        userMapper.updateByPrimaryKeySelective(updateUser);

        //删除用户缓存
        redisStringAdapter.remove(RedisConfig.USER_INFO.copy().appendKey(currentUserInfo.getId()));
    }

    /**
     * 校验当前用户是否能够提升达人等级
     *
     * @param currentUserInfo 用户信息
     * @return 校验结果
     */
    private Message checkUserUpgrade(User currentUserInfo) {
        //判断用户是否存在
        if (currentUserInfo == null) {
            return Message.build(false, "当前用户不存在");
        }
        Integer talentLevel = currentUserInfo.getTalentLevel();

        //如果为普通用户则不能进行升级
        if (Objects.equals(TalentLevelEnum.ORDINARY_USER.getCode(), talentLevel)) {
            return Message.build(false, "只能对达人用户进行升级");
        } else if (Objects.equals(TalentLevelEnum.BIG_TALENT.getCode(), talentLevel)) {
            return Message.build(false, "无法进行升级，当前已经是大达人");
        }
        return Message.build(true);
    }

    /**
     * 校验当前用户是否能够提升达人等级
     *
     * @param currentUserInfo 用户信息
     * @return 校验结果
     */
    private Message checkBecomeTalent(User currentUserInfo) {
        //判断用户是否存在
        if (currentUserInfo == null) {
            return Message.build(false, "当前用户不存在");
        }
        Integer talentLevel = currentUserInfo.getTalentLevel();

        //如果为普通用户则不能进行升级
        if (Boolean.FALSE.equals(Objects.equals(TalentLevelEnum.ORDINARY_USER.getCode(), talentLevel))) {
            return Message.build(false, "只能将普通用户提升为达人");
        }
        return Message.build(true);
    }

    /**
     * 生成升级达人等级的分布式锁key
     *
     * @param userId 用户id
     * @return key值
     */
    private String genUpgradeTalentLockKey(Long userId) {
        return RedisConfig.USER_UPGRADE_TALENT_LOCK.appendKey(userId).gen();
    }
}